/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.property;

import org.w3c.css.sac.LexicalUnit;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.CSSValue;
import org.w3c.dom.css.Counter;
import org.w3c.dom.css.RGBColor;
import org.w3c.dom.css.Rect;

import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.StyleDatabaseAware;

/**
 * Implementation of CSSPrimitiveValue.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class AbstractCSSPrimitiveValue extends AbstractCSSValue implements
		CSSPrimitiveValue, StyleDatabaseAware {

	private short primitiveType = CSSPrimitiveValue.CSS_UNKNOWN;

	private LexicalUnit lexicalUnit = null;
	
	private StyleDatabase styleDb = null;

	private boolean subproperty = false;

	protected AbstractCSSPrimitiveValue(short unitType) {
		super(CSSValue.CSS_PRIMITIVE_VALUE);
		primitiveType = unitType;
	}

	protected AbstractCSSPrimitiveValue() {
		this(CSSPrimitiveValue.CSS_UNKNOWN);
	}

	public short getPrimitiveType() {
		return primitiveType;
	}

	public void setFloatValue(short unitType, float floatValue)
			throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR, "Not a Float");
	}

    /**
     *  This method is used to get a float value in a specified unit. If this 
     * CSS value doesn't contain a float value or can't be converted into 
     * the specified unit, a <code>DOMException</code> is raised. 
     * @param unitType A unit code to get the float value. The unit code can 
     *   only be a float unit type (i.e. <code>CSS_NUMBER</code>, 
     *   <code>CSS_PERCENTAGE</code>, <code>CSS_EMS</code>, 
     *   <code>CSS_EXS</code>, <code>CSS_PX</code>, <code>CSS_CM</code>, 
     *   <code>CSS_MM</code>, <code>CSS_IN</code>, <code>CSS_PT</code>, 
     *   <code>CSS_PC</code>, <code>CSS_DEG</code>, <code>CSS_RAD</code>, 
     *   <code>CSS_GRAD</code>, <code>CSS_MS</code>, <code>CSS_S</code>, 
     *   <code>CSS_HZ</code>, <code>CSS_KHZ</code>, 
     *   <code>CSS_DIMENSION</code>). 
     * @return  The float value in the specified unit. 
     * @exception DOMException
     *    INVALID_ACCESS_ERR: Raised if the CSS value doesn't contain a float 
     *   value or if the float value can't be converted into the specified 
     *   unit. 
     */
	public float getFloatValue(short unitType) throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR, "Not a Float");
	}

	public void setStringValue(short stringType, String stringValue)
			throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR, "Not a String");
	}

	public String getStringValue() throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR, "Not a String");
	}

	public Counter getCounterValue() throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR, "Not a Counter");
	}

	public Rect getRectValue() throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR, "Not a Rect");
	}

	public RGBColor getRGBColorValue() throws DOMException {
		throw new DOMException(DOMException.INVALID_ACCESS_ERR,
				"Not an RGB Color");
	}

	public StyleDatabase getStyleDatabase() {
		return styleDb;
	}

	public void setStyleDatabase(StyleDatabase styleDb) {
		this.styleDb = styleDb;
	}
	
	public void setSubproperty(boolean subp) {
		subproperty  = subp;
	}

	public boolean isSubproperty() {
		return subproperty;
	}

	protected void setCSSUnitType(short cssUnitType) {
		this.primitiveType = cssUnitType;
	}

	void setLexicalUnit(LexicalUnit lunit) {
		lexicalUnit = lunit;
		setCSSUnitType(domPrimitiveType(lunit.getLexicalUnitType()));
	}

	public LexicalUnit getLexicalUnit() {
		return lexicalUnit;
	}

	/**
	 * Retrieves the next CSS primitive value from the current lexical unit,
	 * ignoring commas, etc.
	 * 
	 * @return the next CSS primitive value, or null if none.
	 */
	public CSSPrimitiveValue nextPrimitiveValue() {
		LexicalUnit nextLex = getLexicalUnit();
		AbstractCSSPrimitiveValue nextval = null;
		do {
			nextLex = nextLex.getNextLexicalUnit();
			if(nextLex == null){
				return null;
			}
			nextval = (AbstractCSSPrimitiveValue) createCSSPrimitiveValue(nextLex);
		} while(nextval == null);
		nextval.setStyleDatabase(getStyleDatabase());
		return nextval;
	}

	/**
	 * Creates a CSSPrimitiveValue according to the given lexical value.
	 * 
	 * @param lunit the lexical value.
	 * @return the CSS primitive value.
	 */
	static CSSPrimitiveValue createCSSPrimitiveValue(LexicalUnit lunit) {
		short unitType = lunit.getLexicalUnitType();
		AbstractCSSPrimitiveValue primi = null;
		switch (unitType) {
		case LexicalUnit.SAC_IDENT:
			primi = new CSSIdentifierValue();
			primi.setLexicalUnit(lunit);
			break;
		case LexicalUnit.SAC_ATTR:
		case LexicalUnit.SAC_STRING_VALUE:
		case LexicalUnit.SAC_URI:
			primi = new CSSStringValue();
			primi.setLexicalUnit(lunit);
			break;
		case LexicalUnit.SAC_PERCENTAGE:
			primi = new CSSPercentageValue();
			primi.setLexicalUnit(lunit);
			break;
		case LexicalUnit.SAC_REAL:
		case LexicalUnit.SAC_CENTIMETER:
		case LexicalUnit.SAC_DEGREE:
		case LexicalUnit.SAC_DIMENSION:
		case LexicalUnit.SAC_EM:
		case LexicalUnit.SAC_EX:
		case LexicalUnit.SAC_GRADIAN:
		case LexicalUnit.SAC_HERTZ:
		case LexicalUnit.SAC_INCH:
		case LexicalUnit.SAC_KILOHERTZ:
		case LexicalUnit.SAC_MILLIMETER:
		case LexicalUnit.SAC_MILLISECOND:
		case LexicalUnit.SAC_PICA:
		case LexicalUnit.SAC_PIXEL:
		case LexicalUnit.SAC_POINT:
		case LexicalUnit.SAC_RADIAN:
		case LexicalUnit.SAC_SECOND:
			primi = new CSSNumberValue();
			primi.setLexicalUnit(lunit);
			break;
		case LexicalUnit.SAC_INTEGER:
			primi = new CSSNumberValue();
			primi.setFloatValue(domPrimitiveType(unitType), lunit.getIntegerValue());
			break;
		case LexicalUnit.SAC_RGBCOLOR:
			primi = new DOMCSSColorValue();
			primi.setLexicalUnit(lunit);
			break;
		case LexicalUnit.SAC_RECT_FUNCTION:
			primi = new DOMCSSRectValue();
			primi.setLexicalUnit(lunit);
			break;
		case LexicalUnit.SAC_COUNTER_FUNCTION:
			primi = new DOMCSSCounterValue();
			primi.setLexicalUnit(lunit);
			break;
		default:
			primi = null;
		}
		return primi;
	}

	/**
	 * Translate a SAC lexical type into a CSS primitive unit type.
	 * 
	 * @param sacType the lexical type.
	 * @return the unit type according to <code>CSSPrimitiveValue</code>.
	 */
	protected static short domPrimitiveType(int sacType) {
		short primiType;
		switch (sacType) {
		case LexicalUnit.SAC_ATTR:
			primiType = CSSPrimitiveValue.CSS_ATTR;
			break;
		case LexicalUnit.SAC_IDENT:
			primiType = CSSPrimitiveValue.CSS_IDENT;
			break;
		case LexicalUnit.SAC_STRING_VALUE:
			primiType = CSSPrimitiveValue.CSS_STRING;
			break;
		case LexicalUnit.SAC_URI:
			primiType = CSSPrimitiveValue.CSS_URI;
			break;
		case LexicalUnit.SAC_REAL:
			primiType = CSSPrimitiveValue.CSS_NUMBER;
			break;
		case LexicalUnit.SAC_CENTIMETER:
			primiType = CSSPrimitiveValue.CSS_CM;
			break;
		case LexicalUnit.SAC_DEGREE:
			primiType = CSSPrimitiveValue.CSS_DEG;
			break;
		case LexicalUnit.SAC_DIMENSION:
			primiType = CSSPrimitiveValue.CSS_DIMENSION;
			break;
		case LexicalUnit.SAC_EM:
			primiType = CSSPrimitiveValue.CSS_EMS;
			break;
		case LexicalUnit.SAC_EX:
			primiType = CSSPrimitiveValue.CSS_EXS;
			break;
		case LexicalUnit.SAC_GRADIAN:
			primiType = CSSPrimitiveValue.CSS_GRAD;
			break;
		case LexicalUnit.SAC_HERTZ:
			primiType = CSSPrimitiveValue.CSS_HZ;
			break;
		case LexicalUnit.SAC_INCH:
			primiType = CSSPrimitiveValue.CSS_IN;
			break;
		case LexicalUnit.SAC_KILOHERTZ:
			primiType = CSSPrimitiveValue.CSS_KHZ;
			break;
		case LexicalUnit.SAC_MILLIMETER:
			primiType = CSSPrimitiveValue.CSS_MM;
			break;
		case LexicalUnit.SAC_MILLISECOND:
			primiType = CSSPrimitiveValue.CSS_MS;
			break;
		case LexicalUnit.SAC_PERCENTAGE:
			primiType = CSSPrimitiveValue.CSS_PERCENTAGE;
			break;
		case LexicalUnit.SAC_PICA:
			primiType = CSSPrimitiveValue.CSS_PC;
			break;
		case LexicalUnit.SAC_PIXEL:
			primiType = CSSPrimitiveValue.CSS_PX;
			break;
		case LexicalUnit.SAC_POINT:
			primiType = CSSPrimitiveValue.CSS_PT;
			break;
		case LexicalUnit.SAC_RADIAN:
			primiType = CSSPrimitiveValue.CSS_RAD;
			break;
		case LexicalUnit.SAC_SECOND:
			primiType = CSSPrimitiveValue.CSS_S;
			break;
		case LexicalUnit.SAC_INTEGER:
			primiType = CSSPrimitiveValue.CSS_NUMBER;
			break;
		case LexicalUnit.SAC_RGBCOLOR:
			primiType = CSSPrimitiveValue.CSS_RGBCOLOR;
			break;
		case LexicalUnit.SAC_RECT_FUNCTION:
			primiType = CSSPrimitiveValue.CSS_RECT;
			break;
		case LexicalUnit.SAC_COUNTER_FUNCTION:
			primiType = CSSPrimitiveValue.CSS_COUNTER;
			break;
		default:
			primiType = CSSPrimitiveValue.CSS_UNKNOWN;
		}
		return primiType;
	}
}
