/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.property;

import info.informatica.doc.style.css.StyleDeclarationFactory;
import info.informatica.text.TokenParser;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.w3c.dom.css.CSSValue;

/**
 * Database of device-independent CSS 2.1 property information.
 * <p>
 * Also creates CSSPropertyName objects.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public final class PropertyDatabase {
	private ClassLoader classLoader = null;

	private Map<String, CSSPropertyName> nameMap = new HashMap<String, CSSPropertyName>(
			42);

	private List<String> inherited_properties;

	/**
	 * Map from shorthand property names to subproperties
	 */
	private Map<String,String[]> shorthand2subp = new HashMap<String,String[]>();

	/**
	 * Maps shorthand subproperties to their shorthand property name.
	 */
	private Map<String,String> subp2shorthand = new HashMap<String,String>();
	
	/**
	 * Identifier properties
	 */
	private Properties identifiers;
	
	/**
	 * Map of initial property values.
	 */
	private Map<String,CSSValue> initialValueMap;

	private static PropertyDatabase singleton = new PropertyDatabase();

	static Logger log = Logger.getLogger(PropertyDatabase.class.getName());

	protected PropertyDatabase() {
		super();
		/*
		 * Inherited properties
		 */
		inherited_properties = computeInheritedPropertiesList();
		/*
		 * Shorthand properties
		 */
		Properties shand = loadPropertiesfromClasspath("shorthand.properties");
		ArrayList<String> array = new ArrayList<String>();
		Iterator<Entry<Object,Object>> it = shand.entrySet().iterator();
		while(it.hasNext()) {
			Entry<Object, Object> me = it.next();
			String shname = (String) me.getKey();
			String subpties = (String) me.getValue();
			StringTokenizer st = new StringTokenizer(subpties, ",");
			while(st.hasMoreTokens()){
				array.add(st.nextToken().trim());
			}
			addShorthand(shname, array.toArray(new String[0]));
			array.clear();
		}
		/*
		 * Identifiers
		 */
		identifiers = loadPropertiesfromClasspath("identifier.properties");
		/*
		 * Initial values
		 */
		initialValueMap = computeInitialValueMap();
	}

	/**
	 * Gets an instance of this class.
	 * 
	 * @return an instance of PropertyNameFactory.
	 */
	public static PropertyDatabase getInstance() {
		return singleton;
	}

	/**
	 * Does this property inherit value by default?
	 * 
	 * @param name the name of the property.
	 * @return true if inherits by default, false otherwise.
	 */
	public boolean isInherited(String name){
		return inherited_properties.contains(name);
	}

	protected List<String> computeInheritedPropertiesList() {
		/**
		 * List of properties that inherit by default.
		 */
		String[] inherit = { "azimuth", "border-collapse", "border-spacing",
			"caption-side", "color", "cursor", "direction", "elevation",
			"empty-cells", "font-family", "font-size", "font-style",
			"font-variant", "font-weight", "font", "letter-spacing",
			"line-height", "list-style-image", "list-style-position",
			"list-style-type", "list-style", "orphans", "page-break-inside",
			"pitch-range", "pitch", "quotes", "richness", "speak-header",
			"speak-numeral", "speak-punctuation", "speak", "speech-rate",
			"stress", "text-align", "text-indent", "text-transform",
			"visibility", "voice-family", "volume", "white-space", "widows",
			"word-spacing" };
		return Arrays.asList(inherit);
	}

	/**
	 * Gives the initial device-independent initial (default) value 
	 * for the given property.
	 * 
	 * @param propertyName the property name.
	 * @return the initial CSS value, or null if no device-independent 
	 * default could be found.
	 */
	public CSSValue getInitialValue(String propertyName) {
		return initialValueMap.get(propertyName);
	}

	protected Map<String,CSSValue> computeInitialValueMap() {
		String[][] initialArray = {
				{"azimuth", "center"},
				{"background-attachment", "scroll"},
				{"background-position", "0% 0%"},
				{"background-repeat", "repeat"},
				{"border-collapse", "separate"},
				{"border-spacing", "0"},
				{"border-top-style", "none"},
				{"border-right-style", "none"},
				{"border-bottom-style", "none"},
				{"border-left-style", "none"},
				{"border-top-width", "medium"},
				{"border-right-width", "medium"},
				{"border-bottom-width", "medium"},
				{"border-left-width", "medium"},
				{"bottom", "auto"},
				{"caption-side", "top"},
				{"content", "normal"},
				{"counter-increment", "none"},
				{"counter-reset", "none"},
				{"cue-after", "none"},
				{"cue-before", "none"},
				{"cursor", "auto"},
				{"direction", "ltr"},
				{"display", "inline"},
				{"elevation", "level"}, 
				{"empty-cells", "show"},
				{"float", "none"},
				{"font-size", "medium"},
				{"font-style", "normal"}, 
				{"font-variant", "normal"},
				{"font-weight", "normal"},
				{"height", "auto"},
				{"left", "auto"},
				{"letter-spacing", "normal"}, 
				{"line-height", "normal"},
				{"list-style-image", "none"},
				{"list-style-position", "outside"}, 
				{"list-style-type", "disc"},
				{"margin-right", "0"},
				{"margin-left", "0"},
				{"margin-top", "0"},
				{"margin-bottom", "0"},
				{"max-height", "none"},
				{"max-width", "none"},
				{"min-height", "0"},
				{"min-width", "0"},
				{"orphans", "2"},
				{"outline-color", "invert"},
				{"outline-style", "none"},
				{"outline-width", "medium"},
				{"overflow", "visible"},
				{"padding-top", "0"},
				{"padding-right", "0"},
				{"padding-bottom", "0"},
				{"padding-left", "0"},
				{"page-break-after", "auto"}, 
				{"page-break-before", "auto"}, 
				{"page-break-inside", "auto"}, 
				{"pause-after", "0"},
				{"pause-before", "0"},
				{"pitch-range", "50"},
				{"pitch", "medium"},
				{"play-during", "auto"},
				{"position", "static"},
				{"richness", "50"},
				{"right", "auto"},
				{"speak-header", "once"}, 
				{"speak-numeral", "continuous"},
				{"speak-punctuation", "none"},
				{"speak", "normal"},
				{"speech-rate", "medium"}, 
				{"stress", "50"},
				{"table-layout", "auto"},
				{"text-decoration", "none"},
				{"text-indent", "0"},
				{"text-transform", "none"},
				{"top", "auto"},
				{"unicode-bidi", "normal"},
				{"vertical-align", "baseline"},
				{"visibility", "visible"},
				{"voice-family", "male"},
				{"volume", "medium"},
				{"white-space", "normal"},
				{"widows", "2"},
				{"width", "auto"},
				{"word-spacing", "normal"},
				{"z-index", "auto"}
			};
		Map<String,CSSValue> initialValueMap = 
			new HashMap<String,CSSValue>(initialArray.length);
		for(int i=0; i<initialArray.length; i++){
			CSSValue value = StyleDeclarationFactory.parseProperty(initialArray[i][1]);
			initialValueMap.put(initialArray[i][0], value);
		}
		return initialValueMap;
	}

	/**
	 * Is this a shorthand property?
	 * 
	 * @param name the name of the property.
	 * @return true if is a shorthand, false otherwise.
	 */
	public boolean isShorthand(String name){
		return shorthand2subp.containsKey(name);
	}

	/**
	 * Is this the subproperty of a shorthand property?
	 * 
	 * @param name the name of the property.
	 * @return true if is a shorthand subproperty, false otherwise.
	 */
	public boolean isShorthandSubproperty(String name){
		return subp2shorthand.containsKey(name);
	}

	/**
	 * Is this the subproperty of the given shorthand property ?
	 * 
	 * @param shorthand the name of the shorthand property.
	 * @param subpName the name of the subproperty.
	 * @return true if subpName is a subproperty of the given shorthand,
	 *  false otherwise.
	 */
	public boolean isShorthandSubpropertyOf(String shorthand, String subpName){
		String sh = subp2shorthand.get(subpName);
		if(sh == null) {
			return false;
		} else {
			if(sh.equals(shorthand)) {
				return true;
			} else {
				// Check for border-width, border-style, border-color
				sh = subp2shorthand.get(sh);
				if(shorthand.equals(sh)) {
					return true;
				} else {
					return false;
				}
			}
		}
	}

	public String[] getShorthandSubproperties(String shorthandName){
		return shorthand2subp.get(shorthandName);
	}

	/**
	 * Gets the CSSPropertyName object corresponding to the given
	 * property name.
	 * 
	 * @param name the property name.
	 * @return the associated CSSPropertyName object.
	 */
	public CSSPropertyName getPropertyName(String name) {
		CSSPropertyName pname = nameMap.get(name);
		if(pname == null) {
			synchronized(nameMap) {
				if(isShorthand(name)){
					pname = new ShorthandPropertyName(name, isInherited(name));
					((ShorthandPropertyName)pname).setSubProperties(shorthand2subp.get(name));
				} else if(isShorthandSubproperty(name)){
					pname = new SubPropertyName(name, isInherited(name));
					((SubPropertyName)pname).setShorthand(subp2shorthand.get(name));
				} else {
					pname = new CSSPropertyName(name, isInherited(name));
				}
				nameMap.put(name, pname);
			}
		}
		return pname;
	}
	
	protected void addShorthand(String shorthand, String[] subproperties) {
		shorthand2subp.put(shorthand, subproperties);
		for(int i=0; i<subproperties.length; i++){
			String prevSh = subp2shorthand.get(subproperties[i]);
			if(prevSh != null) {
				// Already one map for this key, e.g. we have
				// border-top-width -> border-top and now we got
				// border-top-width -> border-width
				String topShorthand = subp2shorthand.get(shorthand);
				String topPrevSh = subp2shorthand.get(prevSh);
				if(topShorthand == null) {
					if(topPrevSh != null) {
						subp2shorthand.put(shorthand, topPrevSh);
					}
				} else if(topPrevSh == null) {
					subp2shorthand.put(prevSh, topShorthand);
				}
			}
			subp2shorthand.put(subproperties[i], shorthand);
		}
	}

	/**
	 * Determines if the given value is an identifier for the given property
	 * name.
	 * <p>
	 * Generic identifiers such as <code>inherit</code> or <code>none</code>
	 * are not checked.
	 * 
	 * @param propertyName the name of the property.
	 * @param value the value that has to be tested to be an identifier for 
	 * propertyName.
	 * @return true if <code>value</code> is recognized as an identifier 
	 * of <code>propertyName</code>, false otherwise.
	 */
	public boolean isIdentifierValue(String propertyName, String value) {
		String csl = identifiers.getProperty(propertyName);
		if(csl == null){
			log.debug("Could not find identifiers for " + propertyName);
			return false;
		}
		TokenParser tokp = new TokenParser(csl, ",");
		while(tokp.hasNext()){
			String s = tokp.nextToken();
			if(s.trim().equals(value)){
				return true;
			}
		}
		return false;
	}

	public void setClassLoader(ClassLoader loader) {
		classLoader = loader;
	}

	Properties loadPropertiesfromClasspath(final String filename) {
		return java.security.AccessController
				.doPrivileged(new java.security.PrivilegedAction<Properties>() {
					public Properties run() {
						InputStream is;
						if (classLoader != null) {
							is = classLoader
									.getResourceAsStream(resourcePath(filename));
						} else {
							is = this.getClass()
									.getResourceAsStream(resourcePath(filename));
						}
						if(is == null){
							return null;
						}
						Properties p = new Properties();
						try {
							p.load(is);
						} catch (IOException e) {
							return null;
						} finally {
							try {
								is.close();
							} catch (IOException e) {
							}
						}
						return p;
					}
				});
	}
	
	protected String resourcePath(String filename) {
		return '/' + PropertyDatabase.class.getPackage()
			.getName().replace('.', '/') + '/' + filename;
	}

}
