/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.CSSErrorHandler;
import info.informatica.doc.style.css.CSSStyleException;
import info.informatica.doc.style.css.dom.ComputedCSSStyle;
import info.informatica.doc.style.css.visual.box.BlockBox;
import info.informatica.doc.style.css.visual.box.InlineBox;
import info.informatica.doc.style.css.visual.box.InlineTable;
import info.informatica.doc.style.css.visual.box.ListItemBox;
import info.informatica.doc.style.css.visual.box.RunInBox;
import info.informatica.doc.style.css.visual.box.Table;
import info.informatica.doc.style.css.visual.box.TableCellBox;
import info.informatica.doc.style.css.visual.box.TableRowBox;

import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Factory for CSS boxes.
 * <p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public class CSSBoxFactory {
	
	private CSSErrorHandler errorHandler;

	public CSSBoxFactory() {
		super();
	}

	/**
	 * Gets the error handler.
	 * 
	 * @return the CSS error handler.
	 */
	public CSSErrorHandler getErrorHandler() {
		return errorHandler;
	}

	/**
	 * Sets the error handler.
	 * 
	 * @param errorHandler the CSS error handler.
	 */
	public void setErrorHandler(CSSErrorHandler errorHandler) {
		this.errorHandler = errorHandler;
	}

	/**
	 * Creates a box of the appropriate type according to CSS box model.
	 * <p>
	 * Note that table sub-elements are not generated here.
	 * 
	 * @param style the computed style that applies to the box.
	 * @return the generated box, or null if the element generates no box 
	 * (<code>display: none</code>).
	 * @throws CSSStyleException if an error occurs generating the box.
	 */
	public CSSBox create(CSS2ComputedProperties style) 
	throws CSSStyleException {
		CSSBox box = null;
		String display = ((CSSPrimitiveValue)((ComputedCSSStyle)style)
				.getCSSValue("display")).getStringValue();
		if(display.equals("inline")) {
			box = createInlineBox(style);
		} else if(display.equals("block")) {
			box = createBlockBox(style);
		} else if(display.equals("list-item")) {
			box = createListItemBox(style);
		} else if(display.equals("none")) {
		} else if(display.equals("run-in")) {
			box = createRunInBox(style);
		} else if(display.equals("inline-block")) {
			// TODO
		} else if(display.equals("table")) {
			box = createTableBox(style);
		} else if(display.equals("inline-table")) {
			box = createInlineTableBox(style);
		} else if(display.equals("table-row-group")) {
			// Sub-table elements don't get through this anyway
		} else if(display.equals("table-row")) {
			box = createTableRowBox(style);
		} else if(display.equals("table-cell")) {
			box = createTableCellBox(style);
		} else {
			// By default, return an inline
			box = createInlineBox(style);
		}
		if(box != null) {
			box.setErrorHandler(errorHandler);
		}
		return box;
	}

	public CSSInlineBox createInlineBox(CSS2ComputedProperties style) {
		return new InlineBox(style);
	}

	public CSSContainerBox createBlockBox(CSS2ComputedProperties style) {
		return new BlockBox(style);
	}

	public CSSContainerBox createListItemBox(CSS2ComputedProperties style) {
		return new ListItemBox(style);
	}

	public CSSBox createRunInBox(CSS2ComputedProperties style) {
		return new RunInBox(style);
	}

	public CSSContainerBox createTableBox(CSS2ComputedProperties style) {
		return new Table(style);
	}

	public CSSContainerBox createInlineTableBox(CSS2ComputedProperties style) {
		return new InlineTable(style);
	}

	public CSSTableRowBox createTableRowBox(CSS2ComputedProperties style) {
		return new TableRowBox(style);
	}

	public CSSTableCellBox createTableCellBox(CSS2ComputedProperties style) {
		return new TableCellBox(style);
	}

}
