/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.CSSErrorHandler;
import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.dom.ComputedCSSStyle;
import info.informatica.doc.style.css.property.CSSNumberValue;
import info.informatica.doc.style.css.property.CSSPercentageValue;
import info.informatica.doc.style.css.property.CSSStringValue;
import info.informatica.doc.style.css.visual.CSSBox;
import info.informatica.doc.style.css.visual.CSSContainerBox;
import info.informatica.doc.style.css.visual.CSSPoint;

import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.CSSValue;

/**
 * Abstract CSS box.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
abstract public class AbstractCSSBox implements CSSBox {
	
	private CSS2ComputedProperties style;
	
	private CSSContainerBox containingBlock = null;
	
	private CSSErrorHandler errorHandler = null;
	
	public AbstractCSSBox(CSS2ComputedProperties style) {
		super();
		this.style = style;
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getComputedStyle()
	 */
	public CSS2ComputedProperties getComputedStyle() {
		return style;
	}

	protected CSSPrimitiveValue getCSSValue(String property) {
		return (CSSPrimitiveValue) ((ComputedCSSStyle)getComputedStyle())
			.getCSSValue(property);
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getContainingBlock()
	 */
	public CSSContainerBox getContainingBlock() {
		return containingBlock;
	}

	public void setContainingBlock(CSSContainerBox enclosingBlock) {
		this.containingBlock = enclosingBlock;
	}

	StyleDatabase getStyleDatabase() {
		return style.getStyleDatabase();
	}

	protected CSSErrorHandler getErrorHandler() {
		return errorHandler;
	}

	public void setErrorHandler(CSSErrorHandler errorHandler) {
		this.errorHandler = errorHandler;
	}

	public CSSBox getGeneratedSibling() {
		return null;
	}

	protected float computeNumberOrZero(String property) {
		return computeNumberOrZero(getCSSValue(property));
	}

	/**
	 * Computes the float value of a number, or returns zero if finds 
	 * a non-numeric value.
	 * <p>
	 * The container width is used as the base for percentages.
	 * 
	 * @param cssValue the CSS object value.
	 * @return the float value, or zero.
	 */
	protected float computeNumberOrZero(CSSPrimitiveValue cssValue) {
		float value;
		if (cssValue.getPrimitiveType() == CSSPrimitiveValue.CSS_PERCENTAGE) {
			value = cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE);
			// Check if we can avoid calling the container width
			if(value != 0f) {
				value = getContainerWidth() * value / 100f;
			}
		} else if (cssValue.getPrimitiveType() == CSSPrimitiveValue.CSS_EMS) {
			value = getFontSize()
					* cssValue.getFloatValue(CSSPrimitiveValue.CSS_EMS);
		} else if (cssValue instanceof CSSNumberValue) {
			value = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else {
			value = 0;
		}
		return value;
	}

	/**
	 * Gets the width of this box.
	 * <p>
	 * This is defined to be the value of the width property, for 
	 * boxes where it is defined, or the content width otherwise.
	 * 
	 * @return the width of this box.
	 */
	abstract protected float getWidth();

	/**
	 * Gets the height of this box.
	 * <p>
	 * This is defined to be the value of the height property, for 
	 * boxes where it is defined, or the content height otherwise.
	 * 
	 * @return the height of this box.
	 */
	abstract protected float getHeight();

	protected boolean isWidthAuto() {
		CSSPrimitiveValue cssVal = getCSSValue("width");
		if(cssVal instanceof CSSStringValue) {
			// Assume String means 'auto'
			return true;
		}
		return false;
	}

	protected boolean isHeightAuto() {
		CSSPrimitiveValue cssVal = getCSSValue("height");
		if(cssVal instanceof CSSStringValue) {
			// Assume String means 'auto'
			return true;
		}
		return false;
	}

	protected float getContainerWidth() {
		float containerWidth;
		CSSContainerBox container = getContainingBlock();
		if(container != null){
			containerWidth = container.getWidth();
		} else {
			// root block
			containerWidth = getStyleDatabase().getDocumentWidth();
		}
		return containerWidth;
	}

	protected boolean isContainerHeightAuto() {
		CSSContainerBox container = getContainingBlock();
		if(container != null){
			CSSValue cssVal = container
				.getComputedStyle().getPropertyCSSValue("height");
			if(cssVal instanceof CSSStringValue) {
				// Assume String means 'auto'
				return true;
			} else {
				return false;
			}
		} else {
			// root block
			return false;
		}
	}

	/**
	 * Performs a shrink-to-fit computation as defined by the CSS 
	 * specification, paragraph 10.3.5.
	 * 
	 * @param preferredMinimum
	 * @param available
	 * @param preferred
	 * @return the shrinked-to-fit value.
	 */
	protected float shrinkToFit(float preferredMinimum, float available, 
			float preferred) {
		float shrink;
		if(preferredMinimum >= available){
			shrink = preferredMinimum;
		} else {
			shrink = available;
		}
		if(shrink > preferred) {
			shrink = preferred;
		}
		return shrink;
	}

	protected boolean overflowsWithScroll() {
		return getCSSValue("overflow")
			.getStringValue().equals("scroll");
	}

	protected float textLengthToNaturalUnit(int length) {
		return getStyleDatabase().floatValueConversion(length * 
				getStyleDatabase().getExSizeInPt(getFontFamily(), getFontSize()), 
				CSSPrimitiveValue.CSS_PT);
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getLeft()
	 */
	public float getLeft() {
		return computeNumberOrZero("left");
	}
	
	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getRight()
	 */
	public float getRight() {
		return computeNumberOrZero("right");
	}
	
	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getBorderTopWidth()
	 */
	public float getBorderTopWidth() {
		return computeBorderWidth("border-top-width");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getBorderRightWidth()
	 */
	public float getBorderRightWidth() {
		return computeBorderWidth("border-right-width");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getBorderBottomWidth()
	 */
	public float getBorderBottomWidth() {
		return computeBorderWidth("border-bottom-width");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getBorderLeftWidth()
	 */
	public float getBorderLeftWidth() {
		return computeBorderWidth("border-left-width");
	}
	
	private float computeBorderWidth(String borderWidthProperty) {
		CSSPrimitiveValue width = getCSSValue(borderWidthProperty);
		if(width instanceof CSSStringValue){
			return getStyleDatabase().getWidthSize(width.getStringValue());
		} else if(width instanceof CSSNumberValue) {
			return width.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else {
			return 0f;
		}
	}

	/**
	 * Gets the value of the margin-top property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-top property.
	 */
	public float getMarginTop() {
		return computeNumberOrZero("margin-top");
	}

	/**
	 * Gets the value of the margin-bottom property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-bottom property.
	 */
	public float getMarginBottom() {
		return computeNumberOrZero("margin-bottom");
	}

	/**
	 * Gets the value of the margin-right property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-right property.
	 */
	public float getMarginRight() {
		return computeNumberOrZero("margin-right");
	}

	/**
	 * Gets the value of the margin-left property, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return  the value of the margin-left property.
	 */
	public float getMarginLeft() {
		return computeNumberOrZero("margin-left");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getPaddingTop()
	 */
	public float getPaddingTop() {
		return getPaddingSubproperty("padding-top");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getPaddingRight()
	 */
	public float getPaddingRight() {
		return getPaddingSubproperty("padding-right");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getPaddingBottom()
	 */
	public float getPaddingBottom() {
		return getPaddingSubproperty("padding-bottom");
	}

	/* (non-Javadoc)
	 * @see info.informatica.doc.style.css.CSSBox#getPaddingLeft()
	 */
	public float getPaddingLeft() {
		return getPaddingSubproperty("padding-left");
	}

	protected float getPaddingSubproperty(String subpropertyName) {
		float padding;
		CSSPrimitiveValue cssval = getCSSValue(subpropertyName);
		if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_PERCENTAGE) {
			CSS2ComputedProperties styledecl = getComputedStyle();
			String display = styledecl.getPropertyValue("display");
			// check for table cell
			if ("table-cell".equals(display)) {
				// loop until display: table
				CSS2ComputedProperties pStyledecl = styledecl;
				do {
					pStyledecl = pStyledecl.getParentComputedStyle();
					if (pStyledecl == null) {
						if(errorHandler != null) {
							errorHandler.error("Unable to find table ancestor for "
								+ styledecl.getPeerXPath(), subpropertyName, cssval);
						}
						padding = 0;
					} else {
						display = pStyledecl.getPropertyValue("display");
					}
				} while ("table".equals(display));
			} else {
				// loop until display: block
				CSS2ComputedProperties pStyledecl = styledecl;
				do {
					pStyledecl = pStyledecl.getParentComputedStyle();
					if (pStyledecl == null) {
						if(errorHandler != null) {
							errorHandler.error("Unable to find enclosing block for "
								+ styledecl.getPeerXPath(), subpropertyName, cssval);
						}
						padding = 0;
					} else {
						display = pStyledecl.getPropertyValue("display");
					}
				} while ("block".equals(display));
			}
			// Compute width for ancestor
			CSSValue cssWidth = styledecl.getPropertyCSSValue("width");
			if (cssWidth instanceof CSSNumberValue) {
				padding = cssval.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE);
				if(padding != 0f) {
					float parentWidth = ((CSSNumberValue)cssWidth)
						.getFloatValue(getStyleDatabase().getNaturalUnit());
					padding = parentWidth * padding / 100f;
				} else {
					padding = 0f;
				}
			} else {
				// auto (we need a percent of 'auto')
				if(errorHandler != null) {
					errorHandler.error("Got percent of an auto value at "
						+ styledecl.getPeerXPath(), subpropertyName, cssval);
				}
				padding = 0f;
			}
		} else if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_EMS) {
			padding = getFontSize()
					* cssval.getFloatValue(CSSPrimitiveValue.CSS_EMS);
		} else if (cssval instanceof CSSNumberValue) {
			padding = cssval.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else {
			padding = 0f;
		}
		return padding;
	}

	/*
	 * Fonts
	 */

	/**
	 * Gets the 'used' value for the font-family property.
	 * <p>This method requires a style database.</p>
	 * 
	 * @return the value of the font-family property.
	 * @throws IllegalStateException if the style database has not been set.
	 */
	public String getFontFamily() {
		return getComputedStyle().getFontFamily();
	}

	/**
	 * Gets the computed font weight.
	 * 
	 * @return the font weight.
	 */
	public String getFontWeight() {
		return getComputedStyle().getFontWeight();
	}

	/**
	 * Gets the computed value of the font-size property.
	 * <p>May require a style database to work.</p>
	 * 
	 * @return the value of the font-size property, in typographic
	 * points.
	 */
	public int getFontSize() {
		return getComputedStyle().getFontSize();
	}

	public float getLineHeight() {
		return getLineHeight(getFontSize() * 1.16f);
	}

	public float getLineHeight(float defval) {
		float height;
		CSSPrimitiveValue cssval = (CSSPrimitiveValue) getCSSValue("line-height");
		if (cssval == null) {
			// "No 'line-height' found, applying default
			return defval;
		}
		if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_PERCENTAGE) {
			height = getFontSize()
					* cssval.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE)
					/ 100f;
		} else if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_EMS) {
			height = getFontSize()
					* cssval.getFloatValue(CSSPrimitiveValue.CSS_EMS);
		} else if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_IDENT 
				|| cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_STRING) {
			// expect "normal"
			if (!"normal".equals(cssval.getStringValue())) {
				if(errorHandler != null) {
					errorHandler.error("Expected 'normal', found "
						+ cssval.getStringValue());
				}
			}
			height = defval;
		} else if (cssval instanceof CSSNumberValue) {
			height = cssval.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else {
			if(errorHandler != null) {
				errorHandler.error("Expected number or identifier, found "
							+ cssval.getCssText());
			}
			height = defval;
		}
		return height;
	}

	public float getLeading() {
		return getLeading(getFontSize() * 0.16f);
	}

	public float getLeading(float defval) {
		float leading;
		CSSPrimitiveValue cssval = (CSSPrimitiveValue) getCSSValue("line-height");
		if (cssval == null) {
			// No 'line-height' found, applying default
			return defval;
		}
		if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_PERCENTAGE) {
			leading = getFontSize()
					* (cssval.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) - 100f)
					/ 100f;
		} else if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_EMS) {
			leading = getFontSize()
					* (cssval.getFloatValue(CSSPrimitiveValue.CSS_EMS) - 1f);
		} else if (cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_IDENT
				|| cssval.getPrimitiveType() == CSSPrimitiveValue.CSS_STRING) {
			// expect "normal"
			if (!"normal".equals(cssval.getStringValue())) {
				if(errorHandler != null) {
					errorHandler.error("Expected 'normal', found "
							+ cssval.getStringValue());
				}
			}
			leading = defval;
		} else if (cssval instanceof CSSNumberValue) {
			leading = cssval.getFloatValue(getStyleDatabase().getNaturalUnit()) - getFontSize();
		} else {
			if(errorHandler != null) {
				errorHandler.error("Expected number or identifier, found "
							+ cssval.getCssText());
			}
			leading = defval;
		}
		return leading;
	}

	public CSSValue getLineHeightCSSValue() {
		return getCSSValue("line-height");
	}

	public CSSPrimitiveValue getBorderTopColor() {
		return (CSSPrimitiveValue) getCSSValue("border-top-color");
	}

	public CSSPrimitiveValue getBorderRightColor() {
		return (CSSPrimitiveValue) getCSSValue("border-right-color");
	}

	public CSSPrimitiveValue getBorderBottomColor() {
		return (CSSPrimitiveValue) getCSSValue("border-bottom-color");
	}

	public CSSPrimitiveValue getBorderLeftColor() {
		return (CSSPrimitiveValue) getCSSValue("border-left-color");
	}

	public CSSPoint getBackgroundPosition() {
		CSSPrimitiveValue cssVal = (CSSPrimitiveValue) getCSSValue(
				"background-position");
		CSSPoint point = new CSSPoint();
		if(cssVal instanceof CSSNumberValue) {
			float fval = getComputedStyle().computeFloatValue(
					(CSSNumberValue)cssVal);
			if((cssVal instanceof CSSPercentageValue) && fval != 0f){
				point.setLeft(fval * getWidth() * 0.01f);
			} else {
				point.setLeft(fval);
			}
			cssVal = ((CSSNumberValue)cssVal).nextPrimitiveValue();
			if(cssVal == null) {
				point.setTop(getWidth() * 0.5f);
			} else if(cssVal instanceof CSSNumberValue) {
				fval = getComputedStyle().computeFloatValue(
						(CSSNumberValue)cssVal);
				if((cssVal instanceof CSSPercentageValue) && fval != 0f){
					point.setTop(fval * getHeight() * 0.01f);
				} else {
					point.setTop(fval);
				}
			} else if(cssVal instanceof CSSStringValue) {
				String ident = cssVal.getStringValue();
				if(ident.equals("top")) {
					point.setTop(getHeight());
				} else if(ident.equals("bottom")) {
					point.setTop(0f);
				} else {
					// Center
					point.setTop(getHeight() * 0.5f);
				}
			}
		} else if(cssVal instanceof CSSStringValue) {
			String ident = cssVal.getStringValue();
			cssVal = ((CSSStringValue)cssVal).nextPrimitiveValue();
			if(cssVal == null) {
				point.setTop(getWidth() * 0.5f);
			} else if(cssVal instanceof CSSNumberValue) {
				if(ident.equals("left")) {
					point.setLeft(0f);
				} else if(ident.equals("right")) {
					point.setLeft(getWidth());
				} else {
					// Center
					point.setLeft(getWidth() * 0.5f);
				}
				float fval = getComputedStyle().computeFloatValue(
						(CSSNumberValue)cssVal);
				if((cssVal instanceof CSSPercentageValue) && fval != 0f){
					point.setTop(fval * getHeight() * 0.01f);
				} else {
					point.setTop(fval);
				}
			} else if(cssVal instanceof CSSStringValue) {
				String ident2 = cssVal.getStringValue();
				if(ident.equals("left") || ident2.equals("left")) {
					point.setLeft(0f);
				} else if(ident.equals("right") || ident2.equals("right")) {
					point.setLeft(getWidth());
				} else {
					// Center
					point.setLeft(getWidth() * 0.5f);
				}
				if(ident.equals("top") || ident2.equals("top")) {
					point.setLeft(0f);
				} else if(ident.equals("bottom") || ident2.equals("bottom")) {
					point.setLeft(getHeight());
				} else {
					// Center
					point.setTop(getHeight() * 0.5f);
				}
			}
		}
		return point;
	}
	
}
