/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.visual.box;

import info.informatica.doc.dom4j.CSSStylableElement;
import info.informatica.doc.style.css.CSS2ComputedProperties;
import info.informatica.doc.style.css.property.CSSNumberValue;
import info.informatica.doc.style.css.visual.ReplacedElementBox;

import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * Abstract base class for replaced element boxes.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 *
 */
public abstract class AbstractReplacedBox<C> extends AbstractCSSBox implements ReplacedElementBox<C> {

	private CSSStylableElement replacedElement;
	
	public AbstractReplacedBox(CSSStylableElement replaced) {
		super((CSS2ComputedProperties)replaced.getComputedStyle());
		replacedElement = replaced;
	}

	protected CSSStylableElement getReplacedElement() {
		return replacedElement;
	}

	/**
	 * Gets the width of this box, expressed in the 
	 * unit given by the <code>StyleDatabase.getNaturalUnit()</code> method.
	 * 
	 * @return the width of this box.
	 */
	public float getWidth() {
		float boxWidth;
		CSSPrimitiveValue cssValue = getCSSValue("width");
		short primiType = cssValue.getPrimitiveType();
		if(primiType == getStyleDatabase().getNaturalUnit()) {
			boxWidth = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else if(primiType == CSSPrimitiveValue.CSS_PERCENTAGE) {
			boxWidth = getContainerWidth() * 
				cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
		} else if(cssValue instanceof CSSNumberValue) {
			// Expressed in non-natural unit
			boxWidth = cssValue.getFloatValue(primiType);
		} else {
			// value is 'auto'
			// First, check for width attribute
			String sWidth = replacedElement.attributeValue("width");
			if(sWidth != null && sWidth.length()>0) {
				try {
					return Integer.parseInt(sWidth);
				}catch(NumberFormatException e) {
					getErrorHandler().error("Could not parse: " + sWidth, "width", cssValue);
				}
			}
			boxWidth = getIntrinsicWidth();
		}
		return boxWidth;
	}

	public float getHeight() {
		float boxHeight;
		CSSPrimitiveValue cssValue = getCSSValue("height");
		short primiType = cssValue.getPrimitiveType();
		if(primiType == getStyleDatabase().getNaturalUnit()) {
			boxHeight = cssValue.getFloatValue(getStyleDatabase().getNaturalUnit());
		} else if(primiType == CSSPrimitiveValue.CSS_PERCENTAGE) {
			boxHeight = getContainerWidth() * 
				cssValue.getFloatValue(CSSPrimitiveValue.CSS_PERCENTAGE) / 100f;
		} else if(cssValue instanceof CSSNumberValue) {
			// Expressed in non-natural unit
			boxHeight = cssValue.getFloatValue(primiType);
		} else {
			// value is 'auto'
			// First, check for height attribute
			String sHeight = replacedElement.attributeValue("height");
			if(sHeight != null && sHeight.length()>0) {
				try {
					return Integer.parseInt(sHeight);
				}catch(NumberFormatException e) {
					getErrorHandler().error("Could not parse: " + sHeight, "height", cssValue);
				}
			}
			boxHeight = getIntrinsicHeight();
		}
		return boxHeight;
	}

}