/*
 * Decompiled with CFR 0.152.
 */
package io.camunda.zeebe.engine.processing.common;

import io.camunda.zeebe.el.EvaluationContext;
import io.camunda.zeebe.el.EvaluationResult;
import io.camunda.zeebe.el.Expression;
import io.camunda.zeebe.el.ExpressionLanguage;
import io.camunda.zeebe.el.ResultType;
import io.camunda.zeebe.engine.processing.common.Failure;
import io.camunda.zeebe.model.bpmn.util.time.Interval;
import io.camunda.zeebe.protocol.record.value.ErrorType;
import io.camunda.zeebe.util.Either;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.agrona.DirectBuffer;
import org.agrona.concurrent.UnsafeBuffer;

public final class ExpressionProcessor {
    private static final List<ResultType> INTERVAL_RESULT_TYPES = List.of(ResultType.DURATION, ResultType.PERIOD, ResultType.STRING);
    private static final List<ResultType> DATE_TIME_RESULT_TYPES = List.of(ResultType.DATE_TIME, ResultType.STRING);
    private static final List<ResultType> NULLABLE_DATE_TIME_RESULT_TYPES = List.of(ResultType.NULL, ResultType.DATE_TIME, ResultType.STRING);
    private static final EvaluationContext EMPTY_EVALUATION_CONTEXT = x -> null;
    private final DirectBuffer resultView = new UnsafeBuffer();
    private final ExpressionLanguage expressionLanguage;
    private final EvaluationContextLookup evaluationContextLookup;

    public ExpressionProcessor(ExpressionLanguage expressionLanguage, EvaluationContextLookup lookup) {
        this.expressionLanguage = expressionLanguage;
        this.evaluationContextLookup = lookup;
    }

    public ExpressionProcessor withPrimaryContext(EvaluationContext primaryContext) {
        EvaluationContextLookup combinedLookup = scopeKey -> primaryContext.combine(this.evaluationContextLookup.getContext(scopeKey));
        return new ExpressionProcessor(this.expressionLanguage, combinedLookup);
    }

    public ExpressionProcessor withSecondaryContext(EvaluationContext secondaryContext) {
        EvaluationContextLookup combinedLookup = scopeKey -> this.evaluationContextLookup.getContext(scopeKey).combine(secondaryContext);
        return new ExpressionProcessor(this.expressionLanguage, combinedLookup);
    }

    public Either<Failure, String> evaluateStringExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.STRING, scopeKey)).map(EvaluationResult::getString);
    }

    public Either<Failure, DirectBuffer> evaluateStringExpressionAsDirectBuffer(Expression expression, long scopeKey) {
        return this.evaluateStringExpression(expression, scopeKey).map(this::wrapResult);
    }

    public Either<Failure, Long> evaluateLongExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.NUMBER, scopeKey)).map(EvaluationResult::getNumber).map(Number::longValue);
    }

    public Either<Failure, Boolean> evaluateBooleanExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.BOOLEAN, scopeKey)).map(EvaluationResult::getBoolean);
    }

    public Either<Failure, Interval> evaluateIntervalExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, (Collection<ResultType>)INTERVAL_RESULT_TYPES, scopeKey)).flatMap(result -> switch (result.getType()) {
            case ResultType.DURATION -> Either.right((Object)new Interval(result.getDuration()));
            case ResultType.PERIOD -> Either.right((Object)new Interval(result.getPeriod()));
            default -> this.parseIntervalString(expression, scopeKey, (EvaluationResult)result);
        });
    }

    private Either<Failure, Interval> parseIntervalString(Expression expression, long scopeKey, EvaluationResult result) {
        try {
            return Either.right((Object)Interval.parse((String)result.getString()));
        }
        catch (DateTimeParseException e) {
            return Either.left((Object)this.createFailureMessage(result, String.format("Invalid duration format '%s' for expression '%s'.", result.getString(), expression.getExpression()), scopeKey));
        }
    }

    public Either<Failure, ZonedDateTime> evaluateDateTimeExpression(Expression expression, Long scopeKey) {
        return this.evaluateDateTimeExpression(expression, scopeKey, false).flatMap(optionalDateTime -> Either.right((Object)((ZonedDateTime)optionalDateTime.get())));
    }

    public Either<Failure, Optional<ZonedDateTime>> evaluateDateTimeExpression(Expression expression, Long scopeKey, boolean isNullable) {
        List<ResultType> dateTimeResultTypes = isNullable ? NULLABLE_DATE_TIME_RESULT_TYPES : DATE_TIME_RESULT_TYPES;
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, (Collection<ResultType>)dateTimeResultTypes, (long)scopeKey)).flatMap(result -> switch (result.getType()) {
            case ResultType.NULL -> Either.right(Optional.empty());
            case ResultType.DATE_TIME -> Either.right(Optional.of(result.getDateTime()));
            default -> this.evaluateDateTimeExpressionString((EvaluationResult)result, scopeKey, isNullable);
        });
    }

    public Either<Failure, DirectBuffer> evaluateAnyExpression(Expression expression, long scopeKey) {
        Either<Failure, EvaluationResult> evaluationResult = this.evaluateExpressionAsEither(expression, scopeKey);
        return evaluationResult.map(EvaluationResult::toBuffer);
    }

    public Either<Failure, List<DirectBuffer>> evaluateArrayExpression(Expression expression, long scopeKey) {
        Either<Failure, EvaluationResult> evaluationResult = this.evaluateExpressionAsEither(expression, scopeKey);
        return evaluationResult.flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.ARRAY, scopeKey)).map(EvaluationResult::getList);
    }

    public Either<Failure, List<String>> evaluateArrayOfStringsExpression(Expression expression, long scopeKey) {
        Either<Failure, EvaluationResult> evaluationResult = this.evaluateExpressionAsEither(expression, scopeKey);
        return evaluationResult.flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.ARRAY, scopeKey)).map(EvaluationResult::getListOfStrings).flatMap(list -> {
            if (list != null) {
                return Either.right((Object)list);
            }
            return Either.left((Object)this.createFailureMessage((EvaluationResult)evaluationResult.get(), String.format("Expected result of the expression '%s' to be 'ARRAY' containing 'STRING' items, but was 'ARRAY' containing at least one non-'STRING' item.", expression.getExpression()), scopeKey));
        });
    }

    public Either<Failure, String> evaluateMessageCorrelationKeyExpression(Expression expression, long scopeKey) {
        Set<ResultType> expectedTypes = Set.of(ResultType.STRING, ResultType.NUMBER);
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheckCorrelationKey(scopeKey, expectedTypes, (EvaluationResult)result, expression)).map(this::toStringFromStringOrNumber);
    }

    private Either<Failure, EvaluationResult> typeCheckCorrelationKey(long scopeKey, Set<ResultType> expectedTypes, EvaluationResult result, Expression expression) {
        return this.typeCheck(result, expectedTypes, scopeKey).mapLeft(failure -> this.createFailureMessage(result, String.format("Failed to extract the correlation key for '%s': The value must be either a string or a number, but was '%s'.", expression.getExpression(), result.getType()), scopeKey));
    }

    private String toStringFromStringOrNumber(EvaluationResult result) {
        return result.getType() == ResultType.NUMBER ? Long.toString(result.getNumber().longValue()) : result.getString();
    }

    public Either<Failure, DirectBuffer> evaluateVariableMappingExpression(Expression expression, long scopeKey) {
        return this.evaluateExpressionAsEither(expression, scopeKey).flatMap(result -> this.typeCheck((EvaluationResult)result, ResultType.OBJECT, scopeKey)).mapLeft(failure -> new Failure(failure.getMessage(), ErrorType.IO_MAPPING_ERROR, scopeKey)).map(EvaluationResult::toBuffer);
    }

    private Either<Failure, EvaluationResult> typeCheck(EvaluationResult result, ResultType expectedResultType, long scopeKey) {
        if (result.getType() != expectedResultType) {
            return Either.left((Object)this.createFailureMessage(result, String.format("Expected result of the expression '%s' to be '%s', but was '%s'.", result.getExpression(), expectedResultType, result.getType()), scopeKey));
        }
        return Either.right((Object)result);
    }

    private Either<Failure, EvaluationResult> typeCheck(EvaluationResult result, Collection<ResultType> expectedResultTypes, long scopeKey) {
        return expectedResultTypes.stream().map(expected -> this.typeCheck(result, (ResultType)expected, scopeKey)).filter(Either::isRight).findFirst().orElse(Either.left((Object)this.createFailureMessage(result, String.format("Expected result of the expression '%s' to be one of '%s', but was '%s'.", result.getExpression(), expectedResultTypes, result.getType()), scopeKey)));
    }

    private EvaluationResult evaluateExpression(Expression expression, long variableScopeKey) {
        EvaluationContext context = variableScopeKey < 0L ? EMPTY_EVALUATION_CONTEXT : this.evaluationContextLookup.getContext(variableScopeKey);
        return this.expressionLanguage.evaluateExpression(expression, context);
    }

    private Either<Failure, EvaluationResult> evaluateExpressionAsEither(Expression expression, long variableScopeKey) {
        EvaluationResult result = this.evaluateExpression(expression, variableScopeKey);
        return result.isFailure() ? Either.left((Object)this.createFailureMessage(result, result.getFailureMessage(), variableScopeKey)) : Either.right((Object)result);
    }

    private Failure createFailureMessage(EvaluationResult evaluationResult, String failureMessage, long variableScopeKey) {
        Object message = failureMessage;
        List evaluationWarnings = evaluationResult.getWarnings();
        if (!evaluationWarnings.isEmpty()) {
            String formattedWarnings = evaluationWarnings.stream().map(warning -> "[%s] %s".formatted(warning.getType(), warning.getMessage())).collect(Collectors.joining("\n"));
            message = (String)message + " The evaluation reported the following warnings:\n%s".formatted(formattedWarnings);
        }
        return new Failure((String)message, ErrorType.EXTRACT_VALUE_ERROR, variableScopeKey);
    }

    private DirectBuffer wrapResult(String result) {
        this.resultView.wrap(result.getBytes());
        return this.resultView;
    }

    private Either<Failure, Optional<ZonedDateTime>> evaluateDateTimeExpressionString(EvaluationResult result, Long scopeKey, boolean isNullable) {
        String resultString = result.getString();
        if (isNullable && resultString.isEmpty()) {
            return Either.right(Optional.empty());
        }
        try {
            return Either.right(Optional.of(ZonedDateTime.parse(resultString)));
        }
        catch (DateTimeParseException e) {
            return Either.left((Object)this.createFailureMessage(result, String.format("Invalid date-time format '%s' for expression '%s'.", resultString, result.getExpression()), scopeKey));
        }
    }

    @FunctionalInterface
    public static interface EvaluationContextLookup {
        public EvaluationContext getContext(long var1);
    }

    public static final class EvaluationException
    extends RuntimeException {
        public EvaluationException(String message) {
            super(message);
        }
    }
}

