/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2024 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonAlias;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonSetter;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.annotation.Nulls;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * JSON Schema for Common Vulnerability Scoring System version 3.0
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
// TODO fix the order - added entries that aren't here
@JsonPropertyOrder({"version", "vectorString", "attackVector", "attackComplexity", "attackRequirements",
        "privilegesRequired", "userInteraction", "vulnerableSystemConfidentiality", "vulnerableSystemIntegrity",
        "vulnerableSystemAvailability", "subsequentSystemConfidentiality", "subsequentSystemIntegrity",
        "subsequentSystemAvailability", "exploitMaturity", "confidentialityRequirements", "integrityRequirements",
        "availabilityRequirements", "modifiedAttackVector", "modifiedAttackComplexity", "modifiedAttackRequirements",
        "modifiedPrivilegesRequired", "modifiedUserInteraction", "modifiedVulnerableSystemConfidentiality",
        "modifiedVulnerableSystemIntegrity", "modifiedVulnerableSystemAvailability",
        "modifiedSubsequentSystemConfidentiality", "modifiedSubsequentSystemIntegrity",
        "modifiedSubsequentSystemAvailability", "safety", "automatable", "recovery", "valueDensity",
        "vulnerabilityResponseEffort", "providerUrgency", "baseScore", "baseSeverity", "threatScore", "threatSeverity",
        "environmentalScore", "environmentalSeverity"})
public class CvssV4Data implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 8537782209754450697L;

    public CvssV4Data() {
    }

    public CvssV4Data(Version version, String vectorString, AttackVectorType attackVector,
            AttackComplexityType attackComplexity, AttackRequirementsType attackRequirements,
            PrivilegesRequiredType privilegesRequired, UserInteractionType userInteraction,
            CiaType vulnConfidentialityImpact, CiaType vulnIntegrityImpact, CiaType vulnAvailabilityImpact,
            CiaType subConfidentialityImpact, CiaType subIntegrityImpact, CiaType subAvailabilityImpact,
            ExploitMaturityType exploitMaturity, CiaRequirementType confidentialityRequirement,
            CiaRequirementType integrityRequirement, CiaRequirementType availabilityRequirement,
            ModifiedAttackVectorType modifiedAttackVector, ModifiedAttackComplexityType modifiedAttackComplexity,
            ModifiedAttackRequirementsType modifiedAttackRequirements,
            ModifiedPrivilegesRequiredType modifiedPrivilegesRequired,
            ModifiedUserInteractionType modifiedUserInteraction, ModifiedCiaType modifiedVulnConfidentialityImpact,
            ModifiedCiaType modifiedVulnIntegrityImpact, ModifiedCiaType modifiedVulnAvailabilityImpact,
            ModifiedCiaType modifiedSubConfidentialityImpact, ModifiedCiaType modifiedSubIntegrityImpact,
            ModifiedCiaType modifiedSubAvailabilityImpact, SafetyType safety, AutomatableType automatable,
            RecoveryType recovery, ValueDensityType valueDensity,
            VulnerabilityResponseEffortType vulnerabilityResponseEffort, ProviderUrgencyType providerUrgency,
            Double baseScore, SeverityType baseSeverity, Double threatScore, SeverityType threatSeverity,
            Double environmentalScore, SeverityType environmentalSeverity) {
        this.version = version;
        this.vectorString = vectorString;
        this.attackVector = attackVector;
        this.attackComplexity = attackComplexity;
        this.attackRequirements = attackRequirements;
        this.privilegesRequired = privilegesRequired;
        this.userInteraction = userInteraction;
        this.vulnConfidentialityImpact = vulnConfidentialityImpact;
        this.vulnIntegrityImpact = vulnIntegrityImpact;
        this.vulnAvailabilityImpact = vulnAvailabilityImpact;
        this.subConfidentialityImpact = subConfidentialityImpact;
        this.subIntegrityImpact = subIntegrityImpact;
        this.subAvailabilityImpact = subAvailabilityImpact;
        this.exploitMaturity = exploitMaturity;
        this.confidentialityRequirement = confidentialityRequirement;
        this.integrityRequirement = integrityRequirement;
        this.availabilityRequirement = availabilityRequirement;
        this.modifiedAttackVector = modifiedAttackVector;
        this.modifiedAttackComplexity = modifiedAttackComplexity;
        this.modifiedAttackRequirements = modifiedAttackRequirements;
        this.modifiedPrivilegesRequired = modifiedPrivilegesRequired;
        this.modifiedUserInteraction = modifiedUserInteraction;
        this.modifiedVulnConfidentialityImpact = modifiedVulnConfidentialityImpact;
        this.modifiedVulnIntegrityImpact = modifiedVulnIntegrityImpact;
        this.modifiedVulnAvailabilityImpact = modifiedVulnAvailabilityImpact;
        this.modifiedSubConfidentialityImpact = modifiedSubConfidentialityImpact;
        this.modifiedSubIntegrityImpact = modifiedSubIntegrityImpact;
        this.modifiedSubAvailabilityImpact = modifiedSubAvailabilityImpact;
        this.safety = safety;
        this.automatable = automatable;
        this.recovery = recovery;
        this.valueDensity = valueDensity;
        this.vulnerabilityResponseEffort = vulnerabilityResponseEffort;
        this.providerUrgency = providerUrgency;
        this.baseScore = baseScore;
        this.baseSeverity = baseSeverity;
        this.threatScore = threatScore;
        this.threatSeverity = threatSeverity;
        this.environmentalScore = environmentalScore;
        this.environmentalSeverity = environmentalSeverity;
    }

    /**
     * CVSS Version (Required)
     */
    @JsonProperty("version")
    @JsonPropertyDescription("CVSS Version")
    private Version version;
    /**
     * (Required)
     */
    @JsonProperty("vectorString")
    private String vectorString;
    @JsonProperty("attackVector")
    private AttackVectorType attackVector;
    @JsonProperty("attackComplexity")
    private AttackComplexityType attackComplexity;
    @JsonProperty("attackRequirements")
    private AttackRequirementsType attackRequirements;
    @JsonProperty("privilegesRequired")
    private PrivilegesRequiredType privilegesRequired;
    @JsonProperty("userInteraction")
    private UserInteractionType userInteraction;
    @JsonProperty("vulnerableSystemConfidentiality")
    @JsonAlias("vulnConfidentialityImpact")
    private CiaType vulnConfidentialityImpact;
    @JsonProperty("vulnerableSystemIntegrity")
    @JsonAlias("vulnIntegrityImpact")
    private CiaType vulnIntegrityImpact;
    @JsonProperty("vulnerableSystemAvailability")
    @JsonAlias("vulnAvailabilityImpact")
    private CiaType vulnAvailabilityImpact;
    @JsonProperty("subsequentSystemConfidentiality")
    @JsonAlias("subConfidentialityImpact")
    private CiaType subConfidentialityImpact;
    @JsonProperty("subsequentSystemIntegrity")
    @JsonAlias("subIntegrityImpact")
    private CiaType subIntegrityImpact;
    @JsonProperty("subsequentSystemAvailability")
    @JsonAlias("subAvailabilityImpact")
    private CiaType subAvailabilityImpact;
    @JsonProperty("exploitMaturity")
    @JsonSetter(nulls = Nulls.SKIP)
    private ExploitMaturityType exploitMaturity = ExploitMaturityType.NOT_DEFINED;
    @JsonProperty("confidentialityRequirements")
    @JsonAlias("confidentialityRequirement")
    private CiaRequirementType confidentialityRequirement;
    @JsonProperty("integrityRequirements")
    @JsonAlias("integrityRequirement")
    private CiaRequirementType integrityRequirement;
    @JsonProperty("availabilityRequirements")
    @JsonAlias("availabilityRequirement")
    private CiaRequirementType availabilityRequirement;
    @JsonProperty("modifiedAttackVector")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedAttackVectorType modifiedAttackVector = ModifiedAttackVectorType.NOT_DEFINED;
    @JsonProperty("modifiedAttackComplexity")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedAttackComplexityType modifiedAttackComplexity = ModifiedAttackComplexityType.NOT_DEFINED;
    @JsonProperty("modifiedAttackRequirements")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedAttackRequirementsType modifiedAttackRequirements = ModifiedAttackRequirementsType.NOT_DEFINED;
    @JsonProperty("modifiedPrivilegesRequired")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedPrivilegesRequiredType modifiedPrivilegesRequired = ModifiedPrivilegesRequiredType.NOT_DEFINED;
    @JsonProperty("modifiedUserInteraction")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedUserInteractionType modifiedUserInteraction = ModifiedUserInteractionType.NOT_DEFINED;
    @JsonProperty("modifiedVulnerableSystemConfidentiality")
    @JsonAlias("modifiedVulnConfidentialityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedCiaType modifiedVulnConfidentialityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonProperty("modifiedVulnerableSystemIntegrity")
    @JsonAlias("modifiedVulnIntegrityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedCiaType modifiedVulnIntegrityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonProperty("modifiedVulnerableSystemAvailability")
    @JsonAlias("modifiedVulnAvailabilityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedCiaType modifiedVulnAvailabilityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonProperty("modifiedSubsequentSystemConfidentiality")
    @JsonAlias("modifiedSubConfidentialityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedCiaType modifiedSubConfidentialityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonProperty("modifiedSubsequentSystemIntegrity")
    @JsonAlias("modifiedSubIntegrityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedCiaType modifiedSubIntegrityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonProperty("modifiedSubsequentSystemAvailability")
    @JsonAlias("modifiedSubAvailabilityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private ModifiedCiaType modifiedSubAvailabilityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonProperty("safety")
    @JsonAlias("Safety")
    @JsonSetter(nulls = Nulls.SKIP)
    private SafetyType safety = SafetyType.NOT_DEFINED;
    @JsonProperty("automatable")
    @JsonAlias("Automatable")
    @JsonSetter(nulls = Nulls.SKIP)
    private AutomatableType automatable = AutomatableType.NOT_DEFINED;
    @JsonProperty("recovery")
    @JsonAlias("Recovery")
    @JsonSetter(nulls = Nulls.SKIP)
    private RecoveryType recovery = RecoveryType.NOT_DEFINED;
    @JsonProperty("valueDensity")
    @JsonSetter(nulls = Nulls.SKIP)
    private ValueDensityType valueDensity = ValueDensityType.NOT_DEFINED;
    @JsonProperty("vulnerabilityResponseEffort")
    @JsonSetter(nulls = Nulls.SKIP)
    private VulnerabilityResponseEffortType vulnerabilityResponseEffort = VulnerabilityResponseEffortType.NOT_DEFINED;
    @JsonProperty("providerUrgency")
    @JsonSetter(nulls = Nulls.SKIP)
    private ProviderUrgencyType providerUrgency = ProviderUrgencyType.NOT_DEFINED;
    /**
     * (Required)
     */
    @JsonProperty("baseScore")
    private Double baseScore;
    /**
     * (Required)
     */
    @JsonProperty("baseSeverity")
    private SeverityType baseSeverity;
    @JsonProperty("threatScore")
    private Double threatScore;
    @JsonProperty("threatSeverity")
    private SeverityType threatSeverity;
    @JsonProperty("environmentalScore")
    private Double environmentalScore;
    @JsonProperty("environmentalSeverity")
    private SeverityType environmentalSeverity;

    /**
     * CVSS Version (Required)
     *
     * @return version
     */
    @JsonProperty("version")
    public Version getVersion() {
        return version;
    }

    /**
     * (Required)
     *
     * @return vectorString
     */
    @JsonProperty("vectorString")
    public String getVectorString() {
        return vectorString;
    }

    /**
     * @return attackVector
     */
    @JsonProperty("attackVector")
    public AttackVectorType getAttackVector() {
        return attackVector;
    }

    /**
     * @return attackComplexity
     */
    @JsonProperty("attackComplexity")
    public AttackComplexityType getAttackComplexity() {
        return attackComplexity;
    }

    /**
     * @return attackRequirements
     */
    @JsonProperty("attackRequirements")
    public AttackRequirementsType getAttackRequirements() {
        return attackRequirements;
    }

    /**
     * @return privilegesRequired
     */
    @JsonProperty("privilegesRequired")
    public PrivilegesRequiredType getPrivilegesRequired() {
        return privilegesRequired;
    }

    /**
     * @return userInteraction
     */
    @JsonProperty("userInteraction")
    public UserInteractionType getUserInteraction() {
        return userInteraction;
    }

    /**
     * @return vulnConfidentialityImpact
     */
    @JsonProperty("vulnConfidentialityImpact")
    public CiaType getVulnConfidentialityImpact() {
        return vulnConfidentialityImpact;
    }

    /**
     * @return vulnIntegrityImpact
     */
    @JsonProperty("vulnIntegrityImpact")
    public CiaType getVulnIntegrityImpact() {
        return vulnIntegrityImpact;
    }

    /**
     * @return vulnAvailabilityImpact
     */
    @JsonProperty("vulnAvailabilityImpact")
    public CiaType getVulnAvailabilityImpact() {
        return vulnAvailabilityImpact;
    }

    /**
     * @return subConfidentialityImpact
     */
    @JsonProperty("subConfidentialityImpact")
    public CiaType getSubConfidentialityImpact() {
        return subConfidentialityImpact;
    }

    /**
     * @return subIntegrityImpact
     */
    @JsonProperty("subIntegrityImpact")
    public CiaType getSubIntegrityImpact() {
        return subIntegrityImpact;
    }

    /**
     * @return subAvailabilityImpact
     */
    @JsonProperty("subAvailabilityImpact")
    public CiaType getSubAvailabilityImpact() {
        return subAvailabilityImpact;
    }

    /**
     * @return exploitMaturity
     */
    @JsonProperty("exploitMaturity")
    public ExploitMaturityType getExploitMaturity() {
        return exploitMaturity;
    }

    /**
     * (Required)
     *
     * @return baseScore
     */
    @JsonProperty("baseScore")
    public Double getBaseScore() {
        return baseScore;
    }

    /**
     * (Required)
     *
     * @return baseSeverity
     */
    @JsonProperty("baseSeverity")
    public SeverityType getBaseSeverity() {
        return baseSeverity;
    }

    /**
     * @return confidentialityRequirement
     */
    @JsonProperty("confidentialityRequirement")
    public CiaRequirementType getConfidentialityRequirement() {
        return confidentialityRequirement;
    }

    /**
     * @return integrityRequirement
     */
    @JsonProperty("integrityRequirement")
    public CiaRequirementType getIntegrityRequirement() {
        return integrityRequirement;
    }

    /**
     * @return availabilityRequirement
     */
    @JsonProperty("availabilityRequirement")
    public CiaRequirementType getAvailabilityRequirement() {
        return availabilityRequirement;
    }

    /**
     * @return modifiedAttackVector
     */
    @JsonProperty("modifiedAttackVector")
    public ModifiedAttackVectorType getModifiedAttackVector() {
        return modifiedAttackVector;
    }

    /**
     * @return modifiedAttackComplexity
     */
    @JsonProperty("modifiedAttackComplexity")
    public ModifiedAttackComplexityType getModifiedAttackComplexity() {
        return modifiedAttackComplexity;
    }

    /**
     * @return modifiedAttackRequirements
     */
    @JsonProperty("modifiedAttackRequirements")
    public ModifiedAttackRequirementsType getModifiedAttackRequirements() {
        return modifiedAttackRequirements;
    }

    /**
     * @return modifiedPrivilegesRequired
     */
    @JsonProperty("modifiedPrivilegesRequired")
    public ModifiedPrivilegesRequiredType getModifiedPrivilegesRequired() {
        return modifiedPrivilegesRequired;
    }

    /**
     * @return modifiedUserInteraction
     */
    @JsonProperty("modifiedUserInteraction")
    public ModifiedUserInteractionType getModifiedUserInteraction() {
        return modifiedUserInteraction;
    }

    /**
     * @return modifiedVulnConfidentialityImpact
     */
    @JsonProperty("modifiedVulnConfidentialityImpact")
    public ModifiedCiaType getModifiedVulnConfidentialityImpact() {
        return modifiedVulnConfidentialityImpact;
    }

    /**
     * @return modifiedVulnIntegrityImpact
     */
    @JsonProperty("modifiedVulnIntegrityImpact")
    public ModifiedCiaType getModifiedVulnIntegrityImpact() {
        return modifiedVulnIntegrityImpact;
    }

    /**
     * @return modifiedVulnAvailabilityImpact
     */
    @JsonProperty("modifiedVulnAvailabilityImpact")
    public ModifiedCiaType getModifiedVulnAvailabilityImpact() {
        return modifiedVulnAvailabilityImpact;
    }

    /**
     * @return modifiedSubConfidentialityImpact
     */
    @JsonProperty("modifiedSubConfidentialityImpact")
    public ModifiedCiaType getModifiedSubConfidentialityImpact() {
        return modifiedSubConfidentialityImpact;
    }

    /**
     * @return modifiedSubIntegrityImpact
     */
    @JsonProperty("modifiedSubIntegrityImpact")
    public ModifiedCiaType getModifiedSubIntegrityImpact() {
        return modifiedSubIntegrityImpact;
    }

    /**
     * @return modifiedSubAvailabilityImpact
     */
    @JsonProperty("modifiedSubAvailabilityImpact")
    public ModifiedCiaType getModifiedSubAvailabilityImpact() {
        return modifiedSubAvailabilityImpact;
    }

    /**
     * @return safety
     */
    @JsonProperty("Safety")
    @JsonAlias("safety")
    public SafetyType getSafety() {
        return safety;
    }

    /**
     * @return automatable
     */
    @JsonProperty("Automatable")
    @JsonAlias("automatable")
    public AutomatableType getAutomatable() {
        return automatable;
    }

    /**
     * @return recovery
     */
    @JsonProperty("Recovery")
    @JsonAlias("recovery")
    public RecoveryType getRecovery() {
        return recovery;
    }

    /**
     * @return valueDensity
     */
    @JsonProperty("valueDensity")
    public ValueDensityType getValueDensity() {
        return valueDensity;
    }

    /**
     * @return vulnerabilityResponseEffort
     */
    @JsonProperty("vulnerabilityResponseEffort")
    public VulnerabilityResponseEffortType getVulnerabilityResponseEffort() {
        return vulnerabilityResponseEffort;
    }

    /**
     * @return providerUrgency
     */
    @JsonProperty("providerUrgency")
    public ProviderUrgencyType getProviderUrgency() {
        return providerUrgency;
    }

    /**
     * @return threatScore
     */
    @JsonProperty("threatScore")
    public Double getThreatScore() {
        return threatScore;
    }

    /**
     * @return threatSeverity
     */
    @JsonProperty("threatSeverity")
    public SeverityType getThreatSeverity() {
        return threatSeverity;
    }

    /**
     * @return environmentalScore
     */
    @JsonProperty("environmentalScore")
    public Double getEnvironmentalScore() {
        return environmentalScore;
    }

    /**
     * @return environmentalSeverity
     */
    @JsonProperty("environmentalSeverity")
    public SeverityType getEnvironmentalSeverity() {
        return environmentalSeverity;
    }

    @Override
    public String toString() {
        // "pattern":
        // "^CVSS:4[.]0\/AV:[NALP]\/AC:[LH]\/AT:[NP]\/PR:[NLH]\/UI:[NPA]\/VC:[HLN]\/VI:[HLN]\/VA:[HLN]\/SC:[HLN]\/SI:[HLN]\/SA:[HLN]
        // TODO fix the string format
        StringBuilder v = new StringBuilder("CVSS:").append(version == null ? "" : version).append("/AV:")
                .append(attackVector == null ? "" : attackVector.value().charAt(0)).append("/AC:")
                .append(attackComplexity == null ? "" : attackComplexity.value().charAt(0)).append("/AT:")
                .append(attackRequirements == null ? "" : attackRequirements.value().charAt(0)).append("/PR:")
                .append(privilegesRequired == null ? "" : privilegesRequired.value().charAt(0)).append("/UI:")
                .append(userInteraction == null ? "" : userInteraction.value().charAt(0)).append("/VC:")
                .append(vulnConfidentialityImpact == null ? "" : vulnConfidentialityImpact.value().charAt(0))
                .append("/VI:").append(vulnIntegrityImpact == null ? "" : vulnIntegrityImpact.value().charAt(0))
                .append("/VA:").append(vulnAvailabilityImpact == null ? "" : vulnAvailabilityImpact.value().charAt(0))
                .append("/SC:")
                .append(subConfidentialityImpact == null ? "" : subConfidentialityImpact.value().charAt(0))
                .append("/SI:").append(subIntegrityImpact == null ? "" : subIntegrityImpact.value().charAt(0))
                .append("/SA:").append(subAvailabilityImpact == null ? "" : subAvailabilityImpact.value().charAt(0));
        // (\/E:[XAPU])?(\/CR:[XHML])?(\/IR:[XHML])?(\/AR:[XHML])?
        if (exploitMaturity != null) {
            v.append("/E:").append(
                    exploitMaturity == ExploitMaturityType.NOT_DEFINED ? "X" : exploitMaturity.value().charAt(0));
        }
        if (confidentialityRequirement != null) {
            v.append("/CR:").append(confidentialityRequirement == CiaRequirementType.NOT_DEFINED ? "X"
                    : confidentialityRequirement.value().charAt(0));
        }
        if (integrityRequirement != null) {
            v.append("/IR:").append(integrityRequirement == CiaRequirementType.NOT_DEFINED ? "X"
                    : integrityRequirement.value().charAt(0));
        }
        if (availabilityRequirement != null) {
            v.append("/AR:").append(availabilityRequirement == CiaRequirementType.NOT_DEFINED ? "X"
                    : availabilityRequirement.value().charAt(0));
        }
        // (\/MAV:[XNALP])?(\/MAC:[XLH])?(\/MAT:[XNP])?(\/MPR:[XNLH])?(\/MUI:[XNPA])?(\/MVC:[XNLH])?(\/MVI:[XNLH])?(\/MVA:[XNLH])?
        if (modifiedAttackVector != null) {
            v.append("/MAV:").append(modifiedAttackVector == ModifiedAttackVectorType.NOT_DEFINED ? "X"
                    : modifiedAttackVector.value().charAt(0));
        }
        if (modifiedAttackComplexity != null) {
            v.append("/MAC:").append(modifiedAttackComplexity == ModifiedAttackComplexityType.NOT_DEFINED ? "X"
                    : modifiedAttackComplexity.value().charAt(0));
        }
        if (modifiedAttackRequirements != null) {
            v.append("/MAT:").append(modifiedAttackRequirements == ModifiedAttackRequirementsType.NOT_DEFINED ? "X"
                    : modifiedAttackRequirements.value().charAt(0));
        }
        if (modifiedPrivilegesRequired != null) {
            v.append("/MPR:").append(modifiedPrivilegesRequired == ModifiedPrivilegesRequiredType.NOT_DEFINED ? "X"
                    : modifiedPrivilegesRequired.value().charAt(0));
        }
        if (modifiedUserInteraction != null) {
            v.append("/MUI:").append(modifiedUserInteraction == ModifiedUserInteractionType.NOT_DEFINED ? "X"
                    : modifiedUserInteraction.value().charAt(0));
        }
        if (modifiedVulnConfidentialityImpact != null) {
            v.append("/MVC:").append(modifiedVulnConfidentialityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedVulnConfidentialityImpact.value().charAt(0));
        }
        if (modifiedVulnIntegrityImpact != null) {
            v.append("/MVI:").append(modifiedVulnIntegrityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedVulnIntegrityImpact.value().charAt(0));
        }
        if (modifiedVulnAvailabilityImpact != null) {
            v.append("/MVA:").append(modifiedVulnAvailabilityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedVulnAvailabilityImpact.value().charAt(0));
        }
        // (\/MSC:[XNLH])?(\/MSI:[XNLHS])?(\/MSA:[XNLHS])?(\/S:[XNP])?(\/AU:[XNY])?(\/R:[XAUI])?
        if (modifiedSubConfidentialityImpact != null) {
            v.append("/MSC:").append(modifiedSubConfidentialityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedSubConfidentialityImpact.value().charAt(0));
        }
        if (modifiedSubIntegrityImpact != null) {
            v.append("/MSI:").append(modifiedSubIntegrityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedSubIntegrityImpact.value().charAt(0));
        }
        if (modifiedSubAvailabilityImpact != null) {
            v.append("/MSA:").append(modifiedSubAvailabilityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedSubAvailabilityImpact.value().charAt(0));
        }
        if (safety != null) {
            v.append("/S:").append(safety == SafetyType.NOT_DEFINED ? "X" : safety.value().charAt(0));
        }
        if (automatable != null) {
            v.append("/AU:").append(automatable == AutomatableType.NOT_DEFINED ? "X" : automatable.value().charAt(0));
        }
        if (recovery != null) {
            v.append("/R:").append(recovery.value().charAt(0));
        }
        // (\/V:[XDC])?(\/RE:[XLMH])?(\/U:(X|Clear|Green|Amber|Red))?$"
        if (valueDensity != null) {
            v.append("/V:").append(valueDensity == ValueDensityType.NOT_DEFINED ? "X" : valueDensity.value().charAt(0));
        }
        if (vulnerabilityResponseEffort != null) {
            v.append("/RE:").append(vulnerabilityResponseEffort == VulnerabilityResponseEffortType.NOT_DEFINED ? "X"
                    : vulnerabilityResponseEffort.value().charAt(0));
        }
        if (providerUrgency != null) {
            v.append("/U:");
            if (providerUrgency == ProviderUrgencyType.NOT_DEFINED) {
                v.append("X");
            } else {
                v.append(providerUrgency.value().charAt(0)).append(providerUrgency.value().substring(1).toLowerCase());
            }
        }
        return v.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (!(o instanceof CvssV4Data))
            return false;
        CvssV4Data that = (CvssV4Data) o;
        return version == that.version && Objects.equals(vectorString, that.vectorString)
                && attackVector == that.attackVector && attackComplexity == that.attackComplexity
                && attackRequirements == that.attackRequirements && privilegesRequired == that.privilegesRequired
                && userInteraction == that.userInteraction
                && vulnConfidentialityImpact == that.vulnConfidentialityImpact
                && vulnIntegrityImpact == that.vulnIntegrityImpact
                && vulnAvailabilityImpact == that.vulnAvailabilityImpact
                && subConfidentialityImpact == that.subConfidentialityImpact
                && subIntegrityImpact == that.subIntegrityImpact && subAvailabilityImpact == that.subAvailabilityImpact
                && exploitMaturity == that.exploitMaturity
                && confidentialityRequirement == that.confidentialityRequirement
                && integrityRequirement == that.integrityRequirement
                && availabilityRequirement == that.availabilityRequirement
                && modifiedAttackVector == that.modifiedAttackVector
                && modifiedAttackComplexity == that.modifiedAttackComplexity
                && modifiedAttackRequirements == that.modifiedAttackRequirements
                && modifiedPrivilegesRequired == that.modifiedPrivilegesRequired
                && modifiedUserInteraction == that.modifiedUserInteraction
                && modifiedVulnConfidentialityImpact == that.modifiedVulnConfidentialityImpact
                && modifiedVulnIntegrityImpact == that.modifiedVulnIntegrityImpact
                && modifiedVulnAvailabilityImpact == that.modifiedVulnAvailabilityImpact
                && modifiedSubConfidentialityImpact == that.modifiedSubConfidentialityImpact
                && modifiedSubIntegrityImpact == that.modifiedSubIntegrityImpact
                && modifiedSubAvailabilityImpact == that.modifiedSubAvailabilityImpact && safety == that.safety
                && automatable == that.automatable && recovery == that.recovery && valueDensity == that.valueDensity
                && vulnerabilityResponseEffort == that.vulnerabilityResponseEffort
                && providerUrgency == that.providerUrgency && Objects.equals(baseScore, that.baseScore)
                && baseSeverity == that.baseSeverity && Objects.equals(threatScore, that.threatScore)
                && threatSeverity == that.threatSeverity && Objects.equals(environmentalScore, that.environmentalScore)
                && environmentalSeverity == that.environmentalSeverity;
    }

    @Override
    public int hashCode() {
        return Objects.hash(version, vectorString, attackVector, attackComplexity, attackRequirements,
                privilegesRequired, userInteraction, vulnConfidentialityImpact, vulnIntegrityImpact,
                vulnAvailabilityImpact, subConfidentialityImpact, subIntegrityImpact, subAvailabilityImpact,
                exploitMaturity, confidentialityRequirement, integrityRequirement, availabilityRequirement,
                modifiedAttackVector, modifiedAttackComplexity, modifiedAttackRequirements, modifiedPrivilegesRequired,
                modifiedUserInteraction, modifiedVulnConfidentialityImpact, modifiedVulnIntegrityImpact,
                modifiedVulnAvailabilityImpact, modifiedSubConfidentialityImpact, modifiedSubIntegrityImpact,
                modifiedSubAvailabilityImpact, safety, automatable, recovery, valueDensity, vulnerabilityResponseEffort,
                providerUrgency, baseScore, baseSeverity, threatScore, threatSeverity, environmentalScore,
                environmentalSeverity);
    }

    public enum AttackComplexityType {

        HIGH("HIGH"), LOW("LOW");

        private final static Map<String, AttackComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (AttackComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackComplexityType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static AttackComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (AttackComplexityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AttackComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum AttackVectorType {

        NETWORK("NETWORK"), ADJACENT("ADJACENT"), LOCAL("LOCAL"), PHYSICAL("PHYSICAL");

        private final static Map<String, AttackVectorType> CONSTANTS = new HashMap<>();

        static {
            for (AttackVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackVectorType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static AttackVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (AttackVectorType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AttackVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum CiaRequirementType {

        LOW("LOW"), MEDIUM("MEDIUM"), HIGH("HIGH"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, CiaRequirementType> CONSTANTS = new HashMap<>();

        static {
            for (CiaRequirementType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaRequirementType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static CiaRequirementType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (CiaRequirementType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            CiaRequirementType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum CiaType {

        NONE("NONE"), LOW("LOW"), HIGH("HIGH");

        private final static Map<String, CiaType> CONSTANTS = new HashMap<>();

        static {
            for (CiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static CiaType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (CiaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            CiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum ExploitMaturityType {

        UNREPORTED("UNREPORTED"), PROOF_OF_CONCEPT("PROOF_OF_CONCEPT"), ATTACKED("ATTACKED"), NOT_DEFINED(
                "NOT_DEFINED");

        private final static Map<String, ExploitMaturityType> CONSTANTS = new HashMap<>();

        static {
            for (ExploitMaturityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ExploitMaturityType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ExploitMaturityType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ExploitMaturityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ExploitMaturityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum ModifiedAttackComplexityType {

        HIGH("HIGH"), LOW("LOW"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedAttackComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackComplexityType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedAttackComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackComplexityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedAttackComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum AttackRequirementsType {

        NONE("NONE"), PRESENT("PRESENT");

        private final static Map<String, AttackRequirementsType> CONSTANTS = new HashMap<>();

        static {
            for (AttackRequirementsType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackRequirementsType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static AttackRequirementsType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (AttackRequirementsType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AttackRequirementsType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum ModifiedAttackRequirementsType {

        NONE("NONE"), PRESENT("PRESENT"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedAttackRequirementsType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackRequirementsType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackRequirementsType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedAttackRequirementsType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackRequirementsType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedAttackRequirementsType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum ModifiedAttackVectorType {

        NETWORK("NETWORK"), ADJACENT("ADJACENT"), LOCAL("LOCAL"), PHYSICAL("PHYSICAL"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedAttackVectorType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackVectorType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedAttackVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackVectorType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedAttackVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum ModifiedCiaType {

        NONE("NONE"), LOW("LOW"), HIGH("HIGH"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedCiaType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedCiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedCiaType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedCiaType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedCiaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedCiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum SafetyType {

        NEGLIGIBLE("NEGLIGIBLE"), PRESENT("PRESENT"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, SafetyType> CONSTANTS = new HashMap<>();

        static {
            for (SafetyType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        SafetyType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static SafetyType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (SafetyType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            SafetyType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum AutomatableType {

        NO("NO"), YES("YES"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, AutomatableType> CONSTANTS = new HashMap<>();

        static {
            for (AutomatableType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AutomatableType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static AutomatableType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (AutomatableType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AutomatableType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum RecoveryType {

        AUTOMATIC("AUTOMATIC"), USER("USER"), IRRECOVERABLE("IRRECOVERABLE"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, RecoveryType> CONSTANTS = new HashMap<>();

        static {
            for (RecoveryType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        RecoveryType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static RecoveryType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (RecoveryType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            RecoveryType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum ValueDensityType {

        DIFFUSE("DIFFUSE"), CONCENTRATED("CONCENTRATED"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ValueDensityType> CONSTANTS = new HashMap<>();

        static {
            for (ValueDensityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ValueDensityType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ValueDensityType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ValueDensityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ValueDensityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum VulnerabilityResponseEffortType {

        LOW("LOW"), MODERATE("MODERATE"), HIGH("HIGH"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, VulnerabilityResponseEffortType> CONSTANTS = new HashMap<>();

        static {
            for (VulnerabilityResponseEffortType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        VulnerabilityResponseEffortType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static VulnerabilityResponseEffortType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (VulnerabilityResponseEffortType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            VulnerabilityResponseEffortType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum ProviderUrgencyType {

        CLEAR("CLEAR"), GREEN("GREEN"), AMBER("AMBER"), RED("RED"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ProviderUrgencyType> CONSTANTS = new HashMap<>();

        static {
            for (ProviderUrgencyType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ProviderUrgencyType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ProviderUrgencyType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ProviderUrgencyType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ProviderUrgencyType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }
    }

    public enum ModifiedPrivilegesRequiredType {

        HIGH("HIGH"), LOW("LOW"), NONE("NONE"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedPrivilegesRequiredType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedPrivilegesRequiredType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedPrivilegesRequiredType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedPrivilegesRequiredType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedPrivilegesRequiredType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedPrivilegesRequiredType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum ModifiedVulnCiaType {

        NONE("NONE"), LOW("LOW"), HIGH("HIGH"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedVulnCiaType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedVulnCiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedVulnCiaType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedVulnCiaType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedVulnCiaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedVulnCiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum ModifiedUserInteractionType {

        NONE("NONE"), PASSIVE("PASSIVE"), ACTIVE("ACTIVE"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, ModifiedUserInteractionType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedUserInteractionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedUserInteractionType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static ModifiedUserInteractionType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedUserInteractionType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedUserInteractionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum PrivilegesRequiredType {

        HIGH("HIGH"), LOW("LOW"), NONE("NONE");

        private final static Map<String, PrivilegesRequiredType> CONSTANTS = new HashMap<>();

        static {
            for (PrivilegesRequiredType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        PrivilegesRequiredType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static PrivilegesRequiredType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (PrivilegesRequiredType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            PrivilegesRequiredType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum RemediationLevelType {

        OFFICIAL_FIX("OFFICIAL_FIX"), TEMPORARY_FIX("TEMPORARY_FIX"), WORKAROUND("WORKAROUND"), UNAVAILABLE(
                "UNAVAILABLE"), NOT_DEFINED("NOT_DEFINED");

        private final static Map<String, RemediationLevelType> CONSTANTS = new HashMap<>();

        static {
            for (RemediationLevelType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        RemediationLevelType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static RemediationLevelType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (RemediationLevelType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            RemediationLevelType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum SeverityType {

        NONE("NONE"), LOW("LOW"), MEDIUM("MEDIUM"), HIGH("HIGH"), CRITICAL("CRITICAL");

        private final static Map<String, SeverityType> CONSTANTS = new HashMap<>();

        static {
            for (SeverityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        SeverityType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static SeverityType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (SeverityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            SeverityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    public enum UserInteractionType {

        NONE("NONE"), PASSIVE("PASSIVE"), ACTIVE("ACTIVE");

        private final static Map<String, UserInteractionType> CONSTANTS = new HashMap<>();

        static {
            for (UserInteractionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        UserInteractionType(String value) {
            this.value = value;
        }

        @JsonCreator
        public static UserInteractionType fromValue(String value) {
            // allow conversion from vector string
            if (value != null && value.length() == 1) {
                for (UserInteractionType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            UserInteractionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS Version
     */
    public enum Version {

        _4_0("4.0");

        private final static Map<String, Version> CONSTANTS = new HashMap<>();

        static {
            for (Version c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        Version(String value) {
            this.value = value;
        }

        @JsonCreator
        public static Version fromValue(String value) {
            Version constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        @JsonValue
        public String value() {
            return this.value;
        }

    }

}
