"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var PipelineGraphModel_1 = require("./PipelineGraphModel");
exports.sequentialStagesLabelOffset = 70;
/**
 * Main process for laying out the graph. Creates and positions markers for each component, but creates no components.
 *
 *  1. Creates nodes for each stage in the pipeline
 *  2. Position the nodes in columns for each top stage, and in rows within each column based on execution order
 *  3. Create all the connections between nodes that need to be rendered
 *  4. Create a bigLabel per column, and a smallLabel for any child nodes
 *  5. Measure the extents of the graph
 *
 *  @deprecated Don't use this, the latest version exists in the ux-widgets repo, and Blue Ocean also uses that now
 */
function layoutGraph(newStages, layout) {
    var stageNodeColumns = createNodeColumns(newStages);
    var nodeSpacingH = layout.nodeSpacingH, ypStart = layout.ypStart;
    var startNode = {
        x: 0,
        y: 0,
        name: 'Start',
        id: -1,
        isPlaceholder: true,
        key: 'start-node',
        type: 'start',
    };
    var endNode = {
        x: 0,
        y: 0,
        name: 'End',
        id: -2,
        isPlaceholder: true,
        key: 'end-node',
        type: 'end',
    };
    var allNodeColumns = [
        { rows: [[startNode]], x: 0 }
    ].concat(stageNodeColumns, [
        { rows: [[endNode]], x: 0 },
    ]);
    positionNodes(allNodeColumns, layout);
    var bigLabels = createBigLabels(allNodeColumns);
    var smallLabels = createSmallLabels(allNodeColumns);
    var connections = createConnections(allNodeColumns);
    // Calculate the size of the graph
    var measuredWidth = 0;
    var measuredHeight = 200;
    for (var _i = 0, allNodeColumns_1 = allNodeColumns; _i < allNodeColumns_1.length; _i++) {
        var column = allNodeColumns_1[_i];
        for (var _a = 0, _b = column.rows; _a < _b.length; _a++) {
            var row = _b[_a];
            for (var _c = 0, row_1 = row; _c < row_1.length; _c++) {
                var node = row_1[_c];
                measuredWidth = Math.max(measuredWidth, node.x + nodeSpacingH / 2);
                measuredHeight = Math.max(measuredHeight, node.y + ypStart);
            }
        }
    }
    return {
        nodeColumns: allNodeColumns,
        connections: connections,
        bigLabels: bigLabels,
        smallLabels: smallLabels,
        measuredWidth: measuredWidth,
        measuredHeight: measuredHeight,
    };
}
exports.layoutGraph = layoutGraph;
/**
 * Generate an array of columns, based on the top-level stages
 */
function createNodeColumns(topLevelStages) {
    if (topLevelStages === void 0) { topLevelStages = []; }
    var nodeColumns = [];
    var _loop_1 = function (topStage) {
        // If stage has children, we don't draw a node for it, just its children
        var stagesForColumn = topStage.children && topStage.children.length ? topStage.children : [topStage];
        var column = {
            topStage: topStage,
            rows: [],
            x: 0,
        };
        for (var _i = 0, stagesForColumn_1 = stagesForColumn; _i < stagesForColumn_1.length; _i++) {
            var firstStageForRow = stagesForColumn_1[_i];
            if (firstStageForRow) {
                // Skip incomplete data generated by server response limits
                var rowNodes = [];
                var nodeStage = firstStageForRow;
                while (nodeStage) {
                    rowNodes.push({
                        x: 0,
                        y: 0,
                        name: nodeStage.name,
                        id: nodeStage.id,
                        stage: nodeStage,
                        isPlaceholder: false,
                        key: 'n_' + nodeStage.id,
                    });
                    nodeStage = nodeStage.nextSibling;
                }
                column.rows.push(rowNodes);
            }
        }
        if (PipelineGraphModel_1.MATRIOSKA_PATHS) {
            // Sort by row length for visual appeal when connectors don't all line up vertically
            var originalOrder_1 = column.rows.concat();
            column.rows.sort(function (left, right) {
                if (left.length < right.length) {
                    return -1;
                }
                if (left.length > right.length) {
                    return 1;
                }
                return originalOrder_1.indexOf(left) < originalOrder_1.indexOf(right) ? -1 : 1; // For stability
            });
        }
        nodeColumns.push(column);
    };
    for (var _i = 0, topLevelStages_1 = topLevelStages; _i < topLevelStages_1.length; _i++) {
        var topStage = topLevelStages_1[_i];
        _loop_1(topStage);
    }
    return nodeColumns;
}
/**
 * Walks the columns of nodes giving them x and y positions. Mutates the node objects in place for now.
 */
function positionNodes(nodeColumns, _a) {
    var nodeSpacingH = _a.nodeSpacingH, parallelSpacingH = _a.parallelSpacingH, nodeSpacingV = _a.nodeSpacingV, ypStart = _a.ypStart;
    var xp = nodeSpacingH / 2;
    var previousTopNode;
    for (var _i = 0, nodeColumns_1 = nodeColumns; _i < nodeColumns_1.length; _i++) {
        var column = nodeColumns_1[_i];
        var topNode = column.rows[0][0];
        var yp = ypStart; // Reset Y to top for each column
        if (previousTopNode) {
            // Advance X position
            if (previousTopNode.isPlaceholder || topNode.isPlaceholder) {
                // Don't space placeholder nodes (start/end) as wide as normal.
                xp += Math.floor(nodeSpacingH * 0.7);
            }
            else {
                xp += nodeSpacingH;
            }
        }
        var widestRow = 0;
        for (var _b = 0, _c = column.rows; _b < _c.length; _b++) {
            var row = _c[_b];
            widestRow = Math.max(widestRow, row.length);
        }
        var xpStart = xp; // Left-most position in this column
        var maxX = xp;
        for (var _d = 0, _e = column.rows; _d < _e.length; _d++) {
            var row = _e[_d];
            // Offset the beginning of narrower rows towards column center
            xp += Math.round((widestRow - row.length) * parallelSpacingH * 0.5);
            for (var _f = 0, row_2 = row; _f < row_2.length; _f++) {
                var node = row_2[_f];
                if (!node.isPlaceholder && node.stage && node.stage.seqContainerName) {
                    xp += exports.sequentialStagesLabelOffset;
                }
                maxX = Math.max(maxX, xp);
                node.x = xp;
                node.y = yp;
                xp += parallelSpacingH; // Space out nodes in each row
            }
            xp = xpStart; // CR
            yp += nodeSpacingV; // LF
        }
        column.x = Math.round((xpStart + maxX) / 2); // Center of column
        xp = maxX; // Make sure we're at the end of the widest row for this column before next loop
        previousTopNode = topNode;
    }
}
/**
 * Generate label descriptions for big labels at the top of each column
 */
function createBigLabels(columns) {
    var labels = [];
    for (var _i = 0, columns_1 = columns; _i < columns_1.length; _i++) {
        var column = columns_1[_i];
        var node = column.rows[0][0];
        var stage = column.topStage;
        var text = stage ? stage.name : node.name;
        var key = 'l_b_' + node.key;
        labels.push({
            x: column.x,
            y: node.y,
            node: node,
            stage: stage,
            text: text,
            key: key,
        });
    }
    return labels;
}
/**
 * Generate label descriptions for small labels under the nodes
 */
function createSmallLabels(columns) {
    var labels = [];
    for (var _i = 0, columns_2 = columns; _i < columns_2.length; _i++) {
        var column = columns_2[_i];
        for (var _a = 0, _b = column.rows; _a < _b.length; _a++) {
            var row = _b[_a];
            for (var _c = 0, row_3 = row; _c < row_3.length; _c++) {
                var node = row_3[_c];
                // We add small labels to parallel nodes only so skip others
                if (node.isPlaceholder || !node.stage || (node.stage.type !== 'PARALLEL' && node.stage.isSequential !== true)) {
                    continue;
                }
                var label = {
                    x: node.x,
                    y: node.y,
                    text: node.name,
                    key: 'l_s_' + node.key,
                    node: node,
                };
                if (node.isPlaceholder === false) {
                    label.stage = node.stage;
                }
                labels.push(label);
            }
        }
    }
    return labels;
}
/**
 * Generate connection information from column to column
 */
function createConnections(columns) {
    var connections = [];
    var sourceNodes = [];
    var skippedNodes = [];
    for (var _i = 0, columns_3 = columns; _i < columns_3.length; _i++) {
        var column = columns_3[_i];
        if (column.topStage && column.topStage.state === 'skipped') {
            skippedNodes.push(column.rows[0][0]);
            continue;
        }
        // Connections to each row in this column
        if (sourceNodes.length) {
            connections.push({
                sourceNodes: sourceNodes,
                destinationNodes: column.rows.map(function (row) { return row[0]; }),
                skippedNodes: skippedNodes,
            });
        }
        // Connections between nodes within each row
        for (var _a = 0, _b = column.rows; _a < _b.length; _a++) {
            var row = _b[_a];
            for (var i = 0; i < row.length - 1; i++) {
                connections.push({
                    sourceNodes: [row[i]],
                    destinationNodes: [row[i + 1]],
                    skippedNodes: [],
                });
            }
        }
        sourceNodes = column.rows.map(function (row) { return row[row.length - 1]; }); // Last node of each row
        skippedNodes = [];
    }
    return connections;
}
