/*
 * Copyright 2025, Redis Ltd. and Contributors
 * All rights reserved.
 *
 * Licensed under the MIT License.
 */
package io.lettuce.core.cluster.api.sync;

import java.util.Map;
import java.util.List;

import io.lettuce.core.annotations.Experimental;
import io.lettuce.core.search.AggregationReply;
import io.lettuce.core.search.SearchReply;
import io.lettuce.core.search.SpellCheckResult;
import io.lettuce.core.search.Suggestion;
import io.lettuce.core.search.arguments.AggregateArgs;
import io.lettuce.core.search.arguments.CreateArgs;
import io.lettuce.core.search.arguments.ExplainArgs;
import io.lettuce.core.search.arguments.FieldArgs;
import io.lettuce.core.search.arguments.SearchArgs;
import io.lettuce.core.search.arguments.SpellCheckArgs;
import io.lettuce.core.search.arguments.SugAddArgs;
import io.lettuce.core.search.arguments.SugGetArgs;
import io.lettuce.core.search.arguments.SynUpdateArgs;
import io.lettuce.core.search.AggregationReply.Cursor;

/**
 * Synchronous executed commands on a node selection for RediSearch functionality
 *
 * @param <K> Key type.
 * @param <V> Value type.
 * @author Tihomir Mateev
 * @see <a href="https://redis.io/docs/latest/operate/oss_and_stack/stack-with-enterprise/search/">RediSearch</a>
 * @since 6.8
 * @generated by io.lettuce.apigenerator.CreateSyncNodeSelectionClusterApi
 */
public interface NodeSelectionSearchCommands<K, V> {

    /**
     * Create a new search index with the given name and field definitions using default settings.
     *
     * <p>
     * This command creates a new search index that enables full-text search, filtering, and aggregation capabilities on Redis
     * data structures. The index will use default settings for data type (HASH), key prefixes (all keys), and other
     * configuration options.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(K) at creation where K is the number of fields, O(N) if scanning the keyspace is
     * triggered, where N is the number of keys in the keyspace
     * </p>
     *
     * @param index the index name
     * @param fieldArgs the {@link FieldArgs} list defining the searchable fields and their types
     * @return {@code "OK"} if the index was created successfully
     * @see <a href="https://redis.io/docs/latest/commands/ft.create/">FT.CREATE</a>
     * @see CreateArgs
     * @see FieldArgs
     * @see #ftCreate(String, CreateArgs, List)
     * @see #ftDropindex(String)
     */
    @Experimental
    Executions<String> ftCreate(String index, List<FieldArgs<K>> fieldArgs);

    /**
     * Create a new search index with the given name, custom configuration, and field definitions.
     *
     * <p>
     * This command creates a new search index with advanced configuration options that control how the index behaves, what data
     * it indexes, and how it processes documents. This variant provides full control over index creation parameters.
     * </p>
     *
     * <p>
     * The {@link CreateArgs} parameter allows you to specify:
     * </p>
     * <ul>
     * <li><strong>Data type:</strong> HASH (default) or JSON documents</li>
     * <li><strong>Key prefixes:</strong> Which keys to index based on prefix patterns</li>
     * <li><strong>Filters:</strong> Conditional indexing based on field values</li>
     * <li><strong>Language settings:</strong> Default language and language field for stemming</li>
     * <li><strong>Performance options:</strong> NOOFFSETS, NOHL, NOFIELDS, NOFREQS for memory optimization</li>
     * <li><strong>Temporary indexes:</strong> Auto-expiring indexes for short-term use</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(K) at creation where K is the number of fields, O(N) if scanning the keyspace is
     * triggered, where N is the number of keys in the keyspace
     * </p>
     *
     * @param index the index name
     * @param arguments the index {@link CreateArgs} containing configuration options
     * @param fieldArgs the {@link FieldArgs} list defining the searchable fields and their types
     * @return {@code "OK"} if the index was created successfully
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.create/">FT.CREATE</a>
     * @see CreateArgs
     * @see FieldArgs
     * @see #ftCreate(String, List)
     * @see #ftDropindex(String)
     */
    @Experimental
    Executions<String> ftCreate(String index, CreateArgs<K, V> arguments, List<FieldArgs<K>> fieldArgs);

    /**
     * Add an alias to a search index.
     *
     * <p>
     * This command creates an alias that points to an existing search index, allowing applications to reference the index by an
     * alternative name. Aliases provide a level of indirection that enables transparent index management and migration
     * strategies.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Index abstraction:</strong> Applications can use stable alias names while underlying indexes change</li>
     * <li><strong>Blue-green deployments:</strong> Switch traffic between old and new indexes seamlessly</li>
     * <li><strong>A/B testing:</strong> Route different application instances to different indexes</li>
     * <li><strong>Maintenance windows:</strong> Redirect queries during index rebuilds or migrations</li>
     * </ul>
     *
     * <p>
     * <strong>Important notes:</strong>
     * </p>
     * <ul>
     * <li>An index can have multiple aliases, but an alias can only point to one index</li>
     * <li>Aliases cannot reference other aliases (no alias chaining)</li>
     * <li>If the alias already exists, this command will fail with an error</li>
     * <li>Use {@link #ftAliasupdate(String, String)} to reassign an existing alias</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param alias the alias name to create
     * @param index the target index name that the alias will point to
     * @return {@code "OK"} if the alias was successfully created
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.aliasadd/">FT.ALIASADD</a>
     * @see #ftAliasupdate(String, String)
     * @see #ftAliasdel(String)
     */
    @Experimental
    Executions<String> ftAliasadd(String alias, String index);

    /**
     * Update an existing alias to point to a different search index.
     *
     * <p>
     * This command updates an existing alias to point to a different index, or creates the alias if it doesn't exist. Unlike
     * {@link #ftAliasadd(String, String)}, this command will succeed even if the alias already exists, making it useful for
     * atomic alias updates during index migrations.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Atomic updates:</strong> Change alias target without downtime</li>
     * <li><strong>Index migration:</strong> Seamlessly switch from old to new index versions</li>
     * <li><strong>Rollback capability:</strong> Quickly revert to previous index if issues arise</li>
     * <li><strong>Blue-green deployments:</strong> Switch production traffic between index versions</li>
     * </ul>
     *
     * <p>
     * <strong>Important notes:</strong>
     * </p>
     * <ul>
     * <li>If the alias doesn't exist, it will be created (same as {@code ftAliasadd})</li>
     * <li>If the alias exists, it will be updated to point to the new index</li>
     * <li>The previous index association is removed automatically</li>
     * <li>This operation is atomic - no intermediate state where alias is undefined</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param alias the alias name to update or create
     * @param index the target index name that the alias will point to
     * @return {@code "OK"} if the alias was successfully updated
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.aliasupdate/">FT.ALIASUPDATE</a>
     * @see #ftAliasadd(String, String)
     * @see #ftAliasdel(String)
     */
    @Experimental
    Executions<String> ftAliasupdate(String alias, String index);

    /**
     * Remove an alias from a search index.
     *
     * <p>
     * This command removes an existing alias, breaking the association between the alias name and its target index. The
     * underlying index remains unchanged and accessible by its original name.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Cleanup:</strong> Remove unused or obsolete aliases</li>
     * <li><strong>Security:</strong> Revoke access to indexes through specific alias names</li>
     * <li><strong>Maintenance:</strong> Temporarily disable access during maintenance windows</li>
     * <li><strong>Resource management:</strong> Clean up aliases before index deletion</li>
     * </ul>
     *
     * <p>
     * <strong>Important notes:</strong>
     * </p>
     * <ul>
     * <li>Only the alias is removed - the target index is not affected</li>
     * <li>If the alias doesn't exist, this command will fail with an error</li>
     * <li>Applications using the alias will receive errors after deletion</li>
     * <li>Consider using {@link #ftAliasupdate(String, String)} to redirect before deletion</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param alias the alias name to remove
     * @return {@code "OK"} if the alias was successfully removed
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.aliasdel/">FT.ALIASDEL</a>
     * @see #ftAliasadd(String, String)
     * @see #ftAliasupdate(String, String)
     */
    @Experimental
    Executions<String> ftAliasdel(String alias);

    /**
     * Add new attributes to an existing search index.
     *
     * <p>
     * This command allows you to extend an existing search index by adding new searchable fields without recreating the entire
     * index. The new attributes will be applied to future document updates and can optionally be applied to existing documents
     * through reindexing.
     * </p>
     *
     * <p>
     * Key features and considerations:
     * </p>
     * <ul>
     * <li><strong>Non-destructive:</strong> Existing index structure and data remain intact</li>
     * <li><strong>Incremental indexing:</strong> New fields are indexed as documents are updated</li>
     * <li><strong>Reindexing control:</strong> Option to skip initial scan for performance</li>
     * <li><strong>Field limitations:</strong> Text field limits may apply based on index creation options</li>
     * </ul>
     *
     * <p>
     * <strong>Important notes:</strong>
     * </p>
     * <ul>
     * <li>If the index was created without {@code MAXTEXTFIELDS}, you may be limited to 32 total text attributes</li>
     * <li>New attributes are only indexed for documents that are updated after the ALTER command</li>
     * <li>Use {@code SKIPINITIALSCAN} to avoid scanning existing documents if immediate indexing is not required</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(N) where N is the number of keys in the keyspace if initial scan is performed, O(1)
     * if {@code SKIPINITIALSCAN} is used
     * </p>
     *
     * @param index the index name, as a key
     * @param skipInitialScan if {@code true}, skip scanning and indexing existing documents; if {@code false}, scan and index
     *        existing documents with the new attributes
     * @param fieldArgs the {@link FieldArgs} list defining the new searchable fields and their types to add
     * @return {@code "OK"} if the index was successfully altered
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.alter/">FT.ALTER</a>
     * @see FieldArgs
     * @see #ftCreate(String, List)
     * @see #ftCreate(String, CreateArgs, List)
     */
    @Experimental
    Executions<String> ftAlter(String index, boolean skipInitialScan, List<FieldArgs<K>> fieldArgs);

    /**
     * Add new attributes to an existing search index.
     *
     * <p>
     * This command allows you to extend an existing search index by adding new searchable fields without recreating the entire
     * index. The new attributes will be applied to future document updates and can optionally be applied to existing documents
     * through reindexing.
     * </p>
     *
     * <p>
     * Key features and considerations:
     * </p>
     * <ul>
     * <li><strong>Non-destructive:</strong> Existing index structure and data remain intact</li>
     * <li><strong>Incremental indexing:</strong> New fields are indexed as documents are updated</li>
     * <li><strong>Reindexing control:</strong> Option to skip initial scan for performance</li>
     * <li><strong>Field limitations:</strong> Text field limits may apply based on index creation options</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(N) where N is the number of keys in the keyspace if initial scan is performed
     * </p>
     *
     * @param index the index name, as a key
     * @param fieldArgs the {@link FieldArgs} list defining the new searchable fields and their types to add
     * @return {@code "OK"} if the index was successfully altered
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.alter/">FT.ALTER</a>
     * @see FieldArgs
     * @see #ftCreate(String, List)
     * @see #ftCreate(String, CreateArgs, List)
     */
    @Experimental
    Executions<String> ftAlter(String index, List<FieldArgs<K>> fieldArgs);

    /**
     * Return a distinct set of values indexed in a Tag field.
     *
     * <p>
     * This command retrieves all unique values that have been indexed in a specific Tag field within a search index. It's
     * particularly useful for discovering the range of values available in categorical fields such as cities, categories,
     * status values, or any other enumerated data.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Data exploration:</strong> Discover all possible values in a tag field</li>
     * <li><strong>Filter building:</strong> Populate dropdown lists or filter options in applications</li>
     * <li><strong>Data validation:</strong> Verify expected values are present in the index</li>
     * <li><strong>Analytics:</strong> Understand the distribution of categorical data</li>
     * </ul>
     *
     * <p>
     * <strong>Important limitations:</strong>
     * </p>
     * <ul>
     * <li>Only works with Tag fields defined in the index schema</li>
     * <li>No paging or sorting is provided - all values are returned at once</li>
     * <li>Tags are not alphabetically sorted in the response</li>
     * <li>Returned strings are lowercase with whitespaces removed</li>
     * <li>Performance scales with the number of unique values (O(N) complexity)</li>
     * </ul>
     *
     * <p>
     * <strong>Example usage scenarios:</strong>
     * </p>
     * <ul>
     * <li>Retrieving all available product categories for an e-commerce filter</li>
     * <li>Getting all city names indexed for location-based searches</li>
     * <li>Listing all status values (active, inactive, pending) for administrative interfaces</li>
     * <li>Discovering all tags or labels applied to content</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(N) where N is the number of distinct values in the tag field
     * </p>
     *
     * @param index the index name containing the tag field
     * @param fieldName the name of the Tag field defined in the index schema
     * @return a list of all distinct values indexed in the specified tag field. The list contains the raw tag values as they
     *         were indexed (lowercase, whitespace removed).
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.tagvals/">FT.TAGVALS</a>
     * @see #ftCreate(String, List)
     * @see #ftCreate(String, CreateArgs, List)
     */
    @Experimental
    Executions<List<V>> ftTagvals(String index, String fieldName);

    /**
     * Perform spelling correction on a query, returning suggestions for misspelled terms.
     *
     * <p>
     * This command analyzes the query for misspelled terms and provides spelling suggestions based on the indexed terms and
     * optionally custom dictionaries. A misspelled term is a full text term (word) that is:
     * </p>
     * <ul>
     * <li>Not a stop word</li>
     * <li>Not in the index</li>
     * <li>At least 3 characters long</li>
     * </ul>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Query correction:</strong> Improve search experience by suggesting corrections</li>
     * <li><strong>Typo handling:</strong> Handle common typing mistakes and misspellings</li>
     * <li><strong>Search enhancement:</strong> Increase search success rates</li>
     * <li><strong>User experience:</strong> Provide "did you mean" functionality</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index with the indexed terms
     * @param query the search query to check for spelling errors
     * @return spell check result containing misspelled terms and their suggestions
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.spellcheck/">FT.SPELLCHECK</a>
     * @see <a href="https://redis.io/docs/latest/develop/ai/search-and-query/advanced-concepts/spellcheck/">Spellchecking</a>
     * @see #ftSpellcheck(String, Object, SpellCheckArgs)
     * @see #ftDictadd(String, Object[])
     * @see #ftDictdel(String, Object[])
     * @see #ftDictdump(String)
     */
    @Experimental
    Executions<SpellCheckResult<V>> ftSpellcheck(String index, V query);

    /**
     * Perform spelling correction on a query with additional options.
     *
     * <p>
     * This command analyzes the query for misspelled terms and provides spelling suggestions with configurable options for
     * distance, custom dictionaries, and dialect.
     * </p>
     *
     * <p>
     * Available options:
     * </p>
     * <ul>
     * <li><strong>DISTANCE:</strong> Maximum Levenshtein distance for suggestions (default: 1, max: 4)</li>
     * <li><strong>TERMS INCLUDE:</strong> Include terms from custom dictionaries as suggestions</li>
     * <li><strong>TERMS EXCLUDE:</strong> Exclude terms from custom dictionaries from suggestions</li>
     * <li><strong>DIALECT:</strong> Specify dialect version for query execution</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index with the indexed terms
     * @param query the search query to check for spelling errors
     * @param args the spellcheck arguments (distance, terms, dialect)
     * @return spell check result containing misspelled terms and their suggestions
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.spellcheck/">FT.SPELLCHECK</a>
     * @see <a href="https://redis.io/docs/latest/develop/ai/search-and-query/advanced-concepts/spellcheck/">Spellchecking</a>
     * @see #ftSpellcheck(String, Object)
     * @see #ftDictadd(String, Object[])
     * @see #ftDictdel(String, Object[])
     * @see #ftDictdump(String)
     */
    @Experimental
    Executions<SpellCheckResult<V>> ftSpellcheck(String index, V query, SpellCheckArgs<K, V> args);

    /**
     * Add terms to a dictionary.
     *
     * <p>
     * This command adds one or more terms to a dictionary. Dictionaries are used for storing custom stopwords, synonyms, and
     * other term lists that can be referenced in search operations. The dictionary is created if it doesn't exist.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Stopwords:</strong> Create custom stopword lists for filtering</li>
     * <li><strong>Synonyms:</strong> Build synonym dictionaries for query expansion</li>
     * <li><strong>Custom terms:</strong> Store domain-specific terminology</li>
     * <li><strong>Blacklists:</strong> Maintain lists of prohibited terms</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param dict the dictionary name
     * @param terms the terms to add to the dictionary
     * @return the number of new terms that were added
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.dictadd/">FT.DICTADD</a>
     * @see <a href="https://redis.io/docs/latest/develop/ai/search-and-query/advanced-concepts/spellcheck/">Spellchecking</a>
     * @see #ftDictdel(String, Object[])
     * @see #ftDictdump(String)
     */
    @Experimental
    Executions<Long> ftDictadd(String dict, V... terms);

    /**
     * Delete terms from a dictionary.
     *
     * <p>
     * This command removes one or more terms from a dictionary. Only exact matches will be removed from the dictionary.
     * Non-existent terms are ignored.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param dict the dictionary name
     * @param terms the terms to delete from the dictionary
     * @return the number of terms that were deleted
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.dictdel/">FT.DICTDEL</a>
     * @see #ftDictadd(String, Object[])
     * @see #ftDictdump(String)
     */
    @Experimental
    Executions<Long> ftDictdel(String dict, V... terms);

    /**
     * Dump all terms in a dictionary.
     *
     * <p>
     * This command returns all terms stored in the specified dictionary. The terms are returned in no particular order.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(N), where N is the size of the dictionary
     * </p>
     *
     * @param dict the dictionary name
     * @return a list of all terms in the dictionary
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.dictdump/">FT.DICTDUMP</a>
     * @see #ftDictadd(String, Object[])
     * @see #ftDictdel(String, Object[])
     */
    @Experimental
    Executions<List<V>> ftDictdump(String dict);

    /**
     * Return the execution plan for a complex query.
     *
     * <p>
     * This command returns a string representing the execution plan that Redis Search will use to execute the given query. This
     * is useful for understanding how the query will be processed and for optimizing query performance.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Query optimization:</strong> Understand how queries are executed</li>
     * <li><strong>Performance analysis:</strong> Identify potential bottlenecks</li>
     * <li><strong>Debugging:</strong> Troubleshoot complex query behavior</li>
     * <li><strong>Learning:</strong> Understand Redis Search query processing</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @param query the search query to explain
     * @return the execution plan as a string
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.explain/">FT.EXPLAIN</a>
     * @see #ftExplain(String, Object, ExplainArgs)
     * @see #ftSearch(String, Object)
     */
    @Experimental
    Executions<String> ftExplain(String index, V query);

    /**
     * Return the execution plan for a complex query with additional options.
     *
     * <p>
     * This command returns a string representing the execution plan that Redis Search will use to execute the given query under
     * the specified dialect version.
     * </p>
     *
     * <p>
     * Available options:
     * </p>
     * <ul>
     * <li><strong>DIALECT:</strong> Specify dialect version for query execution</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @param query the search query to explain
     * @param args the explain arguments (dialect)
     * @return the execution plan as a string
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.explain/">FT.EXPLAIN</a>
     * @see #ftExplain(String, Object)
     * @see #ftSearch(String, Object)
     */
    @Experimental
    Executions<String> ftExplain(String index, V query, ExplainArgs<K, V> args);

    /**
     * Return a list of all existing indexes.
     *
     * <p>
     * This command returns an array with the names of all existing indexes in the database. This is useful for discovering
     * available indexes and managing index lifecycle.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Index discovery:</strong> Find all available search indexes</li>
     * <li><strong>Management:</strong> List indexes for administrative operations</li>
     * <li><strong>Monitoring:</strong> Track index creation and deletion</li>
     * <li><strong>Debugging:</strong> Verify index existence</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * <p>
     * <strong>Note:</strong> This is a temporary command (indicated by the underscore prefix). In the future, a SCAN-type
     * command will be added for use when a database contains a large number of indices.
     * </p>
     *
     * @return a list of index names
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft._list/">FT._LIST</a>
     * @see #ftCreate(String, CreateArgs, FieldArgs[])
     * @see #ftDropindex(String)
     */
    @Experimental
    Executions<List<V>> ftList();

    /**
     * Dump synonym group contents.
     *
     * <p>
     * This command returns the contents of a synonym group. Synonym groups are used to define terms that should be treated as
     * equivalent during search operations.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Synonym management:</strong> View current synonym definitions</li>
     * <li><strong>Query expansion:</strong> Understand how terms are expanded</li>
     * <li><strong>Debugging:</strong> Verify synonym group contents</li>
     * <li><strong>Administration:</strong> Audit synonym configurations</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @return a map where keys are synonym terms and values are lists of group IDs containing that synonym
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.syndump/">FT.SYNDUMP</a>
     * @see #ftSynupdate(String, Object, Object[])
     * @see #ftSynupdate(String, Object, SynUpdateArgs, Object[])
     */
    @Experimental
    Executions<Map<V, List<V>>> ftSyndump(String index);

    /**
     * Update a synonym group with additional terms.
     *
     * <p>
     * This command creates or updates a synonym group with the specified terms. All terms in a synonym group are treated as
     * equivalent during search operations. The command triggers a scan of all documents by default.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Synonym creation:</strong> Define equivalent terms for search</li>
     * <li><strong>Query expansion:</strong> Improve search recall with synonyms</li>
     * <li><strong>Language support:</strong> Handle different languages and dialects</li>
     * <li><strong>Domain terminology:</strong> Map technical terms to common language</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @param synonymGroupId the synonym group identifier
     * @param terms the terms to add to the synonym group
     * @return OK if executed correctly
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.synupdate/">FT.SYNUPDATE</a>
     * @see #ftSynupdate(String, Object, SynUpdateArgs, Object[])
     * @see #ftSyndump(String)
     */
    @Experimental
    Executions<String> ftSynupdate(String index, V synonymGroupId, V... terms);

    /**
     * Update a synonym group with additional terms and options.
     *
     * <p>
     * This command creates or updates a synonym group with the specified terms and options. The SKIPINITIALSCAN option can be
     * used to avoid scanning existing documents, affecting only documents indexed after the update.
     * </p>
     *
     * <p>
     * Available options:
     * </p>
     * <ul>
     * <li><strong>SKIPINITIALSCAN:</strong> Skip scanning existing documents</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @param synonymGroupId the synonym group identifier
     * @param args the synupdate arguments (skipInitialScan)
     * @param terms the terms to add to the synonym group
     * @return OK if executed correctly
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.synupdate/">FT.SYNUPDATE</a>
     * @see #ftSynupdate(String, Object, Object[])
     * @see #ftSyndump(String)
     */
    @Experimental
    Executions<String> ftSynupdate(String index, V synonymGroupId, SynUpdateArgs<K, V> args, V... terms);

    /**
     * Add a suggestion string to an auto-complete suggestion dictionary.
     *
     * <p>
     * This command adds a suggestion string to an auto-complete suggestion dictionary with a specified score. The auto-complete
     * suggestion dictionary is disconnected from the index definitions and leaves creating and updating suggestions
     * dictionaries to the user.
     * </p>
     *
     * <p>
     * Key features and use cases:
     * </p>
     * <ul>
     * <li><strong>Auto-completion:</strong> Build type-ahead search functionality</li>
     * <li><strong>Search suggestions:</strong> Provide query suggestions to users</li>
     * <li><strong>Fuzzy matching:</strong> Support approximate string matching</li>
     * <li><strong>Weighted results:</strong> Control suggestion ranking with scores</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param key the suggestion dictionary key
     * @param suggestion the suggestion string to index
     * @param score the floating point number of the suggestion string's weight
     * @return the current size of the suggestion dictionary after adding the suggestion
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.sugadd/">FT.SUGADD</a>
     * @see #ftSugadd(Object, Object, double, SugAddArgs)
     * @see #ftSugget(Object, Object)
     * @see #ftSugdel(Object, Object)
     * @see #ftSuglen(Object)
     */
    @Experimental
    Executions<Long> ftSugadd(K key, V suggestion, double score);

    /**
     * Add a suggestion string to an auto-complete suggestion dictionary with additional options.
     *
     * <p>
     * This command adds a suggestion string to an auto-complete suggestion dictionary with a specified score and optional
     * arguments for incremental updates and payload storage.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param key the suggestion dictionary key
     * @param suggestion the suggestion string to index
     * @param score the floating point number of the suggestion string's weight
     * @param args the suggestion add arguments (INCR, PAYLOAD)
     * @return the current size of the suggestion dictionary after adding the suggestion
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.sugadd/">FT.SUGADD</a>
     * @see #ftSugadd(Object, Object, double)
     * @see #ftSugget(Object, Object, SugGetArgs)
     * @see #ftSugdel(Object, Object)
     * @see #ftSuglen(Object)
     */
    @Experimental
    Executions<Long> ftSugadd(K key, V suggestion, double score, SugAddArgs<K, V> args);

    /**
     * Delete a string from a suggestion dictionary.
     *
     * <p>
     * This command removes a suggestion string from an auto-complete suggestion dictionary. Only the exact string match will be
     * removed from the dictionary.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param key the suggestion dictionary key
     * @param suggestion the suggestion string to delete
     * @return {@code true} if the string was found and deleted, {@code false} otherwise
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.sugdel/">FT.SUGDEL</a>
     * @see #ftSugadd(Object, Object, double)
     * @see #ftSugget(Object, Object)
     * @see #ftSuglen(Object)
     */
    @Experimental
    Executions<Boolean> ftSugdel(K key, V suggestion);

    /**
     * Get completion suggestions for a prefix.
     *
     * <p>
     * This command retrieves completion suggestions for a prefix from an auto-complete suggestion dictionary. By default, it
     * returns up to 5 suggestions that match the given prefix.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param key the suggestion dictionary key
     * @param prefix the prefix to complete on
     * @return a list of suggestions matching the prefix
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.sugget/">FT.SUGGET</a>
     * @see #ftSugget(Object, Object, SugGetArgs)
     * @see #ftSugadd(Object, Object, double)
     * @see #ftSugdel(Object, Object)
     * @see #ftSuglen(Object)
     */
    @Experimental
    Executions<List<Suggestion<V>>> ftSugget(K key, V prefix);

    /**
     * Get completion suggestions for a prefix with additional options.
     *
     * <p>
     * This command retrieves completion suggestions for a prefix from an auto-complete suggestion dictionary with optional
     * arguments for fuzzy matching, score inclusion, payload inclusion, and result limiting.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param key the suggestion dictionary key
     * @param prefix the prefix to complete on
     * @param args the suggestion get arguments (FUZZY, WITHSCORES, WITHPAYLOADS, MAX)
     * @return a list of suggestions matching the prefix, optionally with scores and payloads
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.sugget/">FT.SUGGET</a>
     * @see #ftSugget(Object, Object)
     * @see #ftSugadd(Object, Object, double, SugAddArgs)
     * @see #ftSugdel(Object, Object)
     * @see #ftSuglen(Object)
     */
    @Experimental
    Executions<List<Suggestion<V>>> ftSugget(K key, V prefix, SugGetArgs<K, V> args);

    /**
     * Get the size of an auto-complete suggestion dictionary.
     *
     * <p>
     * This command returns the current number of suggestions stored in the auto-complete suggestion dictionary.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param key the suggestion dictionary key
     * @return the current size of the suggestion dictionary
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.suglen/">FT.SUGLEN</a>
     * @see #ftSugadd(Object, Object, double)
     * @see #ftSugget(Object, Object)
     * @see #ftSugdel(Object, Object)
     */
    @Experimental
    Executions<Long> ftSuglen(K key);

    /**
     * Drop a search index without deleting the associated documents.
     *
     * <p>
     * This command removes the search index and all its associated metadata, but preserves the original documents (hashes or
     * JSON objects) that were indexed. This is the safe default behavior that allows you to recreate the index later without
     * losing data.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name, as a key
     * @return {@code "OK"} if the index was successfully dropped
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.dropindex/">FT.DROPINDEX</a>
     * @see #ftDropindex(String, boolean)
     * @see #ftCreate(String, List)
     */
    @Experimental
    Executions<String> ftDropindex(String index);

    /**
     * Drop a search index with optional document deletion.
     *
     * <p>
     * This command removes the search index and optionally deletes all associated documents. When {@code deleteDocuments} is
     * {@code true}, this operation becomes destructive and will permanently remove both the index and all indexed documents
     * from Redis.
     * </p>
     *
     * <p>
     * <strong>Asynchronous Behavior:</strong> If an index creation is still running ({@link #ftCreate(String, List)} is running
     * asynchronously), only the document hashes that have already been indexed are deleted. Documents that are queued for
     * indexing but not yet processed will remain in the database.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1) or O(N) if documents are deleted, where N is the number of keys in the keyspace
     * </p>
     *
     * @param index the index name, as a key
     * @param deleteDocuments if {@code true}, delete the indexed documents as well; if {@code false}, preserve documents
     * @return {@code "OK"} if the index was successfully dropped
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.dropindex/">FT.DROPINDEX</a>
     * @see #ftDropindex(String)
     * @see #ftCreate(String, List)
     */
    @Experimental
    Executions<String> ftDropindex(String index, boolean deleteDocuments);

    /**
     * Search the index with a textual query using default search options.
     *
     * <p>
     * This command performs a full-text search on the specified index using the provided query string. It returns matching
     * documents with their content and metadata. This is the basic search variant that uses default search behavior without
     * additional filtering, sorting, or result customization.
     * </p>
     *
     * <p>
     * The query follows RediSearch query syntax, supporting:
     * </p>
     * <ul>
     * <li><strong>Simple text search:</strong> {@code "hello world"} - searches for documents containing both terms</li>
     * <li><strong>Field-specific search:</strong> {@code "@title:redis"} - searches within specific fields</li>
     * <li><strong>Boolean operators:</strong> {@code "redis AND search"} or {@code "redis | search"}</li>
     * <li><strong>Phrase search:</strong> {@code "\"exact phrase\""} - searches for exact phrase matches</li>
     * <li><strong>Wildcard search:</strong> {@code "redi*"} - prefix matching</li>
     * <li><strong>Numeric ranges:</strong> {@code "@price:[100 200]"} - numeric field filtering</li>
     * <li><strong>Geographic search:</strong> {@code "@location:[lon lat radius unit]"} - geo-spatial queries</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(N) where N is the number of results in the result set
     * </p>
     *
     * @param index the index name, as a key
     * @param query the query string following RediSearch query syntax
     * @return the result of the search command containing matching documents, see {@link SearchReply}
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.search/">FT.SEARCH</a>
     * @see <a href="https://redis.io/docs/latest/develop/interact/search-and-query/query/">Query syntax</a>
     * @see SearchReply
     * @see SearchArgs
     * @see #ftSearch(String, Object, SearchArgs)
     */
    @Experimental
    Executions<SearchReply<K, V>> ftSearch(String index, V query);

    /**
     * Search the index with a textual query using advanced search options and filters.
     *
     * <p>
     * This command performs a full-text search on the specified index with advanced configuration options provided through
     * {@link SearchArgs}. This variant allows fine-grained control over search behavior, result formatting, filtering, sorting,
     * and pagination.
     * </p>
     *
     * <p>
     * The {@link SearchArgs} parameter enables you to specify:
     * </p>
     * <ul>
     * <li><strong>Result options:</strong> NOCONTENT, WITHSCORES, WITHPAYLOADS, WITHSORTKEYS</li>
     * <li><strong>Query behavior:</strong> VERBATIM (no stemming), NOSTOPWORDS</li>
     * <li><strong>Filtering:</strong> Numeric filters, geo filters, field filters</li>
     * <li><strong>Result customization:</strong> RETURN specific fields, SUMMARIZE, HIGHLIGHT</li>
     * <li><strong>Sorting and pagination:</strong> SORTBY, LIMIT offset and count</li>
     * <li><strong>Performance options:</strong> TIMEOUT, SLOP, INORDER</li>
     * <li><strong>Language and scoring:</strong> LANGUAGE, SCORER, EXPLAINSCORE</li>
     * </ul>
     *
     * <h3>Performance Considerations:</h3>
     * <ul>
     * <li>Use NOCONTENT when you only need document IDs</li>
     * <li>Specify RETURN fields to limit data transfer</li>
     * <li>Use SORTABLE fields for efficient sorting</li>
     * <li>Apply filters to reduce result set size</li>
     * <li>Use LIMIT for pagination to avoid large result sets</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(N) where N is the number of results in the result set. Complexity varies based on
     * query type, filters, and sorting requirements.
     * </p>
     *
     * @param index the index name, as a key
     * @param query the query string following RediSearch query syntax
     * @param args the search arguments containing advanced options and filters
     * @return the result of the search command containing matching documents and metadata, see {@link SearchReply}
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.search/">FT.SEARCH</a>
     * @see <a href="https://redis.io/docs/latest/develop/interact/search-and-query/query/">Query syntax</a>
     * @see <a href="https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/">Advanced concepts</a>
     * @see SearchReply
     * @see SearchArgs
     * @see #ftSearch(String, Object)
     */
    @Experimental
    Executions<SearchReply<K, V>> ftSearch(String index, V query, SearchArgs<K, V> args);

    /**
     * Run a search query on an index and perform basic aggregate transformations using default options.
     *
     * <p>
     * This command executes a search query and applies aggregation operations to transform and analyze the results. Unlike
     * {@link #ftSearch(String, Object)}, which returns individual documents, FT.AGGREGATE processes the result set through a
     * pipeline of transformations to produce analytical insights, summaries, and computed values.
     * </p>
     *
     * <p>
     * This basic variant uses default aggregation behavior without additional pipeline operations. For advanced aggregations
     * with grouping, sorting, filtering, and custom transformations, use {@link #ftAggregate(String, Object, AggregateArgs)}.
     * </p>
     *
     * <p>
     * Common use cases for aggregations include:
     * </p>
     * <ul>
     * <li><strong>Analytics:</strong> Count documents, calculate averages, find min/max values</li>
     * <li><strong>Reporting:</strong> Group data by categories, time periods, or geographic regions</li>
     * <li><strong>Data transformation:</strong> Apply mathematical functions, format dates, extract values</li>
     * <li><strong>Performance optimization:</strong> Process large datasets server-side instead of client-side</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1) base complexity, but depends on the query and number of results processed
     * </p>
     *
     * @param index the index name, as a key
     * @param query the base filtering query that retrieves documents for aggregation
     * @return the result of the aggregate command containing processed results, see {@link SearchReply}
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.aggregate/">FT.AGGREGATE</a>
     * @see <a href=
     *      "https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/aggregations/">Aggregations</a>
     * @see SearchReply
     * @see AggregateArgs
     * @see #ftAggregate(String, Object, AggregateArgs)
     */
    @Experimental
    Executions<AggregationReply<K, V>> ftAggregate(String index, V query);

    /**
     * Run a search query on an index and perform advanced aggregate transformations with a processing pipeline.
     *
     * <p>
     * This command executes a search query and applies a sophisticated aggregation pipeline to transform, group, sort, and
     * analyze the results. The {@link AggregateArgs} parameter defines a series of operations that process the data
     * server-side, enabling powerful analytics and data transformation capabilities directly within Redis.
     * </p>
     *
     * <p>
     * The aggregation pipeline supports the following operations:
     * </p>
     * <ul>
     * <li><strong>LOAD:</strong> Load specific document attributes for processing</li>
     * <li><strong>GROUPBY:</strong> Group results by one or more properties</li>
     * <li><strong>REDUCE:</strong> Apply reduction functions (COUNT, SUM, AVG, MIN, MAX, etc.)</li>
     * <li><strong>SORTBY:</strong> Sort results by specified properties</li>
     * <li><strong>APPLY:</strong> Apply mathematical expressions and transformations</li>
     * <li><strong>FILTER:</strong> Filter results based on computed values</li>
     * <li><strong>LIMIT:</strong> Paginate results efficiently</li>
     * <li><strong>WITHCURSOR:</strong> Enable cursor-based pagination for large result sets</li>
     * </ul>
     *
     * <h3>Performance Considerations:</h3>
     * <ul>
     * <li>Use SORTABLE fields for efficient grouping and sorting operations</li>
     * <li>Apply filters early in the pipeline to reduce processing overhead</li>
     * <li>Use WITHCURSOR for large result sets to avoid memory issues</li>
     * <li>Load only necessary attributes to minimize data transfer</li>
     * <li>Consider using LIMIT to restrict result set size</li>
     * </ul>
     *
     * <p>
     * <strong>Time complexity:</strong> Non-deterministic, depends on the query and aggregation operations performed. Generally
     * linear to the number of results processed through the pipeline.
     * </p>
     *
     * @param index the index name, as a key
     * @param query the base filtering query that retrieves documents for aggregation
     * @param args the aggregate arguments defining the processing pipeline and operations
     * @return the result of the aggregate command containing processed and transformed results, see {@link SearchReply}
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.aggregate/">FT.AGGREGATE</a>
     * @see <a href=
     *      "https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/aggregations/">Aggregations</a>
     * @see <a href=
     *      "https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/aggregations/#cursor-api">Cursor
     *      API</a>
     * @see SearchReply
     * @see AggregateArgs
     * @see #ftAggregate(String, Object)
     * @see #ftCursorread(String, Cursor)
     */
    @Experimental
    Executions<AggregationReply<K, V>> ftAggregate(String index, V query, AggregateArgs<K, V> args);

    /**
     * Read next results from an existing cursor and optionally override the batch size.
     *
     * <p>
     * This command is used to read the next batch of results from a cursor that was created by
     * {@link #ftAggregate(String, Object, AggregateArgs)} with the {@code WITHCURSOR} option. Cursors provide an efficient way
     * to iterate through large result sets without loading all results into memory at once.
     * </p>
     *
     * <p>
     * The {@code count} parameter overrides the {@code COUNT} value specified in the original {@code FT.AGGREGATE} command,
     * allowing you to control the batch size for this specific read operation.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @param cursor the cursor obtained from a previous {@code FT.AGGREGATE} or {@code FT.CURSOR READ} command
     * @param count the number of results to read; overrides the {@code COUNT} from {@code FT.AGGREGATE}
     * @return the next batch of results; see {@link AggregationReply}
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.cursor-read/">FT.CURSOR READ</a>
     * @see <a href=
     *      "https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/aggregations/#cursor-api">Cursor
     *      API</a>
     * @see AggregationReply
     * @see #ftAggregate(String, Object, AggregateArgs)
     */
    @Experimental
    Executions<AggregationReply<K, V>> ftCursorread(String index, Cursor cursor, int count);

    /**
     * Read next results from an existing cursor using the default batch size.
     *
     * <p>
     * This command is used to read the next batch of results from a cursor created by
     * {@link #ftAggregate(String, Object, AggregateArgs)} with the {@code WITHCURSOR} option. This variant uses the default
     * batch size that was specified in the original {@code FT.AGGREGATE} command's {@code WITHCURSOR} clause.
     * </p>
     *
     * <p>
     * Cursors provide an efficient way to iterate through large result sets without loading all results into memory at once.
     * When the cursor is exhausted (no more results), the returned {@link SearchReply} will have a cursor id of 0.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name
     * @param cursor the cursor obtained from a previous {@code FT.AGGREGATE} or {@code FT.CURSOR READ} command
     * @return the next batch of results; see {@link AggregationReply}
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.cursor-read/">FT.CURSOR READ</a>
     * @see <a href=
     *      "https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/aggregations/#cursor-api">Cursor
     *      API</a>
     * @see AggregationReply
     * @see #ftAggregate(String, Object, AggregateArgs)
     */
    @Experimental
    Executions<AggregationReply<K, V>> ftCursorread(String index, Cursor cursor);

    /**
     * Delete a cursor and free its associated resources.
     *
     * <p>
     * This command is used to explicitly delete a cursor created by {@link #ftAggregate(String, Object, AggregateArgs)} with
     * the {@code WITHCURSOR} option. Deleting a cursor frees up server resources and should be done when you no longer need to
     * read more results from the cursor.
     * </p>
     *
     * <p>
     * <strong>Important:</strong> Cursors have a default timeout and may be automatically deleted by Redis if not accessed
     * within the timeout period. However, it's good practice to explicitly delete cursors when you're finished with them to
     * free up resources immediately.
     * </p>
     *
     * <p>
     * Once a cursor is deleted, any subsequent attempts to read from it using {@link #ftCursorread(String, Cursor)} or
     * {@link #ftCursorread(String, Cursor, int)} will result in an error.
     * </p>
     *
     * <p>
     * <strong>Time complexity:</strong> O(1)
     * </p>
     *
     * @param index the index name, as a key
     * @param cursor the cursor obtained from a previous {@code FT.AGGREGATE} or {@code FT.CURSOR READ} command
     * @return {@code "OK"} if the cursor was successfully deleted
     * @since 6.8
     * @see <a href="https://redis.io/docs/latest/commands/ft.cursor-del/">FT.CURSOR DEL</a>
     * @see <a href=
     *      "https://redis.io/docs/latest/develop/interact/search-and-query/advanced-concepts/aggregations/#cursor-api">Cursor
     *      API</a>
     * @see #ftAggregate(String, Object, AggregateArgs)
     * @see #ftCursorread(String, Cursor)
     * @see #ftCursorread(String, Cursor, int)
     */
    @Experimental
    Executions<String> ftCursordel(String index, Cursor cursor);

}
