/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.service;

import com.stripe.StripeClient;
import com.stripe.exception.StripeException;
import com.stripe.model.Price;
import com.stripe.model.Subscription;
import com.stripe.model.billing.Meter;
import com.stripe.model.billingportal.Session;
import com.stripe.param.CustomerCreateParams;
import com.stripe.param.SubscriptionCreateParams;
import com.stripe.param.billing.MeterEventCreateParams;
import com.stripe.param.billingportal.SessionCreateParams;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.utils.Utils;
import lombok.Getter;
import lombok.SneakyThrows;
import org.apache.commons.lang3.StringUtils;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import static io.meeds.billing.utils.Utils.HUB_PLAN_CHANGED;
import static io.meeds.billing.utils.Utils.SUBSCRIPTION_CANCELED;
import static io.meeds.billing.utils.Utils.SUBSCRIPTION_PAST_DUE;
import static io.meeds.billing.utils.Utils.SUBSCRIPTION_RENEWAL;
import static io.meeds.billing.utils.Utils.SubscriptionStatus.CANCELED;
import static io.meeds.billing.utils.Utils.TRIAL_ENDING;
import static io.meeds.billing.utils.Utils.TRIAL_EXPIRED;
import static io.meeds.billing.utils.Utils.getTemplateTrialDays;

@Service
public class BillingService {

  private static final Log                      LOG            = ExoLogger.getLogger(BillingService.class);

  private final HubBilling                      hubBilling;

  private final StripeClient                    stripeClient;

  private final SubscriptionEmailReminderService subscriptionEmailReminderService;

  private final SpaceService                    spaceService;

  private final HubSettingService               hubSettingService;

  private final Map<String, String>              meterEventName = new ConcurrentHashMap<>();

  @Getter
  private boolean                                enabled;



  public BillingService(HubBilling hubBilling, SubscriptionEmailReminderService subscriptionEmailReminderService, SpaceService spaceService,  HubSettingService hubSettingService) {
    this.hubBilling = hubBilling;
    this.subscriptionEmailReminderService = subscriptionEmailReminderService;
    this.spaceService = spaceService;
    this.hubSettingService = hubSettingService;
    this.enabled = this.hubBilling.isEnabled() && StringUtils.isNotBlank(this.hubBilling.getSecretKey());
    if (this.hubBilling.isEnabled() && StringUtils.isBlank(this.hubBilling.getSecretKey())) {
      LOG.warn("Billing is enabled but secret key is missing. The billing process will be disabled.");
    }
    this.stripeClient = enabled ? new StripeClient(this.hubBilling.getSecretKey()) : null;
  }

  /**
   * Creates a new Stripe customer for the given space
   * @param space
   * @param identity
   * @return stripe customer ID
   */
  public String createCustomer(Space space, Identity identity) throws StripeException {
    Map<String, String> customerMetadata = new HashMap<String, String>();
    customerMetadata.put("spaceId", space.getId());
    customerMetadata.put("userIdentityId", identity.getId());
    CustomerCreateParams params = CustomerCreateParams.builder()
                                                      .setName(space.getDisplayName())
                                                      .setDescription(space.getDescription())
                                                      .setEmail(identity.getProfile().getEmail())
                                                      .setMetadata(customerMetadata)
                                                      .build();
    return this.stripeClient.customers().create(params).getId();
  }

  /**
   * Creates a new Stripe subscription for the given customer (space).
   * @param space
   * @param customerId
   * @return the created Stripe {@link Subscription}
   */
  public Subscription subscribe(Space space, String customerId, String priceId) throws StripeException {
    Long spaceTemplateId = space.getTemplateId();
    String trialDaysValue = getTemplateTrialDays(spaceTemplateId);
    long trialDays = StringUtils.isEmpty(trialDaysValue) ? 0 : Long.parseLong(trialDaysValue);
    Map<String, String> metadata = new HashMap<>();
    metadata.put("spaceId", space.getId());
    SubscriptionCreateParams.Item subscriptionItem = SubscriptionCreateParams.Item.builder().setPrice(priceId).build();
    SubscriptionCreateParams params =
                                    SubscriptionCreateParams.builder()
                                                            .setPaymentBehavior(SubscriptionCreateParams.PaymentBehavior.DEFAULT_INCOMPLETE)
                                                            .setCustomer(customerId)
                                                            .addItem(subscriptionItem)
                                                            .setMetadata(metadata)
                                                            .setTrialPeriodDays(trialDays)
                                                            .build();
    return stripeClient.subscriptions().create(params);
  }

  /**
   * Creates a Stripe Billing Portal session for the given customer.
   * @param spaceId
   * @param userName
   * @return the stripe customer portal url
   */
  public String createCustomerPortalSession(long spaceId, String userName) throws ObjectNotFoundException, IllegalAccessException {
    HubBillingSettings hubBillingSettings = hubSettingService.getSettingsBySpaceId(spaceId, userName);
    return createCustomerPortalSession(hubBillingSettings.getCustomerId());
  }

  /**
   * Creates a Stripe Billing Portal session for the given customer.
   * @param customerId
   * @return the stripe customer portal url
   */

  public String createCustomerPortalSession(String customerId) {
    try {
      SessionCreateParams params = SessionCreateParams.builder().setCustomer(customerId).build();
      Session session = stripeClient.billingPortal().sessions().create(params);
      return session.getUrl();
    } catch (StripeException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }

  /**
   * Cancels an active Stripe subscription.
   * @param spaceId
   * @param userName
   * @throws ObjectNotFoundException
   * @throws IllegalAccessException
   */
  public void cancelSubscription(long spaceId, String userName) throws ObjectNotFoundException, IllegalAccessException {
    HubBillingSettings hubBillingSettings = hubSettingService.getSettingsBySpaceId(spaceId, userName);
    cancelSubscription(hubBillingSettings.getSubscriptionId());
    hubSettingService.updateSubscriptionStatus(hubBillingSettings.getId(), Utils.SubscriptionStatus.CANCELED.name());
    notifyAdminsOnSubscriptionCancellation(null, spaceId);
  }

  /**
   * Cancels an active Stripe subscription.
   * @param subscriptionId
   */
  public void cancelSubscription(String subscriptionId) {
    try {
      this.stripeClient.subscriptions().cancel(subscriptionId);
    } catch (StripeException e) {
      throw new IllegalArgumentException(e.getMessage());
    }
  }

  public Subscription getSubscription(String subscriptionId) throws Exception {
    return this.stripeClient.subscriptions().retrieve(subscriptionId);
  }

  /**
   * Sends an email about the current subscription status.
   * @param spaceId
   * @param authenticatedUser
   */
  public void notifyOnSubscriptionStatus(long spaceId, String authenticatedUser, boolean maxOfUsersExceeded) throws IllegalAccessException, ObjectNotFoundException {
    Space space = spaceService.getSpaceById(spaceId);
    if (space == null) {
      throw new ObjectNotFoundException("Space not found for Space Id: " + spaceId);
    }
    if (!spaceService.isMember(space, authenticatedUser)) {
      throw new IllegalAccessException("User" + authenticatedUser + " is not allowed to mange space " + spaceId);
    }
    HubBillingSettings hubBillingSettings = hubSettingService.getSettingsBySpaceId(String.valueOf(spaceId));
    if (maxOfUsersExceeded) {
      notifyHubAdminsOnUsersLimitExceeded(authenticatedUser, hubBillingSettings.getSpaceId());
    } else if (StringUtils.equalsIgnoreCase(hubBillingSettings.getSubscriptionStatus(), CANCELED.name())) {
      notifyAdminsOnSubscriptionCancellation(authenticatedUser, hubBillingSettings.getSpaceId());
    } else {
      notifyHubAdminsOnSubscriptionStatus(hubBillingSettings, authenticatedUser);
    }
  }

  /**
   * Sends an email to the hub administrator about the current subscription status.
   * @param hubBillingSettings
   * @param authenticatedUser
   */
  public void notifyHubAdminsOnSubscriptionStatus(HubBillingSettings hubBillingSettings, String authenticatedUser) {
    String context = getNotificationContext(hubBillingSettings.getSubscriptionStatus(), hubBillingSettings.getSubscriptionPreviousStatus());
    if (context != null) {
      sendMailNotification(authenticatedUser, hubBillingSettings.getSpaceId(), context);
    }
  }

  /**
   * Sends an email to the administrator about the max of users exceeded.
   * @param authenticatedUser
   */
  public void notifyHubAdminsOnUsersLimitExceeded(String authenticatedUser, long spaceId) {
    sendMailNotification(authenticatedUser, spaceId, Utils.HUB_USERS_LIMIT_EXCEEDED);
  }

  /**
   * Sends an email to the administrators about the hub plan change.
   * @param spaceId
   */
  public void notifyAdminsOnPlanChange(long spaceId) {
    sendMailNotification(null, spaceId, Utils.HUB_PLAN_CHANGED);
  }

  public void notifyAdminsOnSubscriptionCancellation(String authenticatedUser ,long spaceId) {
    sendMailNotification(authenticatedUser, spaceId, Utils.SUBSCRIPTION_CANCELED);
  }

  /**
   * Retrieves a Stripe {@link Price} object by its unique identifier.
   *
   * @param id the unique identifier of the Stripe price to retrieve
   * @return the {@link Price} object if found; {@code null} if an error occurs or the price cannot be retrieved
   */
  public Price getPriceById(String id) throws StripeException {
    return this.stripeClient.prices().retrieve(id);
  }

  /**
   * Reports the current number of members in a space to Stripe as a metered event.
   * <p>
   * This method retrieves the associated Stripe meter event name from the configured
   * meter ID, counts the members in the corresponding space, and sends a metered
   * usage event to Stripe with the count value.
   * </p>
   *
   * @param hubBillingSettings the billing settings containing the meter ID, customer ID, and space ID
   * @throws ObjectNotFoundException if the space corresponding to the given space ID is not found
   * @throws Exception if an error occurs while communicating with Stripe or retrieving meter details
   */
  public void reportMembersCount(HubBillingSettings hubBillingSettings) throws Exception {
    String meterId = hubBillingSettings.getHubBillingPlan().getMeterId();
    if (meterId == null) {
      return;
    }
    Space space = spaceService.getSpaceById(hubBillingSettings.getSpaceId());
    if (space == null) {
      throw new ObjectNotFoundException("Space with id " + hubBillingSettings.getSpaceId() + " not found");
    }
    String eventName = meterEventName.computeIfAbsent(meterId, this::getEventNameByMeterId);
    if (eventName == null) {
      throw new ObjectNotFoundException("Meter Event with id " + meterId + " not found");
    }
    String customerId = hubBillingSettings.getCustomerId();
    String usersCount = String.valueOf(space.getMembers().length);
    MeterEventCreateParams params = MeterEventCreateParams.builder()
                                                          .setEventName(eventName)
                                                          .putPayload("value", usersCount)
                                                          .putPayload("stripe_customer_id", customerId)
                                                          .build();

    this.stripeClient.billing().meterEvents().create(params);
    LOG.info("Reported usage of {} members for space {} with meter event {}",
            usersCount,
            hubBillingSettings.getSpaceId(),
            eventName);
  }

  private void sendMailNotification(String authenticatedUser, long spaceId, String context) {
    List<String> receivers = context.equals(SUBSCRIPTION_CANCELED)  || context.equals(HUB_PLAN_CHANGED) ? Utils.getAdministrators() : Utils.getSpaceManagers(spaceId);
    subscriptionEmailReminderService.sendEmailNotification(authenticatedUser, receivers, spaceId, context);
  }

  private String getNotificationContext(String subscriptionStatus, String subscriptionPreviousStatus) {
    if (subscriptionStatus == null) {
      return null;
    }
    if (Utils.SubscriptionStatus.TRIALING.name().equalsIgnoreCase(subscriptionStatus)) {
      return TRIAL_ENDING;
    }
    if (Utils.SubscriptionStatus.ACTIVE.name().equalsIgnoreCase(subscriptionStatus)) {
      return SUBSCRIPTION_RENEWAL;
    }
    if (Utils.SubscriptionStatus.INCOMPLETE.name().equalsIgnoreCase(subscriptionStatus)) {
      return SUBSCRIPTION_PAST_DUE;
    }
    if (Utils.SubscriptionStatus.PAST_DUE.name().equalsIgnoreCase(subscriptionStatus)) {
      return Utils.SubscriptionStatus.TRIALING.name().equalsIgnoreCase(subscriptionPreviousStatus) ? TRIAL_EXPIRED
                                                                                                   : SUBSCRIPTION_PAST_DUE;
    }
    return null;
  }

  @SneakyThrows
  private String getEventNameByMeterId(String meterId) {
    Meter meter = stripeClient.billing().meters().retrieve(meterId);
    return meter.getEventName();
  }

}
