/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.storage;

import io.meeds.billing.dao.HubBillingSettingsDAO;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import io.meeds.billing.entity.HubBillingSettingsEntity;
import io.meeds.billing.model.HubBillingSettings;
import io.meeds.billing.utils.Utils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.annotation.Caching;
import org.springframework.stereotype.Component;

import java.util.List;

import static io.meeds.billing.utils.Utils.fromEntity;

@Component
public class HubBillingSettingsStorage {

  public static final String    HUB_SETTING_CACHE_NAME = "billing.HubBillingSettings";

  public static final String    METER_LIST_CACHE_NAME  = "billing.MeteredHubBillingSettingList";

  @Autowired
  private HubBillingSettingsDAO hubBillingSettingsDAO;

  @Caching(evict = {
          @CacheEvict(value = HUB_SETTING_CACHE_NAME, allEntries = true),
          @CacheEvict(value = METER_LIST_CACHE_NAME, allEntries = true)
  })
  public HubBillingSettings save(HubBillingSettings hubBillingSettings) {
    HubBillingSettingsEntity hubBillingSettingsEntity = Utils.toEntity(hubBillingSettings);
    hubBillingSettingsEntity = hubBillingSettingsDAO.save(hubBillingSettingsEntity);
    return fromEntity(hubBillingSettingsEntity);
  }

  @Cacheable(value = HUB_SETTING_CACHE_NAME, key = "#p0")
  public HubBillingSettings getHubBillingSettingsBySpaceId(long spaceId) throws ObjectNotFoundException {
    HubBillingSettingsEntity hubBillingSettingsEntity = hubBillingSettingsDAO.findBySpaceId(spaceId);
    if (hubBillingSettingsEntity != null) {
        return fromEntity(hubBillingSettingsEntity);
    }
    throw new ObjectNotFoundException("No settings found for hub: " + spaceId);
  }

  @Caching(evict = {
          @CacheEvict(value = HUB_SETTING_CACHE_NAME, allEntries = true),
          @CacheEvict(value = METER_LIST_CACHE_NAME, allEntries = true)
  })
  public void updateSubscriptionStatus(Long billingHubId, String newSubscriptionStatus) {
    HubBillingSettingsEntity hubBillingSettingsEntity = hubBillingSettingsDAO.findById(billingHubId).orElse(null);
    if (hubBillingSettingsEntity != null) {
      hubBillingSettingsEntity.setSubscriptionStatus(newSubscriptionStatus);
      hubBillingSettingsDAO.save(hubBillingSettingsEntity);
    } else {
      throw new IllegalArgumentException("No settings found for hub: " + billingHubId);
    }
  }

  @Cacheable(value = METER_LIST_CACHE_NAME, key = "'meteredList'")
  public List<HubBillingSettings> getMeteredHubBillingSettingList() {
    return hubBillingSettingsDAO.findByPlan_MeterIdIsNotNull().stream().map(Utils::fromEntity).toList();
  }

  @Cacheable(value = HUB_SETTING_CACHE_NAME, key = "#p0")
  public HubBillingSettings getHubBillingSettingBySubscriptionId(String subscriptionId) throws ObjectNotFoundException {
    HubBillingSettingsEntity hubBillingSettingsEntity = hubBillingSettingsDAO.findBySubscriptionId(subscriptionId);
    if (hubBillingSettingsEntity != null) {
      return fromEntity(hubBillingSettingsEntity);
    }
    throw new ObjectNotFoundException("No settings found for hub with subscription id: " + subscriptionId);
  }
}
