/*
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.listener;

import com.stripe.model.Price;
import com.stripe.model.Subscription;
import io.meeds.billing.model.HubBilling;
import io.meeds.billing.service.BillingService;
import io.meeds.billing.utils.Utils;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.identity.model.Identity;
import org.exoplatform.social.core.manager.IdentityManager;
import org.exoplatform.social.core.space.SpaceListenerPlugin;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceLifeCycleEvent;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import jakarta.annotation.PostConstruct;

import org.exoplatform.services.listener.ListenerService;

import io.meeds.billing.service.HubSettingService;

import static io.meeds.billing.utils.Utils.getTemplatePriceId;

@Component
public class BillingSpaceListener extends SpaceListenerPlugin {

  private static final Log  LOG = ExoLogger.getLogger(BillingSpaceListener.class);

  @Autowired
  private ListenerService   listenerService;

  @Autowired
  private HubSettingService hubSettingService;

  @Autowired
  private IdentityManager   identityManager;

  @Autowired
  private HubBilling        hubBilling;

  @Autowired
  private SpaceService      spaceService;

  @Autowired
  private BillingService    billingService;

  @PostConstruct
  public void init() {
   spaceService.registerSpaceListenerPlugin(this);
  }

  @Override
  public void spaceCreated(SpaceLifeCycleEvent event) {
    Space space = event.getSpace();
    if (!isBillableSpaceTemplate(space)) {
      return;
    }
    String username = event.getSource();
    try {
      Identity identity = identityManager.getOrCreateUserIdentity(username);
      String priceId = getTemplatePriceId(space.getTemplateId());
      if (priceId == null) {
        LOG.error("Missing Price ID for space '{}'. The Price ID must not be null or empty.", space.getDisplayName());
        return;
      }
      Price price = billingService.getPriceById(priceId);
      String customerId = billingService.createCustomer(space, identity);
      Subscription subscription = billingService.subscribe(space, customerId, price.getId());
      hubSettingService.createSettings(space, subscription, identity);
    } catch (Exception e) {
      LOG.error("Error while creating billing setup for space '{}' with user {} : {}", space.getDisplayName(), username, e.getMessage(), e);
    }
  }
  
  private boolean isBillableSpaceTemplate(Space space) {
    return isBillingEnabled() && hubBilling.getSpaceTemplateIds().contains(space.getTemplateId());
  }

  private boolean isBillingEnabled() {
    return billingService.isEnabled();
  }

}
