/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.listener;

import io.meeds.billing.model.HubBilling;
import jakarta.annotation.PostConstruct;
import org.exoplatform.portal.config.UserPortalConfigService;
import org.exoplatform.portal.config.model.Application;
import org.exoplatform.portal.config.model.PortalConfig;
import org.exoplatform.portal.config.model.TransientApplicationState;
import org.exoplatform.portal.config.serialize.model.SiteLayout;
import org.exoplatform.portal.mop.SiteKey;
import org.exoplatform.portal.mop.service.LayoutService;
import org.exoplatform.services.listener.Event;
import org.exoplatform.services.listener.ListenerBase;
import org.exoplatform.services.listener.ListenerService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.social.core.space.model.Space;
import org.exoplatform.social.core.space.spi.SpaceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import java.util.ArrayList;

@Component
@Order(Ordered.LOWEST_PRECEDENCE)
public class SpaceSiteCreatedListener implements ListenerBase<SiteKey, SiteKey> {

  private static final Log LOG = ExoLogger.getLogger(SpaceSiteCreatedListener.class);

  @Autowired
  private ListenerService  listenerService;

  @Autowired
  private LayoutService    layoutService;

  @Autowired
  private SpaceService     spaceService;

  @Autowired
  private HubBilling       hubBilling;

  @PostConstruct
  public void init() {
    listenerService.addListener(UserPortalConfigService.SITE_TEMPLATE_INSTANTIATED, this);
  }

  @Override
  public void onEvent(Event<SiteKey, SiteKey> event) throws Exception {
    SiteKey siteKey = event.getData();
    if (!siteKey.getName().startsWith("/spaces/")) {
      return;
    }
    Space space = spaceService.getSpaceByGroupId(siteKey.getName());
    if (space == null) {
      return;
    }
    boolean isBillableSpace = hubBilling.isEnabled() && hubBilling.getSpaceTemplateIds().contains(space.getTemplateId());
    if (isBillableSpace) {
      updateSpaceLayout(siteKey);
    }
  }

  private void updateSpaceLayout(SiteKey siteKey) {
    try {
      PortalConfig portalConfig = layoutService.getPortalConfig(siteKey);
      SiteLayout siteLayout = portalConfig.getPortalLayout();
      Application application = new Application();
      application.setAccessPermissions(portalConfig.getAccessPermissions());
      TransientApplicationState transientApplicationState = new TransientApplicationState("billing/HubSubscriptionRestriction");
      application.setState(transientApplicationState);
      siteLayout.setChildren(new ArrayList<>(siteLayout.getChildren()));
      siteLayout.getChildren().add(application);
      portalConfig.setPortalLayout(siteLayout);
      layoutService.save(portalConfig);
    } catch (Exception e) {
      LOG.error("Error while adding portlet 'HubSubscriptionRestriction' to space site layout with id '{}'", siteKey, e);
    }
  }
}
