/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.rest;

import io.meeds.billing.service.BillingService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.exoplatform.commons.ObjectAlreadyExistsException;
import org.exoplatform.commons.exception.ObjectNotFoundException;
import org.exoplatform.services.security.ConversationState;
import org.exoplatform.social.core.manager.IdentityManager;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.annotation.Secured;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.server.ResponseStatusException;

@RestController
@RequestMapping("/billing")
public class BillingRest {

  @Autowired
  private BillingService    billingService;

  @Autowired
  private IdentityManager   identityManager;

  @GetMapping(path = "/customer-portal/{spaceId}", produces = MediaType.TEXT_PLAIN_VALUE)
  @Secured("users")
  @Operation(
          summary = "Get a customer portal session link",
          method = "GET",
          description = "Get a customer portal session link")
  @ApiResponses(value = {
          @ApiResponse(responseCode = "200", description = "Request fulfilled"),
          @ApiResponse(responseCode = "404", description = "Object not found"),
          @ApiResponse(responseCode = "403", description = "Unauthorized operation"),
          @ApiResponse(responseCode = "500", description = "Internal server error") })
  public ResponseEntity<String> createCustomerPortalSession(@Parameter(description = "space identifier")
                                                            @PathVariable("spaceId")
                                                            long spaceId) {
    String authenticatedUser = ConversationState.getCurrent().getIdentity().getUserId();
    try {
      String portalUrl = billingService.createCustomerPortalSession(spaceId, authenticatedUser);
      return ResponseEntity.ok().body(portalUrl);
    } catch (IllegalAccessException e) {
      throw new ResponseStatusException(HttpStatus.FORBIDDEN, e.getMessage());
    } catch (ObjectNotFoundException e) {
      throw new ResponseStatusException(HttpStatus.NOT_FOUND, e.getMessage());
    } catch (IllegalArgumentException e) {
      throw new ResponseStatusException(HttpStatus.BAD_REQUEST, e.getMessage());
    }
  }

  @DeleteMapping(path = "subscription/{spaceId}/cancel")
  @Secured("users")
  @Operation(
          summary = "Cancels a customer subscription",
          method = "DELETE",
          description = "Cancels a customer subscription")
  @ApiResponses(value = {
          @ApiResponse(responseCode = "204", description = "Request fulfilled"),
          @ApiResponse(responseCode = "404", description = "Object not found"),
          @ApiResponse(responseCode = "403", description = "Unauthorized operation"),
          @ApiResponse(responseCode = "500", description = "Internal server error") })
  @ResponseStatus(HttpStatus.NO_CONTENT)
  public void cancelSubscription(@Parameter(description = "space technical identifier")
                                 @PathVariable("spaceId")
                                 long spaceId) {
    String authenticatedUser = ConversationState.getCurrent().getIdentity().getUserId();
    try {
      billingService.cancelSubscription(spaceId, authenticatedUser);
    } catch (IllegalAccessException e) {
      throw new ResponseStatusException(HttpStatus.FORBIDDEN, e.getMessage());
    } catch (ObjectNotFoundException e) {
      throw new ResponseStatusException(HttpStatus.NOT_FOUND, e.getMessage());
    } catch (IllegalArgumentException e) {
      throw new ResponseStatusException(HttpStatus.BAD_REQUEST, e.getMessage());
    }
  }

  @PostMapping(path = "notify/subscription-status/{spaceId}")
  @Secured("users")
  @Operation(
          summary = "Notify space admins on the subscription status",
          method = "POST",
          description = "Notify space admins on the subscription status for example trial ending")
  @ApiResponses(value = {
          @ApiResponse(responseCode = "204", description = "Request fulfilled"),
          @ApiResponse(responseCode = "404", description = "Object not found"),
          @ApiResponse(responseCode = "403", description = "Unauthorized operation") })
  @ResponseStatus(HttpStatus.NO_CONTENT)
  public void notifyOnSubscriptionStatus(@PathVariable("spaceId")
                                              long spaceId,
                                              @RequestParam(name = "maxOfUsersExceeded", required = false, defaultValue = "false")
                                              boolean maxOfUsersExceeded) {
    String authenticatedUser = ConversationState.getCurrent().getIdentity().getUserId();
    try {
      billingService.notifyOnSubscriptionStatus(spaceId, authenticatedUser, maxOfUsersExceeded);
    } catch (IllegalAccessException e) {
      throw new ResponseStatusException(HttpStatus.FORBIDDEN, e.getMessage());
    } catch (ObjectNotFoundException e) {
      throw new ResponseStatusException(HttpStatus.NOT_FOUND, e.getMessage());
    }
  }

  @PostMapping(path = "/enable/{spaceId}")
  @Secured("administrators")
  @Operation(
          summary = "Enable billing for an existing space",
          method = "POST",
          description = "Activates billing on an already created space using the provided price ID.")
  @ApiResponses(value = {
          @ApiResponse(responseCode = "200", description = "Request fulfilled"),
          @ApiResponse(responseCode = "404", description = "Object not found"),
          @ApiResponse(responseCode = "403", description = "Unauthorized operation"),
          @ApiResponse(responseCode = "400", description = "Bad request") })
  public ResponseEntity<String> enableExistingSpaceBilling(@Parameter(description = "The ID of the space to convert into a billable space")
                                                           @PathVariable("spaceId")
                                                           long spaceId,
                                                           @Parameter(description = "The Stripe price ID to associate with the space billing")
                                                           @RequestParam(name = "priceId")
                                                           String priceId,
                                                           @Parameter(description = "space manager username who will be registered as the billing admin")
                                                           @RequestParam(name = "spaceManager")
                                                           String spaceManager) {
    try {
      billingService.enableExistingSpaceBilling(spaceId, priceId, spaceManager);
      return ResponseEntity.ok("billing successfully enabled for space " + spaceId);
    } catch (IllegalAccessException e) {
      throw new ResponseStatusException(HttpStatus.FORBIDDEN, e.getMessage());
    } catch (ObjectAlreadyExistsException | IllegalArgumentException e) {
      throw new ResponseStatusException(HttpStatus.BAD_REQUEST, e.getMessage());
    } catch (ObjectNotFoundException e) {
      throw new ResponseStatusException(HttpStatus.NOT_FOUND, e.getMessage());
    }
  }
}
