/*
 * This file is part of the Meeds project (https://meeds.io/).
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.billing.service;

import io.meeds.billing.utils.Utils;
import io.meeds.portal.permlink.model.PermanentLinkObject;
import io.meeds.portal.permlink.service.PermanentLinkService;
import io.meeds.social.space.plugin.SpacePermanentLinkPlugin;
import lombok.Getter;
import lombok.Setter;
import lombok.SneakyThrows;
import org.apache.commons.lang3.LocaleUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.StringEscapeUtils;
import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.commons.utils.MailUtils;
import org.exoplatform.portal.Constants;
import org.exoplatform.portal.branding.BrandingService;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.services.mail.MailService;
import org.exoplatform.services.organization.OrganizationService;
import org.exoplatform.services.organization.User;
import org.exoplatform.services.organization.UserProfile;
import org.exoplatform.services.resources.ResourceBundleService;
import org.exoplatform.social.notification.LinkProviderUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import javax.mail.Message.RecipientType;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.IOUtils;

import org.springframework.beans.factory.annotation.Value;

import static io.meeds.billing.utils.Utils.HUB_PLAN_CHANGED;
import static io.meeds.billing.utils.Utils.HUB_USERS_LIMIT_EXCEEDED;
import static io.meeds.billing.utils.Utils.SUBSCRIPTION_CANCELED;
import static io.meeds.billing.utils.Utils.SUBSCRIPTION_PAST_DUE;
import static io.meeds.billing.utils.Utils.SUBSCRIPTION_RENEWAL;
import static io.meeds.billing.utils.Utils.TRIAL_ENDING;
import static io.meeds.billing.utils.Utils.TRIAL_EXPIRED;

@Service
public class SubscriptionEmailReminderService {

  private static final Log      LOG                       = ExoLogger.getLogger(SubscriptionEmailReminderService.class);

  private static final Pattern  I18N_LABEL_PATTERN        = Pattern.compile("\\$\\{([a-zA-Z0-9\\.]+)\\}");

  private static final String   USER_FULL_NAME_PARAM      = "$USER_FULL_NAME";

  private static final String   RECIPIENT_FULL_NAME_PARAM = "$RECIPIENT_FULL_NAME";

  private static final String   SITE_NAME_PARAM           = "$SITE_NAME";

  private static final String   COMPANY_LINK_PARAM        = "$COMPANY_LINK";

  private static final String   BODY_FIRST_PART_PARAM     = "$BODY_FIRST_PART";

  private static final String   BODY_SECOND_PART_PARAM    = "$BODY_SECOND_PART";

  private static final String   PRIMARY_COLOR_PARAM       = "$PRIMARY_COLOR";

  @Value("${meeds.billing.subscription.reminder.email.templatePath:assets/subscription-email-reminder.html}")
  @Getter
  @Setter
  private String                emailBodyPath;

  @Setter
  private String                emailBodyTemplate;

  @Autowired
  private MailService           mailService;

  @Autowired
  private OrganizationService   organizationService;

  @Autowired
  private ResourceBundleService resourceBundleService;

  @Autowired
  private BrandingService       brandingService;

  @Autowired
  private PermanentLinkService  permanentLinkService;

  
  @SneakyThrows
  public void sendEmailNotification(String userName, List<String> recipients, long spaceId, String context) {
    String senderEmail = userName != null ? getUserMail(userName) : MailUtils.getSenderEmail();
    String senderFullName = userName != null ? getUserFullName(userName) : MailUtils.getSenderName();
    String from = userName == null && senderFullName != null ? senderFullName + "<" + senderEmail + ">" : senderEmail;
    String lang = getUserLang(userName);
    String subject = getEmailSubject(context, lang);

    for (String recipient : recipients) {
      try {
        sendEmail(from, userName, senderFullName, recipient, spaceId, context, lang, subject);
      } catch (Exception e) {
        LOG.warn("Unable to send email to {} , {}", recipient, e.getMessage());
      }
    }
  }

  private void sendEmail(String from,
                         String senderUserName,
                         String senderFullName,
                         String recipient,
                         long spaceId,
                         String context,
                         String lang,
                         String subject) throws Exception {
    String body = getEmailBody(senderUserName, senderFullName, getUserFullName(recipient), spaceId, context, lang);
    MimeMessage mimeMessage = new MimeMessage(mailService.getMailSession());
    mimeMessage.setFrom(from);
    mimeMessage.setRecipient(RecipientType.TO, new InternetAddress(getUserMail(recipient)));
    mimeMessage.setSubject(StringEscapeUtils.unescapeHtml4(subject), "UTF-8");
    mimeMessage.setSentDate(new Date());
    mimeMessage.setContent(body, "text/html; charset=utf-8");
    mailService.sendMessage(mimeMessage);
  }
  
  private String getEmailBody(String username,
                              String userFullName,
                              String recipientFullName,
                              long spaceId,
                              String context,
                              String lang) {
    String content = getEmailBody();
    content = content.replace(BODY_FIRST_PART_PARAM, getBodyFirstPart(username, userFullName, spaceId, context, lang));
    content = content.replace(BODY_SECOND_PART_PARAM, getBodySecondPart(spaceId, context, lang));
    content = content.replace(USER_FULL_NAME_PARAM, userFullName);
    content = content.replace(RECIPIENT_FULL_NAME_PARAM, recipientFullName);
    content = content.replace(SITE_NAME_PARAM, brandingService.getCompanyName());
    content = content.replace(COMPANY_LINK_PARAM, LinkProviderUtils.getBaseUrl());
    content = content.replace(PRIMARY_COLOR_PARAM, brandingService.getThemeStyle().get("primaryColor"));
    content = replaceI18nKey(content, lang);
    return content;
  }

  @SneakyThrows
  private String getEmailBody() {
    if (emailBodyTemplate == null) {
      try (InputStream inputStream = getClass().getClassLoader().getResourceAsStream(emailBodyPath)) {
        emailBodyTemplate = IOUtils.toString(inputStream, StandardCharsets.UTF_8);
      }
    }
    return emailBodyTemplate;
  }

  private String getUserMail(String userName) throws Exception {
    User user = organizationService.getUserHandler().findUserByName(userName);
    if (user == null) {
      throw new IllegalArgumentException();
    }
    return user.getEmail();
  }

  private String getUserFullName(String userName) throws Exception {
    User user = organizationService.getUserHandler().findUserByName(userName);
    if (user == null) {
      throw new IllegalArgumentException();
    }
    return user.getDisplayName();
  }

  private String getEmailSubject(String context, String lang) {
    Locale locale = LocaleUtils.toLocale(lang);
    switch (context) {
    case TRIAL_ENDING -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.subscription.trialEnding", locale);
    }
    case SUBSCRIPTION_RENEWAL -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.subscription.renewal", locale);
    }
    case TRIAL_EXPIRED -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.subscription.trialExpired", locale);
    }
    case SUBSCRIPTION_PAST_DUE -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.subscription.pastDue", locale);
    }
    case HUB_USERS_LIMIT_EXCEEDED -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.hub.maxOfUsersExceeded", locale);
    }
    case HUB_PLAN_CHANGED -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.hub.planChanged", locale);
    }
    case SUBSCRIPTION_CANCELED -> {
      return this.resourceBundleService.getSharedString("Notification.subject.billing.hub.subscriptionCanceled", locale);
    }
    }
    return "";
  }

  private String getBodyFirstPart(String username, String userFullName, long spaceId, String context, String lang) {
    if (StringUtils.isBlank(context)) {
      return "";
    }
    Locale locale = LocaleUtils.toLocale(lang);
    boolean userTriggered = username != null;

    String contentKey = switch (context) {
      case HUB_PLAN_CHANGED -> "Notification.body.firstPart.hub.planChanged";
      case SUBSCRIPTION_CANCELED -> userTriggered
              ? "Notification.body.subscription.cancelled.userTriggered"
              : "Notification.body.subscription.cancelled.systemTriggered";
      case SUBSCRIPTION_RENEWAL -> "Notification.body.subscription.renewal";
      case TRIAL_ENDING -> "Notification.body.subscription.trialEnding";
      case TRIAL_EXPIRED -> "Notification.body.subscription.trialExpired";
      case SUBSCRIPTION_PAST_DUE -> "Notification.body.subscription.pastDue";
      case HUB_USERS_LIMIT_EXCEEDED -> "Notification.body.firstPart.hub.maxOfUsersExceeded";
      default -> "";
    };
    String content = resourceBundleService.getSharedString(contentKey, locale);
    if (!userTriggered) {
      return content.replace("{0}", getSpaceFullLink(spaceId, null, null, null));
    }
    content = content.replace("{0}", getSenderFullNameLink(username, userFullName));
    content = content.contains("{1}") ? content.replace("{1}", getSpaceFullLink(spaceId, null, null, null)) : content;
    return content;
  }
  
  private String getBodySecondPart(long spaceId, String context, String lang) {
    String linkPart = "";
    if (context.equalsIgnoreCase(TRIAL_ENDING) || context.equalsIgnoreCase(SUBSCRIPTION_RENEWAL)) {
      linkPart = getSpaceFullLink(spaceId, "settings", "Notification.body.subscription.settings", lang);
    } else {
      linkPart = getSpaceFullLink(spaceId, null, "Notification.body.subscription.homeAccess", lang);
    }
    Locale locale = LocaleUtils.toLocale(lang);
    String bodySecondPart = "";
    switch (context) {
    case TRIAL_ENDING, SUBSCRIPTION_RENEWAL ->
      bodySecondPart = resourceBundleService.getSharedString("Notification.body.subscription.manage", locale);
    case TRIAL_EXPIRED, SUBSCRIPTION_PAST_DUE ->
      bodySecondPart = resourceBundleService.getSharedString("Notification.body.subscription.expiration.manage", locale);
    case HUB_USERS_LIMIT_EXCEEDED ->
      bodySecondPart = resourceBundleService.getSharedString("Notification.body.secondPart.hub.maxOfUsersExceeded", locale);
    case HUB_PLAN_CHANGED ->
      bodySecondPart = resourceBundleService.getSharedString("Notification.body.secondPart.planChanged", locale);
      case SUBSCRIPTION_CANCELED -> 
      bodySecondPart = resourceBundleService.getSharedString("Notification.body.secondPart.subscription.cancelled", locale);
    }
    return bodySecondPart.replace("{0}", linkPart);
  }

  private String getSenderFullNameLink(String username, String userFullName) {
    String profileLink = LinkProviderUtils.getRedirectUrl("user", username);
    return "<strong><a target=\"_blank\" style=\"color: #2f5e92; text-decoration: none; font-family: 'HelveticaNeue Bold', Helvetica, Arial, sans-serif\" href=\""
        + profileLink + "\">" + userFullName + "</a></strong>";
  }

  private String getUserLang(String userName) throws Exception {
    UserProfile userProfile = organizationService.getUserProfileHandler().findUserProfileByName(userName);
    if (userProfile != null && userProfile.getAttribute(Constants.USER_LANGUAGE) != null) {
      return userProfile.getAttribute(Constants.USER_LANGUAGE);
    } else {
      return ResourceBundleService.DEFAULT_CROWDIN_LANGUAGE;
    }
  }

  private String replaceI18nKey(String content, String lang) {
    Matcher matcher = I18N_LABEL_PATTERN.matcher(content);
    while (matcher.find()) {
      String i18nKey = matcher.group(1);
      String label = resourceBundleService.getSharedString(i18nKey, LocaleUtils.toLocale(lang));
      content = content.replace(matcher.group(), label);
    }
    return content;
  }

  private String getSpaceFullLink(long spaceId, String applicationUri, String i18nKey, String lang) {
    PermanentLinkObject object = new PermanentLinkObject(SpacePermanentLinkPlugin.OBJECT_TYPE, String.valueOf(spaceId));
    if (applicationUri != null) {
      object.addParameter(SpacePermanentLinkPlugin.APPLICATION_URI, applicationUri);
    }

    String spaceLink = CommonsUtils.getCurrentDomain() + permanentLinkService.getPermanentLink(object);
    String linkLabel = i18nKey != null ? resourceBundleService.getSharedString(i18nKey, LocaleUtils.toLocale(lang))
                                       : Utils.getSpaceDisplayName(spaceId);
    return new StringBuilder().append("<strong><a target=\"_blank\" ")
                              .append("style=\"color: #2f5e92; text-decoration: none; font-family: 'HelveticaNeue Bold', Helvetica, Arial, sans-serif\" ")
                              .append("href=\"")
                              .append(spaceLink)
                              .append("\">")
                              .append(linkLabel)
                              .append("</a></strong>")
                              .toString();
  }
}
