/*
 * This file is part of the Meeds project (https://meeds.io/).
 * 
 * Copyright (C) 2020 - 2021 Meeds Association contact@meeds.io
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.exoplatform.social.core.jpa.storage.dao.jpa;

import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import org.exoplatform.commons.persistence.impl.GenericDAOJPAImpl;
import org.exoplatform.social.core.jpa.storage.entity.MetadataEntity;

import jakarta.persistence.NoResultException;
import jakarta.persistence.TypedQuery;

public class MetadataDAO extends GenericDAOJPAImpl<MetadataEntity, Long> {

  private static final String AUDIENCE_IDS       = "audienceIds";

  private static final String CREATOR_ID         = "creatorId";

  private static final String METADATA_TYPE      = "type";

  private static final String METADATA_NAME_PART = "term";

  public List<String> getMetadataNamesByAudiences(long metadataTypeId,
                                                  Set<Long> audienceIds,
                                                  long limit) {
    if (CollectionUtils.isEmpty(audienceIds)) {
      return Collections.emptyList();
    }
    TypedQuery<String> query = getEntityManager().createNamedQuery("SocMetadataEntity.getMetadataNamesByAudiences",
                                                                   String.class);
    query.setParameter(METADATA_TYPE, metadataTypeId);
    query.setParameter(AUDIENCE_IDS, audienceIds);
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    List<String> result = query.getResultList();
    if (CollectionUtils.isEmpty(result)) {
      return Collections.emptyList();
    } else {
      return result.stream().distinct().collect(Collectors.toList());
    }
  }

  public List<String> getMetadataNamesByCreator(long metadataTypeId,
                                                long creatorIdentityId,
                                                long limit) {
    TypedQuery<String> query = getEntityManager().createNamedQuery("SocMetadataEntity.getMetadataNamesByCreator",
                                                                   String.class);
    query.setParameter(METADATA_TYPE, metadataTypeId);
    query.setParameter(CREATOR_ID, creatorIdentityId);
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    List<String> result = query.getResultList();
    if (CollectionUtils.isEmpty(result)) {
      return Collections.emptyList();
    } else {
      return result.stream().distinct().collect(Collectors.toList());
    }
  }

  public List<String> getMetadataNamesByUser(long metadataTypeId,
                                             long creatorIdentityId,
                                             Set<Long> audienceIds,
                                             long limit) {
    TypedQuery<String> query = getEntityManager().createNamedQuery("SocMetadataEntity.getMetadataNamesByUser",
                                                                   String.class);
    query.setParameter(METADATA_TYPE, metadataTypeId);
    query.setParameter(AUDIENCE_IDS, audienceIds);
    query.setParameter(CREATOR_ID, creatorIdentityId);
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    List<String> result = query.getResultList();
    if (CollectionUtils.isEmpty(result)) {
      return Collections.emptyList();
    } else {
      return result.stream().distinct().collect(Collectors.toList());
    }
  }

  public List<String> findMetadataNameByAudiencesAndQuery(String term,
                                                          long metadataTypeId,
                                                          Set<Long> audienceIds,
                                                          long limit) {
    if (CollectionUtils.isEmpty(audienceIds)) {
      return Collections.emptyList();
    }
    TypedQuery<String> query = getEntityManager().createNamedQuery("SocMetadataEntity.findMetadataNameByAudiencesAndQuery",
                                                                   String.class);
    query.setParameter(METADATA_TYPE, metadataTypeId);
    query.setParameter(AUDIENCE_IDS, audienceIds);
    query.setParameter(METADATA_NAME_PART, "%" + StringUtils.lowerCase(term) + "%");
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    List<String> result = query.getResultList();
    if (CollectionUtils.isEmpty(result)) {
      return Collections.emptyList();
    } else {
      return result.stream().distinct().collect(Collectors.toList());
    }
  }

  public List<String> findMetadataNameByCreatorAndQuery(String term, long metadataTypeId, long creatorIdentityId, long limit) {
    TypedQuery<String> query = getEntityManager().createNamedQuery("SocMetadataEntity.findMetadataNameByCreatorAndQuery",
                                                                   String.class);
    query.setParameter(METADATA_TYPE, metadataTypeId);
    query.setParameter(CREATOR_ID, creatorIdentityId);
    query.setParameter(METADATA_NAME_PART, "%" + StringUtils.lowerCase(term) + "%");
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    List<String> result = query.getResultList();
    if (CollectionUtils.isEmpty(result)) {
      return Collections.emptyList();
    } else {
      return result.stream().distinct().collect(Collectors.toList());
    }
  }

  public List<String> findMetadataNamesByUserAndQuery(String term,
                                                      long metadataTypeId,
                                                      long creatorIdentityId,
                                                      Set<Long> audienceIds,
                                                      long limit) {
    TypedQuery<String> query = getEntityManager().createNamedQuery("SocMetadataEntity.findMetadataNameByUserAndQuery",
                                                                   String.class);
    query.setParameter(METADATA_TYPE, metadataTypeId);
    query.setParameter(AUDIENCE_IDS, audienceIds);
    query.setParameter(CREATOR_ID, creatorIdentityId);
    query.setParameter(METADATA_NAME_PART, "%" + StringUtils.lowerCase(term) + "%");
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    List<String> result = query.getResultList();
    if (CollectionUtils.isEmpty(result)) {
      return Collections.emptyList();
    } else {
      return result.stream().distinct().collect(Collectors.toList());
    }
  }

  public MetadataEntity findMetadata(long type, String name, long audienceId) {
    TypedQuery<MetadataEntity> query = getEntityManager().createNamedQuery("SocMetadataEntity.findMetadata",
                                                                           MetadataEntity.class);
    query.setParameter("type", type);
    query.setParameter("name", name);
    query.setParameter("audienceId", audienceId);
    try {
      return query.getSingleResult();
    } catch (NoResultException e) {
      return null;
    }
  }

  public List<MetadataEntity> getMetadatas(long type, long limit) {
    TypedQuery<MetadataEntity> query = getEntityManager().createNamedQuery("SocMetadataEntity.getMetadatas",
                                                                           MetadataEntity.class);
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    query.setParameter("type", type);
    try {
      return query.getResultList();
    } catch (NoResultException e) {
      return Collections.emptyList();
    }
  }

  public List<Long> getMetadataIds(long type, int offset, int limit) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("SocMetadataEntity.getMetadataIds", Long.class);
    if (limit > 0) {
      query.setMaxResults(limit);
    }
    if (offset > 0) {
      query.setFirstResult(offset);
    }
    query.setParameter("type", type);
    try {
      return query.getResultList();
    } catch (NoResultException e) {
      return Collections.emptyList();
    }
  }

  public List<Long> getMetadataIdsByProperty(String propertyKey,
                                             String propertyValue,
                                             long offset,
                                             long limit,
                                             boolean orderByName) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery(orderByName ?
                                                                             "SocMetadataEntity.getMetadatasByPropertyOrderByName" :
                                                                             "SocMetadataEntity.getMetadatasByPropertyOrderById",
                                                                 Long.class);
    if (limit > 0) {
      query.setMaxResults((int) limit);
    }
    if (offset > 0) {
      query.setFirstResult((int) offset);
    }
    query.setParameter("propertyName", propertyKey);
    query.setParameter("propertyValue", propertyValue);
    return query.getResultList();
  }

  public long countMetadataIdsByProperty(String propertyKey, String propertyValue) {
    TypedQuery<Long> query = getEntityManager().createNamedQuery("SocMetadataEntity.countMetadataIdsByProperty", Long.class);
    query.setParameter("propertyName", propertyKey);
    query.setParameter("propertyValue", propertyValue);
    try {
      Long result = query.getSingleResult();
      return result == null ? 0l : result.longValue();
    } catch (NoResultException e) {
      return 0l;
    }
  }

}
