/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.services.organization.api;

import junit.framework.TestCase;

import org.exoplatform.commons.utils.ListAccess;
import org.exoplatform.container.StandaloneContainer;
import org.exoplatform.services.organization.BaseOrganizationService;
import org.exoplatform.services.organization.DisabledUserException;
import org.exoplatform.services.organization.User;
import org.exoplatform.services.organization.UserHandler;
import org.exoplatform.services.organization.UserStatus;
import org.exoplatform.services.security.ConversationRegistry;

import java.net.URL;

public class TestUserHandler extends TestCase
{

   protected ConversationRegistry registry;

   private BaseOrganizationService organizationService;

   private UserHandler uHandler;

   protected void setUp() throws Exception
   {
      super.setUp();

      if (registry == null)
      {
         URL containerConfURL =
                              TestUserHandler.class.getResource("/conf/standalone/test-configuration.xml");
         assertNotNull(containerConfURL);

         String containerConf = containerConfURL.toString();
         StandaloneContainer.addConfigurationURL(containerConf);
         StandaloneContainer container = StandaloneContainer.getInstance();

         organizationService =
            (BaseOrganizationService)container
               .getComponentInstance(org.exoplatform.services.organization.OrganizationService.class);
         assertNotNull(organizationService);

         uHandler = organizationService.getUserHandler();

         registry = container.getComponentInstanceOfType(ConversationRegistry.class);
         assertNotNull(registry);
      }
   }

   /**
    * Authenticate users.
    */
   public void testAuthenticate()
   {
      try
      {
         assertTrue(uHandler.authenticate("demo", "exo"));
         assertFalse(uHandler.authenticate("demo", "exo_"));
         assertFalse(uHandler.authenticate("_demo_", "exo"));

      }
      catch (Exception e)
      {
         e.printStackTrace();
         fail("Exception should not be thrown.");
      }
   }


   public void testUserEnabling() throws Exception
   {
      ListAccess<User> users = uHandler.findAllUsers();
      assertTrue(contains(users, "demo"));
      users = uHandler.findAllUsers(UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByGroupId("/platform/users");
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByGroupId("/platform/users", UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByQuery(null);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByQuery(null, UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      assertNotNull(uHandler.findUserByName("demo"));
      assertNotNull(uHandler.findUserByName("demo", UserStatus.ANY));
      assertTrue(uHandler.authenticate("demo", "exo"));

      uHandler.setEnabled("demo", false, true);
      users = uHandler.findAllUsers();
      assertFalse(contains(users, "demo"));
      users = uHandler.findAllUsers(UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByGroupId("/platform/users");
      assertFalse(contains(users, "demo"));
      users = uHandler.findUsersByGroupId("/platform/users", UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByQuery(null);
      assertFalse(contains(users, "demo"));
      users = uHandler.findUsersByQuery(null, UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      assertNull(uHandler.findUserByName("demo"));
      assertNotNull(uHandler.findUserByName("demo", UserStatus.ANY));
      try
      {
         uHandler.authenticate("demo", "exo");
         fail("A DisabledUserException was epected");
      }
      catch (DisabledUserException e)
      {
         // expected
      }

      uHandler.setEnabled("demo", true, true);
      users = uHandler.findAllUsers();
      assertTrue(contains(users, "demo"));
      users = uHandler.findAllUsers(UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByGroupId("/platform/users");
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByGroupId("/platform/users", UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByQuery(null);
      assertTrue(contains(users, "demo"));
      users = uHandler.findUsersByQuery(null, UserStatus.ANY);
      assertTrue(contains(users, "demo"));
      assertNotNull(uHandler.findUserByName("demo"));
      assertNotNull(uHandler.findUserByName("demo", UserStatus.ANY));
      assertTrue(uHandler.authenticate("demo", "exo"));
   }

   private static boolean contains(ListAccess<User> users, String username) throws Exception
   {
      User[] aUsers = users.load(0, users.getSize());
      for (User user : aUsers)
      {
         if (user.getUserName().equals(username))
            return true;
      }
      return false;
   }
}
