/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.social.service.rest;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.exoplatform.container.PortalContainer;
import org.exoplatform.social.service.test.AbstractServiceTest;

import static org.exoplatform.social.service.rest.RestChecker.checkAuthenticatedRequest;
import static org.exoplatform.social.service.rest.RestChecker.checkSupportedFormat;
import static org.exoplatform.social.service.rest.RestChecker.checkValidPortalContainerName;

/**
 * Unit test for {@link RestChecker}.
 *
 */
public class RestCheckerTest extends AbstractServiceTest {

  /**
   * Tests {@link RestChecker#checkAuthenticatedRequest()}.
   */
  public void testCheckAuthenticatedRequest() {
    try {
      checkAuthenticatedRequest();
      fail("Expecting WebApplicationException");
    } catch (WebApplicationException wae) {
      assertEquals("wae.getResponse().getStatus() must return: " + Response.Status.UNAUTHORIZED.getStatusCode(),
                   Response.Status.UNAUTHORIZED.getStatusCode(), wae.getResponse().getStatus());
    }

    startSessionAs("demo");
    checkAuthenticatedRequest();
  }

  /**
   * Tests {@link RestChecker#checkValidPortalContainerName(String)}.
   */
  public void testCheckValidPortalContainerName() {
    try {
      checkValidPortalContainerName("wrong");
      fail("Expecting WebApplicationException");
    } catch (WebApplicationException wae) {
      assertEquals("wae.getResponse().getStatus() must return: " + Response.Status.BAD_REQUEST.getStatusCode(),
                   Response.Status.BAD_REQUEST.getStatusCode(), wae.getResponse().getStatus());
    }

    PortalContainer portalContainer = checkValidPortalContainerName("portal");
    assertNotNull("portalContainer must not be null", portalContainer);
  }

  /**
   * Tests {@link RestChecker#checkSupportedFormat(String, String[])}.
   */
  public void testCheckSupportedFormat() {
    String[] supportedFormats = new String[]{"json", "xml"};
    try {
      checkSupportedFormat("atom", supportedFormats);
      fail("Expecting WebApplicationException");
    } catch (WebApplicationException wae) {
      assertEquals("wae.getResponse().getStatus() must return: "+Response.Status.UNSUPPORTED_MEDIA_TYPE.getStatusCode(),
                   Response.Status.UNSUPPORTED_MEDIA_TYPE.getStatusCode(), wae.getResponse().getStatus());
    }

    MediaType jsonMediaType = checkSupportedFormat("json", supportedFormats);
    assertNotNull("jsonMediaType must not be null", jsonMediaType);
    assertEquals("jsonMediaType.getType() must return: " + MediaType.APPLICATION_JSON_TYPE.getType(),
            MediaType.APPLICATION_JSON_TYPE.getType(), jsonMediaType.getType());
    assertEquals("jsonMediaType.getSubtype() must return: " + MediaType.APPLICATION_JSON_TYPE.getSubtype(),
                 MediaType.APPLICATION_JSON_TYPE.getSubtype(), jsonMediaType.getSubtype());

    MediaType xmlMediaType = checkSupportedFormat("xml", supportedFormats);
    assertNotNull("xmlMediaType must not be null", xmlMediaType);
    assertEquals("xmlMediaType.getType() must return: " + MediaType.APPLICATION_XML_TYPE.getType(),
                 MediaType.APPLICATION_XML_TYPE.getType(), xmlMediaType.getType());
    assertEquals("xmlMediaType.getSubtype() must return: " + MediaType.APPLICATION_XML_TYPE.getSubtype(),
                 MediaType.APPLICATION_XML_TYPE.getSubtype(), xmlMediaType.getSubtype());
  }

}
