/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.social.common.xmlprocessor.filters;

import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.social.common.xmlprocessor.BaseXMLFilterPlugin;
import org.exoplatform.social.common.xmlprocessor.DOMParser;
import org.exoplatform.social.common.xmlprocessor.Tokenizer;
import org.exoplatform.social.common.xmlprocessor.model.Node;

import java.util.LinkedList;
import java.util.List;

/**
 * This OpenLinkNewTabFilterPlugin is a plugin for XMLProcessor which will force all links have to be open in new tab.
 * It will set attribute target="_blank" if this attribute is omitted or had other value.
 * For example:
 * {@literal <a href="http://abc.com">abc</a> => <a href="http://abc.com" target="_blank">abc</a>}
 * {@literal <a href="http://abc.com" target="frame_name">abc</a> => <a href="http://abc.com" target="_blank">abc</a>}
 *
 */
public class OpenLinkNewTabFilterPlugin extends BaseXMLFilterPlugin {

  public static final String TARGET = "target";
  @Override
  public Object doFilter(Object input) {
    if (input instanceof String) {
      return filterLinkTags((String)input);
    }
    return input;
  }

  private String filterLinkTags(String xmlString){
    List<String> xmlTokens = Tokenizer.tokenize(xmlString);
    Node rootNode = DOMParser.createDOMTree(xmlTokens);
    nodeFilter(rootNode);
    return rootNode.toString();
  }

  private void nodeFilter(Node currentNode) {
    if ("a".equalsIgnoreCase(currentNode.getTitle())) {
      String currentDomain = CommonsUtils.getCurrentDomain();
      String target = currentNode.getAttributes().get(TARGET);
      if (target != null && currentNode.getAttributes().get("href").indexOf(currentDomain) == -1 ) {
        currentNode.getAttributes().put(TARGET, "_blank");
      } else {
        currentNode.getAttributes().put(TARGET, "_self");
      }
      return;
    }
    LinkedList<Node> currentChildNode = currentNode.getChildNodes();
    for (int i = 0; i < currentChildNode.size(); i++) {
      nodeFilter(currentChildNode.get(i));
    }
  }
}
