/*
 * Decompiled with CFR 0.152.
 */
package org.exoplatform.upload;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpSession;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import org.apache.commons.fileupload2.core.DiskFileItem;
import org.apache.commons.fileupload2.core.DiskFileItemFactory;
import org.apache.commons.fileupload2.core.FileUploadException;
import org.apache.commons.fileupload2.core.ProgressListener;
import org.apache.commons.fileupload2.jakarta.JakartaServletDiskFileUpload;
import org.exoplatform.container.xml.InitParams;
import org.exoplatform.container.xml.PortalContainerInfo;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;
import org.exoplatform.upload.MimeTypeUploadPlugin;
import org.exoplatform.upload.RequestStreamReader;
import org.exoplatform.upload.UploadResource;

public class UploadService {
    private static final Log log = ExoLogger.getLogger(UploadService.class);
    private static final int[] illegalChars = new int[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 34, 42, 47, 58, 60, 62, 63, 92, 124};
    private static final Pattern UPLOAD_ID_PATTERN = Pattern.compile("[^(\\./\\\\)]*");
    private List<MimeTypeUploadPlugin> plugins;
    private Map<String, UploadResource> uploadResources = new LinkedHashMap<String, UploadResource>();
    private String uploadLocation_;
    private UploadLimit defaultUploadLimitMB_;
    private Map<String, UploadLimit> uploadLimits = new LinkedHashMap<String, UploadLimit>();
    public static final String UPLOAD_RESOURCES_STACK = "uploadResourcesStack";

    public UploadService(PortalContainerInfo pinfo, InitParams params) throws Exception {
        String tmpDir = System.getProperty("java.io.tmpdir");
        this.defaultUploadLimitMB_ = params == null || params.getValueParam("upload.limit.size") == null ? new UploadLimit(0, UploadUnit.MB) : new UploadLimit(Integer.parseInt(params.getValueParam("upload.limit.size").getValue()), UploadUnit.MB);
        this.uploadLocation_ = tmpDir + "/" + pinfo.getContainerName() + "/eXoUpload";
    }

    public void register(MimeTypeUploadPlugin plugin) {
        if (this.plugins == null) {
            this.plugins = new ArrayList<MimeTypeUploadPlugin>();
        }
        this.plugins.add(plugin);
    }

    public void createUploadResource(HttpServletRequest request) throws FileUploadException {
        String uploadId = request.getParameter("uploadId");
        this.createUploadResource(uploadId, request);
    }

    public void createUploadResource(String uploadId, HttpServletRequest request) throws FileUploadException {
        if (uploadId == null || !UPLOAD_ID_PATTERN.matcher(uploadId).matches()) {
            throw new FileUploadException("Upload id " + uploadId + " is not valid, it cannot be null or contain '.' , '/' or '\\'");
        }
        UploadResource upResource = new UploadResource(uploadId);
        upResource.setFileName("");
        upResource.setStoreLocation(this.uploadLocation_ + File.separator + uploadId);
        this.uploadResources.put(upResource.getUploadId(), upResource);
        this.putToStackInSession(request.getSession(true), uploadId);
        double requestContentLength = request.getContentLength();
        upResource.setEstimatedSize(requestContentLength);
        if (this.isLimited(upResource, requestContentLength)) {
            upResource.setStatus(2);
            return;
        }
        File uploadRootPath = new File(this.uploadLocation_);
        if (!uploadRootPath.exists()) {
            uploadRootPath.mkdirs();
        }
        DiskFileItem fileItem = this.getFileItem(request, upResource);
        upResource.setFileName(fileItem.getName());
        upResource.setMimeType(fileItem.getContentType());
        if (upResource.getStatus() == 1) {
            this.writeFile(upResource, fileItem);
        }
        if (this.plugins != null) {
            for (MimeTypeUploadPlugin plugin : this.plugins) {
                String mimeType = plugin.getMimeType(fileItem.getName());
                if (mimeType == null) continue;
                upResource.setMimeType(mimeType);
            }
        }
    }

    public void createUploadResource(String uploadId, String encoding, String contentType, double contentLength, InputStream inputStream) throws Exception {
        if (uploadId == null || !UPLOAD_ID_PATTERN.matcher(uploadId).matches()) {
            throw new FileUploadException("Upload id can contain only digits and hyphens");
        }
        File uploadDir = new File(this.uploadLocation_);
        if (!uploadDir.exists()) {
            uploadDir.mkdirs();
        }
        UploadResource upResource = new UploadResource(uploadId);
        RequestStreamReader reader = new RequestStreamReader(upResource);
        this.uploadResources.put(upResource.getUploadId(), upResource);
        if (this.isLimited(upResource, contentLength)) {
            upResource.setStatus(2);
            return;
        }
        Map<String, String> headers = reader.parseHeaders(inputStream, encoding);
        String fileName = reader.getFileName(headers);
        if (fileName == null) {
            fileName = uploadId;
        }
        fileName = fileName.substring(fileName.lastIndexOf(92) + 1);
        upResource.setFileName(fileName);
        upResource.setMimeType(headers.get("content-type"));
        upResource.setStoreLocation(this.uploadLocation_ + "/" + uploadId + "." + fileName);
        upResource.setEstimatedSize(contentLength);
        File fileStore = new File(upResource.getStoreLocation());
        if (!fileStore.exists()) {
            fileStore.createNewFile();
        }
        FileOutputStream output = new FileOutputStream(fileStore);
        reader.readBodyData(inputStream, contentType, output);
        if (upResource.getStatus() == 0) {
            upResource.setStatus(1);
            return;
        }
        this.uploadResources.remove(uploadId);
        fileStore.delete();
    }

    private void putToStackInSession(HttpSession session, String uploadId) {
        HashSet<String> uploadResouceIds = (HashSet<String>)session.getAttribute(UPLOAD_RESOURCES_STACK);
        if (uploadResouceIds == null) {
            uploadResouceIds = new HashSet<String>();
        }
        uploadResouceIds.add(uploadId);
        session.setAttribute(UPLOAD_RESOURCES_STACK, uploadResouceIds);
    }

    public UploadResource getUploadResource(String uploadId) {
        return this.uploadResources.get(uploadId);
    }

    public void cleanUp(HttpSession session) {
        log.debug((Object)"Cleaning up uploaded files for temporariness");
        Set uploadIds = (Set)session.getAttribute(UPLOAD_RESOURCES_STACK);
        if (uploadIds != null) {
            for (String id : uploadIds) {
                this.removeUploadResource(id);
                this.uploadLimits.remove(id);
            }
        }
    }

    @Deprecated
    public void removeUpload(String uploadId) {
        this.removeUploadResource(uploadId);
    }

    public void removeUploadResource(String uploadId) {
        if (uploadId == null) {
            return;
        }
        UploadResource upResource = this.uploadResources.get(uploadId);
        if (upResource != null) {
            this.uploadResources.remove(uploadId);
            if (upResource.getStoreLocation() != null) {
                File file = new File(upResource.getStoreLocation());
                file.delete();
            }
        }
    }

    public void addUploadLimit(String uploadId, Integer limitMB) {
        this.addUploadLimit(uploadId, limitMB, UploadUnit.MB);
    }

    public void addUploadLimit(String uploadId, Integer limit, UploadUnit unit) {
        if (limit == null) {
            this.uploadLimits.put(uploadId, this.defaultUploadLimitMB_);
        } else if (unit == null) {
            this.uploadLimits.put(uploadId, new UploadLimit(limit, UploadUnit.MB));
        } else {
            this.uploadLimits.put(uploadId, new UploadLimit(limit, unit));
        }
    }

    public void removeUploadLimit(String uploadId) {
        this.uploadLimits.remove(uploadId);
    }

    public Map<String, UploadLimit> getUploadLimits() {
        return this.uploadLimits;
    }

    public String correctFileName(String fileName) {
        if (fileName == null || fileName.isEmpty()) {
            return "NULL";
        }
        char[] chars = fileName.toCharArray();
        for (int i = 0; i < chars.length; ++i) {
            if (Arrays.binarySearch(illegalChars, (int)chars[i]) < 0) continue;
            chars[i] = 95;
        }
        return new String(chars);
    }

    public boolean isLimited(UploadResource upResource, double contentLength) {
        UploadLimit limit = this.defaultUploadLimitMB_;
        if (this.uploadLimits.containsKey(upResource.getUploadId())) {
            limit = this.uploadLimits.get(upResource.getUploadId());
        }
        double estimatedSize = contentLength / (double)limit.division;
        if (limit.getLimit() > 0 && estimatedSize > (double)limit.getLimit()) {
            if (log.isDebugEnabled()) {
                log.debug((Object)("Upload cancelled because file bigger than size limit : " + estimatedSize + " " + limit.unit + " > " + limit.getLimit() + " " + limit.unit));
            }
            return true;
        }
        return false;
    }

    public UploadLimit getLimitForResource(UploadResource upResource) {
        UploadLimit limit = this.defaultUploadLimitMB_;
        if (this.uploadLimits.containsKey(upResource.getUploadId())) {
            limit = this.uploadLimits.get(upResource.getUploadId());
        }
        return limit;
    }

    private DiskFileItem getFileItem(HttpServletRequest request, UploadResource upResource) throws FileUploadException {
        ArrayList<DiskFileItem> itemList = new ArrayList<DiskFileItem>();
        DiskFileItemFactory factory = ((DiskFileItemFactory.Builder)((DiskFileItemFactory.Builder)((DiskFileItemFactory.Builder)DiskFileItemFactory.builder().setBufferSize(0)).setFile(this.uploadLocation_)).setCharset(StandardCharsets.UTF_8)).get();
        JakartaServletDiskFileUpload servletUpload = new JakartaServletDiskFileUpload(factory);
        servletUpload.setHeaderCharset(StandardCharsets.UTF_8);
        ProgressListener listener = (bytesRead, contentLength, items) -> {
            if (bytesRead == contentLength) {
                if (itemList.size() == 1) {
                    this.writeFile(upResource, (DiskFileItem)itemList.get(items));
                }
                upResource.addUploadedBytes((double)contentLength - upResource.getUploadedSize());
                upResource.setStatus(1);
            } else {
                upResource.addUploadedBytes((double)bytesRead - upResource.getUploadedSize());
            }
        };
        servletUpload.setProgressListener(listener);
        List fileItems = servletUpload.parseRequest(request);
        if (fileItems == null || fileItems.size() != 1 || ((DiskFileItem)fileItems.get(0)).isFormField()) {
            this.removeUploadResource(upResource.getUploadId());
            throw new FileUploadException("You can upload 1 file per request");
        }
        DiskFileItem fileItem = (DiskFileItem)fileItems.get(0);
        itemList.add(fileItem);
        return fileItem;
    }

    private void writeFile(UploadResource upResource, DiskFileItem diskFileItem) {
        try {
            diskFileItem.write(Paths.get(upResource.getStoreLocation(), new String[0]));
        }
        catch (IOException e) {
            throw new IllegalStateException("Upload id " + upResource.getUploadId() + " can't be uploaded", e);
        }
    }

    public static class UploadLimit {
        private int limit;
        private int division;
        private UploadUnit unit;

        public UploadLimit(int limit, UploadUnit unit) {
            this.limit = limit;
            this.unit = unit;
            if (unit == UploadUnit.KB) {
                this.division = 1024;
            } else if (unit == UploadUnit.MB) {
                this.division = 0x100000;
            } else if (unit == UploadUnit.GB) {
                this.division = 0x40000000;
            }
        }

        public int getLimit() {
            return this.limit;
        }

        public String getUnit() {
            return this.unit.toString();
        }
    }

    public static enum UploadUnit {
        KB,
        MB,
        GB;

    }
}

