/*
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2024 Meeds Lab contact@meedslab.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package io.meeds.crowdin.gamification.storage;

import static io.meeds.crowdin.gamification.utils.Utils.APPROVALS;
import static io.meeds.crowdin.gamification.utils.Utils.AUTHORIZATION;
import static io.meeds.crowdin.gamification.utils.Utils.COMMENT_CREATED_TRIGGER;
import static io.meeds.crowdin.gamification.utils.Utils.COMMENT_DELETED_TRIGGER;
import static io.meeds.crowdin.gamification.utils.Utils.CROWDIN_API_URL;
import static io.meeds.crowdin.gamification.utils.Utils.CROWDIN_CONNECTION_ERROR;
import static io.meeds.crowdin.gamification.utils.Utils.CROWDIN_EVENTS;
import static io.meeds.crowdin.gamification.utils.Utils.PROJECTS;
import static io.meeds.crowdin.gamification.utils.Utils.TOKEN;
import static io.meeds.crowdin.gamification.utils.Utils.WEBHOOKS;
import static io.meeds.crowdin.gamification.utils.Utils.generateRandomSecret;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.MediaType;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HTTP;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import org.exoplatform.commons.utils.CommonsUtils;
import org.exoplatform.services.log.ExoLogger;
import org.exoplatform.services.log.Log;

import io.meeds.crowdin.gamification.exception.CrowdinConnectionException;
import io.meeds.crowdin.gamification.model.RemoteApproval;
import io.meeds.crowdin.gamification.model.RemoteDirectory;
import io.meeds.crowdin.gamification.model.RemoteLanguage;
import io.meeds.crowdin.gamification.model.RemoteProject;
import io.meeds.crowdin.gamification.model.WebHook;

@Component
public class CrowdinConsumerStorage {

  public static final String TOKEN_EXPIRED_OR_INVALID = "crowdin.tokenExpiredOrInvalid";

  private HttpClient         client;

  private static final Log   LOG                      = ExoLogger.getLogger(CrowdinConsumerStorage.class);

  public List<RemoteProject> getProjects(String accessToken) throws IllegalAccessException {
    try {

      URI uri = URI.create(CROWDIN_API_URL + PROJECTS + "?hasManagerAccess=1");
      String response = processGet(uri, accessToken);
      JSONArray jsonArray = new JSONObject(response).getJSONArray("data");

      List<RemoteProject> projects = new ArrayList<>();

      // loop through the array and parse the projects
      for (int i = 0; i < jsonArray.length(); i++) {
        JSONObject jsonObjectData = jsonArray.getJSONObject(i);
        JSONObject jsonObject = jsonObjectData.getJSONObject("data");
        // parse the project
        RemoteProject project = new RemoteProject();
        project.setId(jsonObject.getInt("id"));
        project.setName(jsonObject.getString("name"));
        project.setIdentifier(jsonObject.getString("identifier"));

        projects.add(project);
      }
      return projects;
    } catch (IllegalArgumentException e) {
      throw new IllegalArgumentException(e);
    } catch (CrowdinConnectionException e) {
      throw new IllegalAccessException(TOKEN_EXPIRED_OR_INVALID);
    }
  }

  public WebHook createWebhook(long projectId, String[] triggers, String accessToken) throws IllegalAccessException {
    try {
      URI uri = URI.create(CROWDIN_API_URL + PROJECTS + projectId + "/webhooks");
      String secret = generateRandomSecret(8);
      JSONArray events = new JSONArray();
      Arrays.stream(triggers).forEach(events::put);

      JSONObject requestJson = new JSONObject();
      requestJson.put("name", "Meeds");
      requestJson.put("url", CommonsUtils.getCurrentDomain() + "/gamification-crowdin/rest/crowdin/webhooks");
      requestJson.put("events", events);
      requestJson.put("requestType", "POST");
      requestJson.put("isActive", true);
      requestJson.put("batchingEnabled", true);
      requestJson.put("contentType", "application/json");

      JSONObject headers = new JSONObject();
      headers.put("Authorization", "Bearer " + secret);
      requestJson.put("headers", headers);

      Map<String, Object> fileMap = new HashMap<>();
      fileMap.put("id", "{{fileId}}");
      fileMap.put("directoryId", "{{directoryId}}");

      Map<String, Object> projectMap = new HashMap<>();
      projectMap.put("id", "{{projectId}}");
      projectMap.put("sourceLanguageId", "{{projectSourceLanguageId}}");
      projectMap.put("identifier", "{{projectIdentifier}}");

      Map<String, Object> stringMap = new HashMap<>();
      stringMap.put("id", "{{stringId}}");
      stringMap.put("text", "{{stringText}}");
      stringMap.put("file", fileMap);
      stringMap.put("project", projectMap);

      Map<String, Object> userMap = new HashMap<>();
      userMap.put("id", "{{userId}}");
      userMap.put("username", "{{userUsername}}");
      userMap.put("fullName", "{{userFullName}}");
      userMap.put("avatarUrl", "{{userAvatarUrl}}");

      Map<String, Object> languageMap = new HashMap<>();
      languageMap.put("id", "{{targetLanguageId}}");

      Map<String, Object> translationMap = new HashMap<>();
      translationMap.put("id", "{{translationId}}");
      translationMap.put("provider", "{{translationProvider}}");
      translationMap.put("targetLanguage", languageMap);
      translationMap.put("user", userMap);
      translationMap.put("string", stringMap);

      Map<String, Object> commentMap = new HashMap<>();
      commentMap.put("id", "{{commentId}}");
      commentMap.put("targetLanguage", languageMap);
      commentMap.put("user", userMap);
      commentMap.put("string", stringMap);

      Map<String, Object> translationEventMap = new HashMap<>();
      translationEventMap.put("event", "{{event}}");
      translationEventMap.put("translation", translationMap);

      Map<String, Object> commentEventMap = new HashMap<>();
      commentEventMap.put("event", "{{event}}");
      commentEventMap.put("comment", commentMap);

      Map<String, Object> payload = new HashMap<>();

      Arrays.stream(CROWDIN_EVENTS).forEach(event -> {
        if (List.of(COMMENT_CREATED_TRIGGER, COMMENT_DELETED_TRIGGER).contains(event)) {
          payload.put(event, commentEventMap);
        } else {
          payload.put(event, translationEventMap);
        }
      });

      requestJson.put("payload", payload);

      String response = processPost(uri, requestJson.toString(), accessToken);

      JSONObject responseJson = new JSONObject(response);

      JSONObject dataJson = responseJson.getJSONObject("data");

      WebHook localWebHook = new WebHook();
      localWebHook.setWebhookId(dataJson.getLong("id"));
      localWebHook.setProjectId(dataJson.getLong("projectId"));
      localWebHook.setTriggers(List.of(triggers));
      localWebHook.setToken(accessToken);
      localWebHook.setSecret(secret);
      return localWebHook;

    } catch (IllegalArgumentException e) {
      LOG.error(e);
      throw new IllegalArgumentException(e);
    } catch (CrowdinConnectionException e) {
      LOG.error(e);
      throw new IllegalAccessException(TOKEN_EXPIRED_OR_INVALID);
    }
  }

  private String processGet(URI uri, String accessToken) throws CrowdinConnectionException {
    HttpClient httpClient = getHttpClient();
    HttpGet request = new HttpGet(uri);
    try {
      request.setHeader(AUTHORIZATION, TOKEN + accessToken);
      return processRequest(httpClient, request);
    } catch (IOException e) {
      throw new CrowdinConnectionException(CROWDIN_CONNECTION_ERROR, e);
    }
  }

  private String processPost(URI uri, String jsonString, String accessToken) throws CrowdinConnectionException {
    HttpClient httpClient = getHttpClient();
    HttpPost request = new HttpPost(uri);
    StringEntity entity = new StringEntity(jsonString, org.apache.http.entity.ContentType.APPLICATION_JSON);
    try {
      request.setHeader(HTTP.CONTENT_TYPE, MediaType.APPLICATION_JSON);
      request.setHeader(AUTHORIZATION, TOKEN + accessToken);
      request.setEntity(entity);
      return processRequest(httpClient, request);
    } catch (IOException e) {
      throw new CrowdinConnectionException(CROWDIN_CONNECTION_ERROR, e);
    }
  }

  private String processDelete(URI uri, String accessToken) throws CrowdinConnectionException {
    HttpClient httpClient = getHttpClient();
    HttpDelete request = new HttpDelete(uri);
    try {
      request.setHeader(AUTHORIZATION, TOKEN + accessToken);
      return processRequest(httpClient, request);
    } catch (IOException e) {
      throw new CrowdinConnectionException(CROWDIN_CONNECTION_ERROR, e);
    }
  }

  private String processRequest(HttpClient httpClient, HttpRequestBase request) throws IOException, CrowdinConnectionException {
    HttpResponse response = httpClient.execute(request);
    boolean isSuccess = response != null
            && (response.getStatusLine().getStatusCode() >= 200 && response.getStatusLine().getStatusCode() < 300);
    if (isSuccess) {
      return processSuccessResponse(response);
    } else if (response != null && response.getStatusLine().getStatusCode() == 404) {
      return null;
    } else {
      processErrorResponse(response);
      return null;
    }
  }

  private String processSuccessResponse(HttpResponse response) throws IOException {
    if (response.getStatusLine().getStatusCode() == HttpStatus.SC_NO_CONTENT) {
      return String.valueOf(HttpStatus.SC_NO_CONTENT);
    } else if ((response.getStatusLine().getStatusCode() == HttpStatus.SC_CREATED
            || response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) && response.getEntity() != null
            && response.getEntity().getContentLength() != 0) {
      try (InputStream is = response.getEntity().getContent()) {
        return IOUtils.toString(is, StandardCharsets.UTF_8);
      }
    } else {
      return null;
    }
  }

  private void processErrorResponse(HttpResponse response) throws CrowdinConnectionException, IOException {
    if (response == null) {
      throw new CrowdinConnectionException("Error when connecting crowdin");
    } else if (response.getEntity() != null) {
      try (InputStream is = response.getEntity().getContent()) {
        String errorMessage = IOUtils.toString(is, StandardCharsets.UTF_8);
        if (StringUtils.contains(errorMessage, "")) {
          throw new CrowdinConnectionException(errorMessage);
        } else {
          throw new CrowdinConnectionException(CROWDIN_CONNECTION_ERROR + errorMessage);
        }
      }
    } else {
      throw new CrowdinConnectionException(CROWDIN_CONNECTION_ERROR + response.getStatusLine().getStatusCode());
    }
  }

  private HttpClient getHttpClient() {
    if (client == null) {
      HttpClientConnectionManager clientConnectionManager = getClientConnectionManager();
      HttpClientBuilder httpClientBuilder = HttpClients.custom()
              .setConnectionManager(clientConnectionManager)
              .setConnectionReuseStrategy(new DefaultConnectionReuseStrategy());
      client = httpClientBuilder.build();
    }
    return client;
  }

  private HttpClientConnectionManager getClientConnectionManager() {
    PoolingHttpClientConnectionManager connectionManager = new PoolingHttpClientConnectionManager();
    connectionManager.setDefaultMaxPerRoute(10);
    return connectionManager;
  }

  public RemoteProject retrieveRemoteProject(long projectRemoteId,
                                             boolean includeLanguages,
                                             String accessToken) throws IllegalAccessException {
    try {

      URI uri = URI.create(CROWDIN_API_URL + PROJECTS + projectRemoteId);
      String response = processGet(uri, accessToken);
      if (response == null) {
        return null;
      }
      return getRemoteProject(response, includeLanguages);
    } catch (IllegalArgumentException e) {
      throw new IllegalArgumentException(e);
    } catch (CrowdinConnectionException e) {
      throw new IllegalAccessException(TOKEN_EXPIRED_OR_INVALID);
    }
  }

  private static RemoteProject getRemoteProject(String response, boolean includeLanguages) {
    JSONObject responseJson = new JSONObject(response);
    JSONObject jsonObject = responseJson.getJSONObject("data");

    RemoteProject project = new RemoteProject();
    project.setId(jsonObject.getInt("id"));
    project.setName(jsonObject.getString("name"));
    project.setIdentifier(jsonObject.getString("identifier"));
    if (jsonObject.get("logo") instanceof String) {
      project.setAvatarUrl(jsonObject.getString("logo"));
    }

    if (includeLanguages) {
      JSONArray languages = jsonObject.getJSONArray("targetLanguages");
      List<RemoteLanguage> remoteLanguages = new ArrayList<>();
      for (int i = 0; i < languages.length(); i++) {
        JSONObject language = languages.getJSONObject(i);
        RemoteLanguage remoteLanguage = new RemoteLanguage();
        remoteLanguage.setId(language.getString("id"));
        remoteLanguage.setName(language.getString("name"));
        remoteLanguages.add(remoteLanguage);
      }
      project.setLanguages(remoteLanguages);
    }

    return project;
  }

  public String deleteWebhook(WebHook webHook) {
    URI uri = URI.create(CROWDIN_API_URL + PROJECTS + webHook.getProjectId() + WEBHOOKS + webHook.getWebhookId());
    try {
      return processDelete(uri, webHook.getToken());
    } catch (CrowdinConnectionException e) {
      throw new IllegalStateException("Unable to delete Crowdin hook");
    }
  }

  public List<RemoteDirectory> getProjectDirectories(long remoteProjectId,
                                                     int offset,
                                                     int limit,
                                                     String accessToken) throws IllegalAccessException {
    try {
      URI uri = URI.create(CROWDIN_API_URL + PROJECTS + remoteProjectId + "/directories?offset=" + offset + "&limit=" + limit);
      String response = processGet(uri, accessToken);
      JSONArray jsonArray = new JSONObject(response).getJSONArray("data");

      List<RemoteDirectory> directories = new ArrayList<>();

      // loop through the array and parse the projects
      for (int i = 0; i < jsonArray.length(); i++) {
        JSONObject jsonObjectData = jsonArray.getJSONObject(i);
        JSONObject jsonObject = jsonObjectData.getJSONObject("data");
        // parse the project
        RemoteDirectory directory = new RemoteDirectory();
        directory.setId(jsonObject.getInt("id"));
        directory.setProjectId(jsonObject.getLong("projectId"));
        directory.setPath(jsonObject.getString("path"));

        directories.add(directory);
      }
      return directories;
    } catch (IllegalArgumentException e) {
      throw new IllegalArgumentException(e);
    } catch (CrowdinConnectionException e) {
      throw new IllegalAccessException(TOKEN_EXPIRED_OR_INVALID);
    }
  }

  public RemoteApproval getApproval(String accessToken, String projectId,String translationId) {

    try {

      URI uri = URI.create(CROWDIN_API_URL + PROJECTS + projectId + APPROVALS + "?translationId=" + translationId);

      String response = processGet(uri, accessToken);
      JSONArray jsonArray = new JSONObject(response).getJSONArray("data");

      if (jsonArray.isEmpty()) {
        LOG.warn("Return empty approvals response for crowdin translation with id {}.", translationId);
        return null;
      }

      JSONObject jsonObject = jsonArray.getJSONObject(0).getJSONObject("data");

      RemoteApproval remoteApproval = new RemoteApproval();

      JSONObject userJsonObject = jsonObject.getJSONObject("user");

      remoteApproval.setId(jsonObject.getInt("id"));

      remoteApproval.setUserName(userJsonObject.getString("username"));

      return remoteApproval;

    } catch (CrowdinConnectionException e) {
      LOG.warn("Unable to retrieve approval for crowdin translation with id {}.", translationId, e);
      return null;
    }
  }


  public void clearCache() {
    // implemented in cached storage
  }
}
