package io.meeds.gamification.storage;

import static io.meeds.gamification.storage.mapper.RealizationMapper.fromEntity;
import static io.meeds.gamification.storage.mapper.RealizationMapper.toEntity;

import java.util.*;

import io.meeds.gamification.constant.IdentityType;
import io.meeds.gamification.constant.RealizationStatus;
import io.meeds.gamification.dao.RealizationDAO;
import io.meeds.gamification.entity.RealizationEntity;
import io.meeds.gamification.model.PiechartLeaderboard;
import io.meeds.gamification.model.ProfileReputation;
import io.meeds.gamification.model.RealizationDTO;
import io.meeds.gamification.model.StandardLeaderboard;
import io.meeds.gamification.model.filter.RealizationFilter;

public class RealizationStorage {

  private ProgramStorage programStorage;

  private RuleStorage    ruleStorage;

  private RealizationDAO gamificationHistoryDAO;

  private static final List<RealizationStatus> PENDING_STATUS = Collections.singletonList(RealizationStatus.PENDING);

  public RealizationStorage(ProgramStorage programStorage,
                            RuleStorage ruleStorage,
                            RealizationDAO gamificationHistoryDAO) {
    this.programStorage = programStorage;
    this.ruleStorage = ruleStorage;
    this.gamificationHistoryDAO = gamificationHistoryDAO;
  }

  public List<RealizationDTO> getRealizationsByFilter(RealizationFilter realizationFilter,
                                                      int offset,
                                                      int limit) {
    List<Long> ids = gamificationHistoryDAO.findRealizationsByFilter(realizationFilter,
                                                                     offset,
                                                                     limit);
    return ids.stream().map(this::getRealizationById).toList();
  }

  public int countRealizationsByFilter(RealizationFilter realizationFilter) {
    return gamificationHistoryDAO.countRealizationsByFilter(realizationFilter);
  }

  public RealizationDTO getRealizationById(long id) {
    return fromEntity(programStorage, gamificationHistoryDAO.find(id));
  }

  public RealizationDTO updateRealization(RealizationDTO realization) {
    RealizationEntity realizationEntity = toEntity(ruleStorage, realization);
    realizationEntity = gamificationHistoryDAO.update(realizationEntity);
    return fromEntity(programStorage, realizationEntity);
  }

  public RealizationDTO createRealization(RealizationDTO realization) {
    RealizationEntity realizationEntity = toEntity(ruleStorage, realization);
    realizationEntity.setId(null);
    realizationEntity.setCreatedDate(new Date());
    return fromEntity(programStorage, gamificationHistoryDAO.create(realizationEntity));
  }

  public List<RealizationDTO> findRealizationsByObjectIdAndObjectType(String objectId, String objectType) {
    List<Long> ids = gamificationHistoryDAO.getRealizationsByObjectIdAndObjectType(objectId, objectType);
    return ids.stream().map(this::getRealizationById).toList();
  }

  public boolean hasPendingRealization(long ruleId, String earnerIdentityId) {
    RealizationFilter realizationFilter = new RealizationFilter();
    realizationFilter.setEarnerIds(new ArrayList<>(Collections.singleton(earnerIdentityId)));
    realizationFilter.setEarnerType(IdentityType.USER);
    realizationFilter.setStatuses(PENDING_STATUS);
    realizationFilter.setRuleIds(new ArrayList<>(Collections.singleton(ruleId)));
    return gamificationHistoryDAO.countRealizationsByFilter(realizationFilter) > 0;
  }

  public int getLeaderboardRankByDates(IdentityType identityType, long earnerIdentityId, Date fromDate, Date toDate) {
    return gamificationHistoryDAO.getLeaderboardRankByDates(identityType, earnerIdentityId, fromDate, toDate);
  }

  public int getLeaderboardRankByDatesAndProgramIds(IdentityType identityType,
                                                    long earnerIdentityId,
                                                    Date fromDate,
                                                    Date toDate,
                                                    Long... programIds) {
    return gamificationHistoryDAO.getLeaderboardRankByDatesAndProgramIds(identityType,
                                                                         earnerIdentityId,
                                                                         fromDate,
                                                                         toDate,
                                                                         programIds);
  }

  public int getLeaderboardRank(IdentityType identityType, long earnerIdentityId) {
    return gamificationHistoryDAO.getLeaderboardRank(identityType, earnerIdentityId);
  }

  public int getLeaderboardRankByProgramIds(IdentityType identityType, long earnerIdentityId, Long... programIds) {
    return gamificationHistoryDAO.getLeaderboardRankByProgramIds(identityType, earnerIdentityId, programIds);
  }

  public List<StandardLeaderboard> getLeaderboardByDates(Date fromDate,
                                                         Date toDate,
                                                         IdentityType identityType,
                                                         int offset,
                                                         int limit) {
    return gamificationHistoryDAO.getLeaderboardByDates(fromDate, toDate, identityType, offset, limit);
  }

  public List<StandardLeaderboard> getLeaderboard(IdentityType identityType, int offset, int limit) {
    return gamificationHistoryDAO.getLeaderboard(identityType, offset, limit);
  }

  public List<StandardLeaderboard> getLeaderboardByDatesByProgramIds(Date fromDate,
                                                                     Date toDate,
                                                                     IdentityType identityType,
                                                                     int offset,
                                                                     int limit,
                                                                     Long... programIds) {
    return gamificationHistoryDAO.getLeaderboardByDatesAndProgramIds(fromDate, toDate, identityType, offset, limit, programIds);
  }

  public List<StandardLeaderboard> getLeaderboardByProgramIds(IdentityType identityType,
                                                              int offset,
                                                              int limit,
                                                              Long... programIds) {
    return gamificationHistoryDAO.getLeaderboardByProgramIds(identityType, offset, limit, programIds);
  }

  public List<ProfileReputation> getScorePerProgramByIdentityId(String earnerIdentityId) {
    return gamificationHistoryDAO.getScorePerProgramByIdentityId(earnerIdentityId);
  }

  public List<PiechartLeaderboard> getLeaderboardStatsByIdentityIdAndDates(String earnerIdentityId,
                                                                           Long spaceId,
                                                                           Date startDate,
                                                                           Date endDate) {
    return gamificationHistoryDAO.getLeaderboardStatsByIdentityIdAndDates(earnerIdentityId, spaceId, startDate, endDate);
  }

  public List<PiechartLeaderboard> getLeaderboardStatsByIdentityId(String earnerIdentityId, Long spaceId) {
    return gamificationHistoryDAO.getLeaderboardStatsByIdentityId(earnerIdentityId, spaceId);
  }

  public long getScoreByIdentityIdAndBetweenDates(String earnerIdentityId, Date fromDate, Date toDate, Long ...programIds) {
    return gamificationHistoryDAO.getScoreByIdentityIdAndBetweenDates(earnerIdentityId, fromDate, toDate, programIds);
  }

  public long getScoreByIdentityId(String earnerIdentityId) {
    return gamificationHistoryDAO.getScoreByIdentityId(earnerIdentityId);
  }

  public Map<Long, Long> getScoresByIdentityIdsAndBetweenDates(List<String> earnersId, Date fromDate, Date toDate) {
    return gamificationHistoryDAO.getScoreByIdentityIdsAndBetweenDates(earnersId, fromDate, toDate);
  }

  public RealizationDTO findLastRealizationByRuleIdAndEarnerIdAndReceiverAndObjectId(long ruleId,
                                                                                     String earnerId,
                                                                                     String receiverId,
                                                                                     String objectId,
                                                                                     String objectType) {
    Long id = gamificationHistoryDAO.findLastRealizationByRuleIdAndEarnerIdAndReceiverAndObjectId(ruleId,
                                                                                                  earnerId,
                                                                                                  receiverId,
                                                                                                  objectId,
                                                                                                  objectType);
    return id == null || id == 0 ? null : getRealizationById(id);
  }

  public int countRealizationsByRuleIdAndEarnerId(long earnerIdentityId, long ruleId) {
    return gamificationHistoryDAO.countRealizationsByRuleIdAndEarnerId(earnerIdentityId, ruleId);
  }

  public int countRealizationsInPeriod(long earnerIdentityId, long ruleId, Date sinceDate) {
    return gamificationHistoryDAO.countRealizationsInPeriod(earnerIdentityId, ruleId, sinceDate);
  }

  public List<Long> getParticipantsBetweenDates(Date fromDate, Date toDate) {
    return gamificationHistoryDAO.getParticipantsBetweenDates(fromDate, toDate);
  }

  public long countParticipantsBetweenDates(Date fromDate, Date toDate) {
    return gamificationHistoryDAO.countParticipantsBetweenDates(fromDate, toDate);
  }

}
