/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.gatein.wci.tomcat;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.catalina.Container;
import org.apache.catalina.ContainerEvent;
import org.apache.catalina.ContainerListener;
import org.apache.catalina.Context;
import org.apache.catalina.Engine;
import org.apache.catalina.Host;
import org.apache.catalina.Lifecycle;
import org.apache.catalina.LifecycleEvent;
import org.apache.catalina.LifecycleListener;
import org.apache.catalina.LifecycleState;
import org.apache.catalina.core.StandardContext;
import org.gatein.wci.RequestDispatchCallback;
import org.gatein.wci.ServletContainer;
import org.gatein.wci.ServletContainerFactory;
import org.gatein.wci.authentication.AuthenticationException;
import org.gatein.wci.command.CommandDispatcher;
import org.gatein.wci.command.TomcatCommandDispatcher;
import org.gatein.wci.security.Credentials;
import org.gatein.wci.session.SessionTask;
import org.gatein.wci.session.SessionTaskVisitor;
import org.gatein.wci.spi.ServletContainerContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;

/**
 * An implementation of the <code>ServletContainerContext</code> for Tomcat 7.
 *
 */
public class TomcatServletContainerContext implements ServletContainerContext, ContainerListener, LifecycleListener
{
  private final static Logger                  log                     =
                                                   LoggerFactory.getLogger(TomcatServletContainerContext.class);

   private static TomcatServletContainerContext instance;

   /** . */
   private final CommandDispatcher dispatcher = new TomcatCommandDispatcher("/tomcatgateinservlet");

   /** The monitored hosts. */
   private final Set<String> monitoredHosts = new HashSet<String>();

   /** The monitored contexts. */
   private final Set<String> monitoredContexts = new HashSet<String>();

   /** The monitored contexts which were manually added. */
   private static Map<String, String> manualMonitoredContexts = new HashMap<String, String>();

   /** . */
   private final Engine engine;

   /** . */
   private Registration registration;

   /** Perform cross-context session invalidation on logout, or not */
   private boolean crossContextLogout = true;

   public static TomcatServletContainerContext getInstanceIfPresent() {
     return instance;
   }

   public TomcatServletContainerContext(Engine engine)
   {
      this.engine = engine;
   }

   public Object include(
      ServletContext targetServletContext,
      HttpServletRequest request,
      HttpServletResponse response,
      RequestDispatchCallback callback,
      Object handback) throws ServletException, IOException
   {
      if (manualMonitoredContexts.containsKey(targetServletContext.getServletContextName()))
      {
         String dispatherPath = manualMonitoredContexts.get(targetServletContext.getServletContextName());
         CommandDispatcher dispatcher = new TomcatCommandDispatcher(dispatherPath);
         return dispatcher.include(targetServletContext, request, response, callback, handback);
      }
      else
      {
         return dispatcher.include(targetServletContext, request, response, callback, handback);
      }
   }

   public void setCallback(Registration registration)
   {
      this.registration = registration;
   }

   public void unsetCallback(Registration registration)
   {
      this.registration = null;
   }

   public void setCrossContextLogout(boolean val)
   {
      crossContextLogout = val;
   }

   public void login(HttpServletRequest request, HttpServletResponse response, Credentials credentials) throws ServletException, IOException
   {
      request.getSession();
      try
      {
         request.login(credentials.getUsername(), credentials.getPassword());
      }
      catch (ServletException se)
      {
         throw new AuthenticationException(se);
      }
   }

   public void logout(HttpServletRequest request, HttpServletResponse response) throws ServletException
   {
      HttpSession sess = request.getSession(false);

      if (sess == null)
         return;

      if (!crossContextLogout)
         return;

      final String sessId = sess.getId();
      ServletContainerFactory.getServletContainer().visit(new SessionTaskVisitor(sessId, new SessionTask()
      {
         @Override
         public boolean executeTask(HttpSession session)
         {
            ClassLoader portalContainerCL = Thread.currentThread().getContextClassLoader();
            ClassLoader webAppCL = session.getServletContext().getClassLoader();

            Thread.currentThread().setContextClassLoader(webAppCL);
            try {
                session.invalidate();
            } finally {
                Thread.currentThread().setContextClassLoader(portalContainerCL);
            }

            return true;
         }

      }));
      request.logout();
   }

   public String getContainerInfo()
   {
      return "Tomcat/7.x";
   }

   public synchronized void containerEvent(ContainerEvent event)
   {
      if (event.getData() instanceof Host)
      {
         Host host = (Host)event.getData();

         //
         if (Container.ADD_CHILD_EVENT.equals(event.getType()))
         {
            registerHost(host);
         }
         else if (Container.REMOVE_CHILD_EVENT.equals(event.getType()))
         {
            unregisterHost(host);
         }
      }
      else if (event.getData() instanceof StandardContext)
      {
         StandardContext context = (StandardContext)event.getData();

         //
         if (Container.ADD_CHILD_EVENT.equals(event.getType()))
         {
            registerContext(context);
         }
         else if (Container.REMOVE_CHILD_EVENT.equals(event.getType()))
         {
            unregisterContext(context);
         }
      }
   }

   public void lifecycleEvent(LifecycleEvent event)
   {
      if (event.getSource() instanceof Context)
      {
         Context context = (Context)event.getSource();

         //
         if (Lifecycle.AFTER_START_EVENT.equals(event.getType()))
         {
            start(context);
         }
         else if (Lifecycle.BEFORE_STOP_EVENT.equals(event.getType()))
         {
            stop(context);
         }
      }
   }

   public Engine getEngine() {
     return engine;
   }

   void start()
   {
      ServletContainerFactory.registerContext(this);
      instance = this;
      //
      Container[] childrenContainers = engine.findChildren();
      for (Container childContainer : childrenContainers)
      {
         if (childContainer instanceof Host)
         {
            Host host = (Host)childContainer;
            registerHost(host);
         }
      }

      //
      engine.addContainerListener(this);
   }

   void stop()
   {
      instance = null;
      engine.removeContainerListener(this);

      //
      Container[] childrenContainers = engine.findChildren();
      for (Container childContainer : childrenContainers)
      {
         if (childContainer instanceof Host)
         {
            Host host = (Host)childContainer;
            unregisterHost(host);
         }
      }

      //
      if (registration != null) {
        registration.cancel();
        registration = null;
      }
    }

   /**
    * Register an host for registration which means that we fire events for all the contexts it contains and we
    * subscribe for its life cycle events. If the host is already monitored nothing is done.
    *
    * @param host the host to register for monitoring
    */
   private void registerHost(Host host)
   {
      if (!monitoredHosts.contains(host.getName()))
      {
         //
         monitoredHosts.add(host.getName());

         //
         host.addContainerListener(this);

         Container[] childrenContainers = host.findChildren();
         for (Container childContainer : childrenContainers)
         {
            if (childContainer instanceof StandardContext)
            {
               StandardContext context = (StandardContext)childContainer;
               registerContext(context);
            }
         }
      }
   }

   private void unregisterHost(Host host)
   {
      if (monitoredHosts.contains(host.getName()))
      {
         monitoredHosts.remove(host.getName());

         //
         host.removeContainerListener(this);

         //
         Container[] childrenContainers = host.findChildren();
         for (Container childContainer : childrenContainers)
         {
            if (childContainer instanceof StandardContext)
            {
               StandardContext context = (StandardContext)childContainer;
               unregisterContext(context);
            }
         }
      }
   }

   private void registerContext(StandardContext context)
   {
      if (monitoredContexts.contains(context.getName())) {
         return;
      }

      synchronized (monitoredContexts) {
         if (!monitoredContexts.contains(context.getName())) {
            //
            monitoredContexts.add(context.getName());

            context.addLifecycleListener(this);

            //
            if (LifecycleState.STARTED.equals(context.getState())) {
               start(context);
            }
         }
      }
   }

   private void unregisterContext(StandardContext context)
   {
      if (!monitoredContexts.contains(context.getName())) {
         return;
      }

      synchronized (monitoredContexts) {
         if (monitoredContexts.contains(context.getName())) {
            monitoredContexts.remove(context.getName());

            //
            if (LifecycleState.STARTED.equals(context.getState())) {
               stop(context);
            }

            //
            context.removeLifecycleListener(this);
         }
      }
   }

   private void start(Context context)
   {
      try
      {
         // skip if the webapp has explicitly stated it doesn't want native registration
         // usefull when portlets are dependent on servlet ordering
         if (!ServletContainer.isDisabledNativeRegistration(context.getServletContext()))
         {
            log.debug("Context added " + context.getPath());
            TomcatWebAppContext webAppContext = new TomcatWebAppContext(context);

            //
            if (registration != null)
            {
               registration.registerWebApp(webAppContext);
            }
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }

   private void stop(Context context)
   {
      try
      {
         // skip if the webapp has explicitly stated it doesn't want native registration
         // usefull when portlets are dependent on servlet ordering
         if (!ServletContainer.isDisabledNativeRegistration(context.getServletContext()))
         {
            if (registration != null)
            {
               registration.unregisterWebApp(context.getPath());
            }
         }
      }
      catch (Exception e)
      {
         e.printStackTrace();
      }
   }
}
