/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.gatein.wci;

import java.io.IOException;
import java.io.InputStream;

import jakarta.servlet.ServletContext;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpSession;

/**
 * Abstraction of a web application.
 */
public interface WebApp
{

   /**
    * Returns the servlet context of the web application.
    *
    * @return the servlet context
    */
   ServletContext getServletContext();

   /**
    * Returns the class loader of the web application.
    *
    * @return the web application class loader
    */
   ClassLoader getClassLoader();

   /**
    * Returns the context path of the web application.
    *
    * @return the web application context path
    */
   String getContextPath();

   /**
    * Import a file in the war file. The file could not be created for some reasons which are :
    * <ul>
    *   <li>The parent dir exists and is a file</li>
    *   <li>The parent dir does not exist and its creation failed</li>
    *   <li>An underlying exception occurs when writing bytes from the source <code>Inputstream</code> to the target <code>OutputStream</code></li>
    * </ul>
    *
    * @param parentDirRelativePath the parent relative path in the web app starting from the app root
    * @param name                  the name the created file should have
    * @param source                the data of the target file
    * @param overwrite             if false and the file already exists nothing is done
    * @return true if the file has been created
    * @throws IOException if the file cannot be created
    */
   boolean importFile(String parentDirRelativePath, String name, InputStream source, boolean overwrite) throws IOException;

    /**
     * Returns Http session in case that session with sessId exists for this web application
     * @param sessId
     * @return session with given ID or null if session with this ID doesn't exist
     */
   HttpSession getHttpSession(String sessId);

   public void fireRequestInitialized(ServletRequest servletRequest);

   public void fireRequestDestroyed(ServletRequest servletRequest);
}
