/**
 * This file is part of the Meeds project (https://meeds.io/).
 *
 * Copyright (C) 2020 - 2025 Meeds Association contact@meeds.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
package org.exoplatform.container.util;

import junit.framework.TestCase;
import org.exoplatform.commons.utils.ExoEnumeration;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.URL;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;

import jakarta.servlet.ServletContext;

public class TestContainerUtil extends TestCase {
  private ClassLoader oldCLoader;

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    oldCLoader = Thread.currentThread().getContextClassLoader();
  }

  @Override
  protected void tearDown() throws Exception {
    super.tearDown();
    Thread.currentThread().setContextClassLoader(oldCLoader);
  }

  public void testGetServletContextName() {
    final AtomicReference<String> scn = new AtomicReference<String>("myContextName");
    final AtomicReference<String> scp = new AtomicReference<String>("/myContextPath");
    ServletContext context = (ServletContext) Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class<?>[]{ServletContext.class}, new InvocationHandler() {

      public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if ("getServletContextName".equals(method.getName()))
          return scn.get();
        else if ("getContextPath".equals(method.getName()))
          return scp.get();
        return null;
      }
    });
    assertEquals("myContextName", ContainerUtil.getServletContextName(context));
    scn.set(null);
    assertEquals("myContextPath", ContainerUtil.getServletContextName(context));
    scp.set("");
    assertEquals("", ContainerUtil.getServletContextName(context));
    scp.set("/a/b");
    assertEquals("a", ContainerUtil.getServletContextName(context));
    scp.set("/a2/b/");
    assertEquals("a2", ContainerUtil.getServletContextName(context));
    scp.set("a3/b");
    assertEquals("a3", ContainerUtil.getServletContextName(context));
    scp.set("a4/b/");
    assertEquals("a4", ContainerUtil.getServletContextName(context));
    scp.set(null);
    assertNull(ContainerUtil.getServletContextName(context));
  }

  public void testURLsKeepInsertionOrderWhenGettingConfigurationURLs() throws Exception {
    // Given
    Thread.currentThread().setContextClassLoader(new MockClassLoader());

    // When
    Collection<URL> urls = ContainerUtil.getConfigurationURL("dummy");

    // Then
    assertNotNull(urls);
    Iterator<URL> urlIterator = urls.iterator();
    assertEquals("file:/url1", urlIterator.next().toString());
    assertEquals("file:/url2", urlIterator.next().toString());
    assertEquals("file:/url3", urlIterator.next().toString());
  }

  private static class MockClassLoader extends ClassLoader {
    @Override
    public Enumeration<URL> getResources(String name) throws IOException {
      List<URL> resources = Arrays.asList(new URL("file:/url1"), new URL("file:/url2"), new URL("file:/url3"));
      return new ExoEnumeration(resources.iterator());
    }
  }
}
