/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.oidc.runtime;

import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import java.time.Duration;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

public class MemoryCache<T> {
    private volatile Long timerId = null;
    private final Map<String, CacheEntry<T>> cacheMap = new ConcurrentHashMap<String, CacheEntry<T>>();
    private AtomicInteger size = new AtomicInteger();
    private final Duration cacheTimeToLive;
    private final int cacheSize;

    public MemoryCache(Vertx vertx, Optional<Duration> cleanUpTimerInterval, Duration cacheTimeToLive, int cacheSize) {
        this.cacheTimeToLive = cacheTimeToLive;
        this.cacheSize = cacheSize;
        this.init(vertx, cleanUpTimerInterval);
    }

    private void init(Vertx vertx, Optional<Duration> cleanUpTimerInterval) {
        if (cleanUpTimerInterval.isPresent()) {
            this.timerId = vertx.setPeriodic(cleanUpTimerInterval.get().toMillis(), (Handler)new Handler<Long>(){

                public void handle(Long event) {
                    MemoryCache.this.removeInvalidEntries();
                }
            });
        }
    }

    public void add(String key, T result) {
        if (this.cacheSize > 0) {
            if (!this.prepareSpaceForNewCacheEntry()) {
                this.clearCache();
            }
            this.cacheMap.put(key, new CacheEntry<T>(result));
        }
    }

    public T remove(String key) {
        CacheEntry<T> entry = this.removeCacheEntry(key);
        return entry == null ? null : (T)entry.result;
    }

    public T get(String key) {
        CacheEntry<T> entry = this.cacheMap.get(key);
        return entry == null ? null : (T)entry.result;
    }

    public boolean containsKey(String key) {
        return this.cacheMap.containsKey(key);
    }

    private void removeInvalidEntries() {
        long now = MemoryCache.now();
        Iterator<Map.Entry<String, CacheEntry<T>>> it = this.cacheMap.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, CacheEntry<T>> next = it.next();
            if (next == null || !this.isEntryExpired(next.getValue(), now)) continue;
            try {
                it.remove();
                this.size.decrementAndGet();
            }
            catch (IllegalStateException illegalStateException) {}
        }
    }

    private boolean prepareSpaceForNewCacheEntry() {
        int currentSize;
        do {
            if ((currentSize = this.size.get()) != this.cacheSize) continue;
            return false;
        } while (!this.size.compareAndSet(currentSize, currentSize + 1));
        return true;
    }

    private CacheEntry<T> removeCacheEntry(String token) {
        CacheEntry<T> entry = this.cacheMap.remove(token);
        if (entry != null) {
            this.size.decrementAndGet();
        }
        return entry;
    }

    private boolean isEntryExpired(CacheEntry<T> entry, long now) {
        return entry.createdTime + this.cacheTimeToLive.toMillis() < now;
    }

    private static long now() {
        return System.currentTimeMillis();
    }

    public int getCacheSize() {
        return this.cacheMap.size();
    }

    public void clearCache() {
        this.cacheMap.clear();
        this.size.set(0);
    }

    public void stopTimer(Vertx vertx) {
        if (this.timerId != null && vertx.cancelTimer(this.timerId.longValue())) {
            this.timerId = null;
        }
    }

    public boolean isTimerRunning() {
        return this.timerId != null;
    }

    private static class CacheEntry<T> {
        volatile T result;
        long createdTime = System.currentTimeMillis();

        public CacheEntry(T result) {
            this.result = result;
        }
    }
}

