/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.map;

import net.openhft.koloboke.collect.*;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Map;



/**
 * An immutable factory of {@code DoubleByteMap}s.
 *
 * 
 * 
 * @param <F> the concrete factory type which extends this interface
 * @see DoubleByteMap
 */
public interface DoubleByteMapFactory<F extends DoubleByteMapFactory<F>>
        extends ContainerFactory<F> {

    

    

    

    

    

    
    

    
    




    /**
     * Returns the value to which {@linkplain DoubleByteMap#defaultValue() default value} of the maps
     * constructed by this factory is set. Default value is {@code
     * (byte) 0}.
     *
     * @return the default value of the maps constructed by this factory
     */
    byte getDefaultValue();

    /**
     * Returns a copy of this factory, with exception that it constructs maps with
     * {@linkplain DoubleByteMap#defaultValue() default value} set to the given {@code byte} value.
     *
     * @param defaultValue the new default {@code byte} value
     * @return a copy of this factory, which constructs maps with the given {@code defaultValue}
     */
    @Nonnull
    F withDefaultValue(byte defaultValue);

    

    /**
     * Constructs a new empty mutable map of the {@linkplain #getDefaultExpectedSize()
     * default expected size}.
     *
    
     
     * @return a new empty mutable map
     */
    @Nonnull
     DoubleByteMap newMutableMap();

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
     DoubleByteMap newMutableMap(int expectedSize);

    

    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4,
            @Nonnull Map<Double, Byte> map5, int expectedSize);



    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.DoubleByteConsumer> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleByteMap newMutableMap(
            @Nonnull double[] keys, @Nonnull byte[] values, int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleByteMap newMutableMap(
            @Nonnull Double[] keys, @Nonnull Byte[] values, int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Byte> values, int expectedSize);
    

    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
     
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4,
            @Nonnull Map<Double, Byte> map5);



    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
     
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.DoubleByteConsumer> entriesSupplier
            );

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleByteMap newMutableMap(
            @Nonnull double[] keys, @Nonnull byte[] values);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleByteMap newMutableMap(
            @Nonnull Double[] keys, @Nonnull Byte[] values);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
     
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleByteMap newMutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Byte> values);

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
     DoubleByteMap newMutableMapOf(double k1, byte v1);

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
     DoubleByteMap newMutableMapOf(double k1, byte v1,
            double k2, byte v2);

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
     DoubleByteMap newMutableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3);

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
     DoubleByteMap newMutableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3,
            double k4, byte v4);

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
     DoubleByteMap newMutableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3,
            double k4, byte v4, double k5, byte v5);
    /**
     * Constructs a new empty updatable map of the {@linkplain #getDefaultExpectedSize()
     * default expected size}.
     *
    
     
     * @return a new empty updatable map
     */
    @Nonnull
     DoubleByteMap newUpdatableMap();

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(int expectedSize);

    

    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4,
            @Nonnull Map<Double, Byte> map5, int expectedSize);



    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.DoubleByteConsumer> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(
            @Nonnull double[] keys, @Nonnull byte[] values, int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(
            @Nonnull Double[] keys, @Nonnull Byte[] values, int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Byte> values, int expectedSize);
    

    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
     
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4,
            @Nonnull Map<Double, Byte> map5);



    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
     
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.DoubleByteConsumer> entriesSupplier
            );

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(
            @Nonnull double[] keys, @Nonnull byte[] values);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(
            @Nonnull Double[] keys, @Nonnull Byte[] values);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
     
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleByteMap newUpdatableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Byte> values);

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
     DoubleByteMap newUpdatableMapOf(double k1, byte v1);

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
     DoubleByteMap newUpdatableMapOf(double k1, byte v1,
            double k2, byte v2);

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
     DoubleByteMap newUpdatableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3);

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
     DoubleByteMap newUpdatableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3,
            double k4, byte v4);

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
     DoubleByteMap newUpdatableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3,
            double k4, byte v4, double k5, byte v5);

    

    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4,
            @Nonnull Map<Double, Byte> map5, int expectedSize);



    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.DoubleByteConsumer> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleByteMap newImmutableMap(
            @Nonnull double[] keys, @Nonnull byte[] values, int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleByteMap newImmutableMap(
            @Nonnull Double[] keys, @Nonnull Byte[] values, int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Byte> values, int expectedSize);
    

    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
     
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
     
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Map<Double, Byte> map1,
            @Nonnull Map<Double, Byte> map2,
            @Nonnull Map<Double, Byte> map3,
            @Nonnull Map<Double, Byte> map4,
            @Nonnull Map<Double, Byte> map5);



    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
     
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.DoubleByteConsumer> entriesSupplier
            );

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
     DoubleByteMap newImmutableMap(
            @Nonnull double[] keys, @Nonnull byte[] values);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *         or {@code
     *         values} contain {@code null} elements
     */
    @Nonnull
     DoubleByteMap newImmutableMap(
            @Nonnull Double[] keys, @Nonnull Byte[] values);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
     
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
     DoubleByteMap newImmutableMap(@Nonnull Iterable<Double> keys,
            @Nonnull Iterable<Byte> values);

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
     
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
     DoubleByteMap newImmutableMapOf(double k1, byte v1);

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
     
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
     DoubleByteMap newImmutableMapOf(double k1, byte v1,
            double k2, byte v2);

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
     
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
     DoubleByteMap newImmutableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3);

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
     
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
     DoubleByteMap newImmutableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3,
            double k4, byte v4);

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
     
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
     DoubleByteMap newImmutableMapOf(double k1, byte v1,
            double k2, byte v2, double k3, byte v3,
            double k4, byte v4, double k5, byte v5);
}

