/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.map;

import net.openhft.koloboke.collect.*;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Map;



/**
 * An immutable factory of {@code LongObjMap}s.
 *
 * 
 *  @param <V> the most general value type of the maps that could be constructed
 *                           by this factory 
 * @param <F> the concrete factory type which extends this interface
 * @see LongObjMap
 */
public interface LongObjMapFactory<V, F extends LongObjMapFactory<V, F>>
        extends ContainerFactory<F> {

    

    

    

    

    

    
    

    
    



    /**
     * Returns the equivalence to which {@linkplain LongObjMap#valueEquivalence() value
     * equivalence} of the maps constructed by this factory is set. Defaults
     * to {@link Equivalence#defaultEquality()}.
     *
     * @return the value equivalence of the maps constructed by this factory
     */
    @Nonnull Equivalence<V> getValueEquivalence();

    /**
     * Returns a copy of this factory, with exception that it constructs maps with
     * {@linkplain LongObjMap#valueEquivalence() value equivalence} set to the given
     * {@code Equivalence}.
     *
     * @param valueEquivalence the new value equivalence
     * @return a copy of this factory, which constructs maps with the given {@code valueEquivalence}
     */
    @Nonnull
    F withValueEquivalence(@Nonnull Equivalence<? super V> valueEquivalence);


    

    /**
     * Constructs a new empty mutable map of the {@linkplain #getDefaultExpectedSize()
     * default expected size}.
     *
    
      * @param <V2> the value type of the returned map 
     * @return a new empty mutable map
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap();

    /**
     * Constructs a new empty mutable map of the given expected size.
     *
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new empty mutable map of the given expected size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(int expectedSize);

    

    


    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4, int expectedSize);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4,
            @Nonnull Map<Long, ? extends V2> map5, int expectedSize);



    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongObjConsumer<V2>> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(
            @Nonnull long[] keys, @Nonnull V2[] values, int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *           contain {@code null} elements
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(
            @Nonnull Long[] keys, @Nonnull V2[] values, int expectedSize);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Iterable<Long> keys,
            @Nonnull Iterable<? extends V2> values, int expectedSize);
    

    

    /**
     * Constructs a new mutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4);

    /**
     * Constructs a new mutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4,
            @Nonnull Map<Long, ? extends V2> map5);



    /**
     * Constructs a new mutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongObjConsumer<V2>> entriesSupplier
            );

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(
            @Nonnull long[] keys, @Nonnull V2[] values);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *           contain {@code null} elements
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(
            @Nonnull Long[] keys, @Nonnull V2[] values);

    /**
     * Constructs a new mutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMap(@Nonnull Iterable<Long> keys,
            @Nonnull Iterable<? extends V2> values);

    /**
     * Constructs a new mutable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map of the single specified mapping
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMapOf(long k1, V2 v1);

    /**
     * Constructs a new mutable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map of the two specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMapOf(long k1, V2 v1,
            long k2, V2 v2);

    /**
     * Constructs a new mutable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map of the three specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3);

    /**
     * Constructs a new mutable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map of the four specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3,
            long k4, V2 v4);

    /**
     * Constructs a new mutable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new mutable map of the five specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newMutableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3,
            long k4, V2 v4, long k5, V2 v5);
    /**
     * Constructs a new empty updatable map of the {@linkplain #getDefaultExpectedSize()
     * default expected size}.
     *
    
      * @param <V2> the value type of the returned map 
     * @return a new empty updatable map
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap();

    /**
     * Constructs a new empty updatable map of the given expected size.
     *
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new empty updatable map of the given expected size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(int expectedSize);

    

    


    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4, int expectedSize);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4,
            @Nonnull Map<Long, ? extends V2> map5, int expectedSize);



    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongObjConsumer<V2>> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(
            @Nonnull long[] keys, @Nonnull V2[] values, int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *           contain {@code null} elements
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(
            @Nonnull Long[] keys, @Nonnull V2[] values, int expectedSize);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Iterable<Long> keys,
            @Nonnull Iterable<? extends V2> values, int expectedSize);
    

    

    /**
     * Constructs a new updatable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4);

    /**
     * Constructs a new updatable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4,
            @Nonnull Map<Long, ? extends V2> map5);



    /**
     * Constructs a new updatable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongObjConsumer<V2>> entriesSupplier
            );

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(
            @Nonnull long[] keys, @Nonnull V2[] values);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *           contain {@code null} elements
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(
            @Nonnull Long[] keys, @Nonnull V2[] values);

    /**
     * Constructs a new updatable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMap(@Nonnull Iterable<Long> keys,
            @Nonnull Iterable<? extends V2> values);

    /**
     * Constructs a new updatable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map of the single specified mapping
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMapOf(long k1, V2 v1);

    /**
     * Constructs a new updatable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map of the two specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMapOf(long k1, V2 v1,
            long k2, V2 v2);

    /**
     * Constructs a new updatable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map of the three specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3);

    /**
     * Constructs a new updatable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map of the four specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3,
            long k4, V2 v4);

    /**
     * Constructs a new updatable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new updatable map of the five specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newUpdatableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3,
            long k4, V2 v4, long k5, V2 v5);

    

    


    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4, int expectedSize);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4,
            @Nonnull Map<Long, ? extends V2> map5, int expectedSize);



    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongObjConsumer<V2>> entriesSupplier
            , int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(
            @Nonnull long[] keys, @Nonnull V2[] values, int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *           contain {@code null} elements
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(
            @Nonnull Long[] keys, @Nonnull V2[] values, int expectedSize);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * @param expectedSize the expected size of the returned map
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Iterable<Long> keys,
            @Nonnull Iterable<? extends V2> values, int expectedSize);
    

    

    /**
     * Constructs a new immutable map with the same mappings as the specified {@code map}.
     *
     * 
     *
     * @param map the map whose mappings are to be placed in the returned map
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the same mappings as the specified {@code map}
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the {@code map2} have priority over mappings from the {@code map1} with
     * the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4);

    /**
     * Constructs a new immutable map which merge the mappings of the specified maps. On conflict,
     * mappings from the maps passed later in the argument list have priority over mappings
     * from the maps passed earlier with the same keys.
     *
     * @param map1 the first map to merge
     * @param map2 the second map to merge
     * @param map3 the third map to merge
     * @param map4 the fourth map to merge
     * @param map5 the fifth map to merge
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map which merge the mappings of the specified maps
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Map<Long, ? extends V2> map1,
            @Nonnull Map<Long, ? extends V2> map2,
            @Nonnull Map<Long, ? extends V2> map3,
            @Nonnull Map<Long, ? extends V2> map4,
            @Nonnull Map<Long, ? extends V2> map5);



    /**
     * Constructs a new immutable map filled with mappings consumed by the callback within the given
     * closure. Mappings supplied later within the closure have priority over the mappings
     * passed earlier with the same keys.
     *
     * <p>Example: TODO
     *
     * @param entriesSupplier the function which supply mappings for the returned map via
     *        the callback passed in
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with mappings consumed by the callback within the given closure
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull
            Consumer<net.openhft.koloboke.function.LongObjConsumer<V2>> entriesSupplier
            );

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(
            @Nonnull long[] keys, @Nonnull V2[] values);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} arrays at the same index. If {@code keys} array have
     * duplicate elements, value corresponding the key with the highest index is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} array at the same index
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} arrays have different
     *         length
     * @throws NullPointerException if {@code keys}
     *           contain {@code null} elements
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(
            @Nonnull Long[] keys, @Nonnull V2[] values);

    /**
     * Constructs a new immutable map with the given mappings, i. e. pairs of elements from
     * the {@code keys} and {@code values} iterables at the same iteration position. If {@code keys}
     * have duplicate elements, value corresponding the key appeared last in the iteration is left
     * in the returned map.
     *
     * @param keys the keys of the returned map
     * @param values the values of the returned map, each value is associated with the element
     *        of the {@code keys} iterable at the same iteration position
     * 
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map with the given mappings
     * @throws IllegalArgumentException if {@code keys} and {@code values} have different size
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMap(@Nonnull Iterable<Long> keys,
            @Nonnull Iterable<? extends V2> values);

    /**
     * Constructs a new immutable map of the single specified mapping.
     *
     * @param k1 the key of the sole mapping
     * @param v1 the value of the sole mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map of the single specified mapping
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMapOf(long k1, V2 v1);

    /**
     * Constructs a new immutable map of the two specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map of the two specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMapOf(long k1, V2 v1,
            long k2, V2 v2);

    /**
     * Constructs a new immutable map of the three specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map of the three specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3);

    /**
     * Constructs a new immutable map of the four specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map of the four specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3,
            long k4, V2 v4);

    /**
     * Constructs a new immutable map of the five specified mappings.
     *
     * @param k1 the key of the first mapping
     * @param v1 the value of the first mapping
     * @param k2 the key of the second mapping
     * @param v2 the value of the second mapping
     * @param k3 the key of the third mapping
     * @param v3 the value of the third mapping
     * @param k4 the key of the fourth mapping
     * @param v4 the value of the fourth mapping
     * @param k5 the key of the fifth mapping
     * @param v5 the value of the fifth mapping
    
      * @param <V2> the value type of the returned map 
     * @return a new immutable map of the five specified mappings
     */
    @Nonnull
    <V2 extends V> LongObjMap<V2> newImmutableMapOf(long k1, V2 v1,
            long k2, V2 v2, long k3, V2 v3,
            long k4, V2 v4, long k5, V2 v5);
}

