/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.set;

import net.openhft.koloboke.collect.ContainerFactory;
import net.openhft.koloboke.collect.Equivalence;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import java.util.Iterator;


/**
 * An immutable factory of {@code ObjSet}s.
 *
 *  @param <E> the most general element type of the sets that could be constructed
 *                          by this factory 
 * @param <F> the concrete factory type which extends this interface
 * @see ObjSet
 */
public interface ObjSetFactory<E, 
        F extends ObjSetFactory<E, F>> extends ContainerFactory<F> {

    /**
     * Returns the equivalence to which {@linkplain ObjSet#equivalence() equivalence} of the sets
     * constructed by this factory is set.
     *
     * @return the key equivalence of the maps constructed by this factory
     */
    @Nonnull
    Equivalence<E> getEquivalence();

    

    

    

    


    /**
     * Constructs a new empty mutable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
      * @param <E2> the element type of the returned set 
     * @return a new empty mutable set
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet();

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(int expectedSize);

    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  an instance of {@code ObjSet} and has the same {@linkplain
     * ObjSet#equivalence() equivalence} with this factory (and thus the constructed set),
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elements, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4, int expectedSize);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4,
            @Nonnull Iterable<? extends E2> elems5, int expectedSize);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterator<? extends E2> elements, int expectedSize);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E2>> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull E2[] elements, int expectedSize);


    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elements);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4);

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set which merge the elements of the specified iterables
     */
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4,
            @Nonnull Iterable<? extends E2> elems5);

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull Iterator<? extends E2> elements);

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E2>> elementsSupplier
            );

    

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSet(@Nonnull E2[] elements);



    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * @param e1 the sole element
     * @param <E2> the element type of the returned set 
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSetOf(E2 e1);

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSetOf(E2 e1, E2 e2);

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSetOf(E2 e1, E2 e2, E2 e3);

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSetOf(E2 e1, E2 e2, E2 e3,
            E2 e4);

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
     * @param <E2> the element type of the returned set 
     * @return a new mutable set of the specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newMutableSetOf(E2 e1, E2 e2, E2 e3,
            E2 e4, E2 e5, E2... restElements);

    /**
     * Constructs a new empty updatable set of the {@linkplain #getDefaultExpectedSize() default
     * expected size}.
     *
      * @param <E2> the element type of the returned set 
     * @return a new empty updatable set
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet();

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(int expectedSize);

    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  an instance of {@code ObjSet} and has the same {@linkplain
     * ObjSet#equivalence() equivalence} with this factory (and thus the constructed set),
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elements, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4, int expectedSize);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4,
            @Nonnull Iterable<? extends E2> elems5, int expectedSize);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterator<? extends E2> elements, int expectedSize);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E2>> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull E2[] elements, int expectedSize);


    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elements);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4);

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set which merge the elements of the specified iterables
     */
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4,
            @Nonnull Iterable<? extends E2> elems5);

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull Iterator<? extends E2> elements);

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E2>> elementsSupplier
            );

    

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSet(@Nonnull E2[] elements);



    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * @param e1 the sole element
     * @param <E2> the element type of the returned set 
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSetOf(E2 e1);

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSetOf(E2 e1, E2 e2);

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSetOf(E2 e1, E2 e2, E2 e3);

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSetOf(E2 e1, E2 e2, E2 e3,
            E2 e4);

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
     * @param <E2> the element type of the returned set 
     * @return a new updatable set of the specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newUpdatableSetOf(E2 e1, E2 e2, E2 e3,
            E2 e4, E2 e5, E2... restElements);

    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  an instance of {@code ObjSet} and has the same {@linkplain
     * ObjSet#equivalence() equivalence} with this factory (and thus the constructed set),
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elements, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4, int expectedSize);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4,
            @Nonnull Iterable<? extends E2> elems5, int expectedSize);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterator<? extends E2> elements, int expectedSize);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E2>> elementsSupplier
            , int expectedSize);

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull E2[] elements, int expectedSize);


    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elements);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4);

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set which merge the elements of the specified iterables
     */
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterable<? extends E2> elems1,
            @Nonnull Iterable<? extends E2> elems2,
            @Nonnull Iterable<? extends E2> elems3,
            @Nonnull Iterable<? extends E2> elems4,
            @Nonnull Iterable<? extends E2> elems5);

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull Iterator<? extends E2> elements);

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.Consumer<E2>> elementsSupplier
            );

    

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSet(@Nonnull E2[] elements);



    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * @param e1 the sole element
     * @param <E2> the element type of the returned set 
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSetOf(E2 e1);

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSetOf(E2 e1, E2 e2);

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSetOf(E2 e1, E2 e2, E2 e3);

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSetOf(E2 e1, E2 e2, E2 e3,
            E2 e4);

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
     * @param <E2> the element type of the returned set 
     * @return a new immutable set of the specified elements
     */
    @Nonnull
    <E2 extends E> ObjSet<E2> newImmutableSetOf(E2 e1, E2 e2, E2 e3,
            E2 e4, E2 e5, E2... restElements);
}

