/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.openhft.koloboke.collect.set.hash;

import net.openhft.koloboke.collect.set.IntSet;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import java.util.*;


/**
 * This class consists only of static factory methods to construct {@code HashIntSet}s, and
 * the default {@link HashIntSetFactory} static provider ({@link #getDefaultFactory()}).
 *
 * @see HashIntSet
 */
public final class HashIntSets {

    private static class DefaultFactoryHolder {
        private static final HashIntSetFactory defaultFactory =
                ServiceLoader.load(HashIntSetFactory.class).iterator().next();
    }

    /**
     * Returns the default implementation of {@link HashIntSetFactory}, to which all static methods
     * in this class delegate.
     *
     * @return the default implementation of {@link HashIntSetFactory}
    
     * @throws RuntimeException if no implementations of {@link HashIntSetFactory} are provided
     */
    @Nonnull
    public static  HashIntSetFactory getDefaultFactory() {
        return (HashIntSetFactory) DefaultFactoryHolder.defaultFactory;
    }
    
    

    


    /**
     * Constructs a new empty mutable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet() newMutableSet()}.
     *
     
     * @return a new empty mutable set
     */
    @Nonnull
    public static  HashIntSet newMutableSet() {
        return getDefaultFactory().newMutableSet();
    }

    /**
     * Constructs a new empty mutable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(int) newMutableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty mutable set of the given expected size
     */
    @Nonnull
    public static  HashIntSet newMutableSet(int expectedSize) {
        return getDefaultFactory().newMutableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterable, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4,
            @Nonnull Iterable<Integer> elems5, int expectedSize) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterator, int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterator<Integer> elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Consumer, int) newMutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.IntConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newMutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * int[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashIntSet newMutableSet(@Nonnull int[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Integer[], int) newMutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashIntSet newMutableSet(@Nonnull Integer[] elements, int expectedSize) {
        return getDefaultFactory().newMutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new mutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterable) newMutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterable, Iterable) newMutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2) {
        return getDefaultFactory().newMutableSet(elems1, elems2);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(Iterable,
     * Iterable, Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new mutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newMutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new mutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4,
            @Nonnull Iterable<Integer> elems5) {
        return getDefaultFactory().newMutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new mutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Iterator) newMutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new mutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashIntSet newMutableSet(
            @Nonnull Iterator<Integer> elements) {
        return getDefaultFactory().newMutableSet(elements);
    }

    /**
     * Constructs a new mutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Consumer) newMutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new mutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntSet newMutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.IntConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newMutableSet(elementsSupplier);
    }

    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * int[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     */
    @Nonnull
    public static  HashIntSet newMutableSet(@Nonnull int[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }
    
    /**
     * Constructs a new mutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSet(
     * Integer[]) newMutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new mutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashIntSet newMutableSet(@Nonnull Integer[] elements) {
        return getDefaultFactory().newMutableSet(elements);
    }


    /**
     * Constructs a new mutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSetOf(
     * int) newMutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new mutable singleton set of the given element
     */
    @Nonnull
    public static  HashIntSet newMutableSetOf(int e1) {
        return getDefaultFactory().newMutableSetOf(e1);
    }

    /**
     * Constructs a new mutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSetOf(
     * int, int) newMutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new mutable set of the two specified elements
     */
    @Nonnull
    public static  HashIntSet newMutableSetOf(int e1, int e2) {
        return getDefaultFactory().newMutableSetOf(e1, e2);
    }

    /**
     * Constructs a new mutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSetOf(
     * int, int, int) newMutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new mutable set of the three specified elements
     */
    @Nonnull
    public static  HashIntSet newMutableSetOf(int e1, int e2, int e3) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new mutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSetOf(
     * int, int, int, int) newMutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new mutable set of the four specified elements
     */
    @Nonnull
    public static  HashIntSet newMutableSetOf(int e1, int e2, int e3, int e4) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new mutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newMutableSetOf(int, int, int,
     * int, int, int...) newMutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new mutable set of the specified elements
     */
    @Nonnull
    public static  HashIntSet newMutableSetOf(int e1, int e2, int e3, int e4,
            int e5, int... restElements) {
        return getDefaultFactory().newMutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    /**
     * Constructs a new empty updatable set of the default expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet() newUpdatableSet()}.
     *
     
     * @return a new empty updatable set
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet() {
        return getDefaultFactory().newUpdatableSet();
    }

    /**
     * Constructs a new empty updatable set of the given expected size.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(int) newUpdatableSet(expectedSize)}.
     *
     * @param expectedSize the expected size of the returned set
    
     * @return a new empty updatable set of the given expected size
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(int expectedSize) {
        return getDefaultFactory().newUpdatableSet(expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterable, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4,
            @Nonnull Iterable<Integer> elems5, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterator, int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterator<Integer> elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Consumer, int) newUpdatableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.IntConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * int[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(@Nonnull int[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Integer[], int) newUpdatableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashIntSet newUpdatableSet(@Nonnull Integer[] elements, int expectedSize) {
        return getDefaultFactory().newUpdatableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new updatable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterable) newUpdatableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterable, Iterable) newUpdatableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(Iterable,
     * Iterable, Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new updatable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newUpdatableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new updatable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4,
            @Nonnull Iterable<Integer> elems5) {
        return getDefaultFactory().newUpdatableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new updatable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Iterator) newUpdatableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new updatable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(
            @Nonnull Iterator<Integer> elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }

    /**
     * Constructs a new updatable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Consumer) newUpdatableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new updatable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.IntConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newUpdatableSet(elementsSupplier);
    }

    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * int[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     */
    @Nonnull
    public static  HashIntSet newUpdatableSet(@Nonnull int[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }
    
    /**
     * Constructs a new updatable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSet(
     * Integer[]) newUpdatableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new updatable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashIntSet newUpdatableSet(@Nonnull Integer[] elements) {
        return getDefaultFactory().newUpdatableSet(elements);
    }


    /**
     * Constructs a new updatable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSetOf(
     * int) newUpdatableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new updatable singleton set of the given element
     */
    @Nonnull
    public static  HashIntSet newUpdatableSetOf(int e1) {
        return getDefaultFactory().newUpdatableSetOf(e1);
    }

    /**
     * Constructs a new updatable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSetOf(
     * int, int) newUpdatableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new updatable set of the two specified elements
     */
    @Nonnull
    public static  HashIntSet newUpdatableSetOf(int e1, int e2) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2);
    }

    /**
     * Constructs a new updatable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSetOf(
     * int, int, int) newUpdatableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new updatable set of the three specified elements
     */
    @Nonnull
    public static  HashIntSet newUpdatableSetOf(int e1, int e2, int e3) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new updatable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSetOf(
     * int, int, int, int) newUpdatableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new updatable set of the four specified elements
     */
    @Nonnull
    public static  HashIntSet newUpdatableSetOf(int e1, int e2, int e3, int e4) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new updatable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newUpdatableSetOf(int, int, int,
     * int, int, int...) newUpdatableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new updatable set of the specified elements
     */
    @Nonnull
    public static  HashIntSet newUpdatableSetOf(int e1, int e2, int e3, int e4,
            int e5, int... restElements) {
        return getDefaultFactory().newUpdatableSetOf(e1, e2, e3, e4, e5, restElements);
    }


    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     * <p>If the specified iterable is  a {@link java.util.Set}
     *  the {@code expectedSize} argument is ignored.
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterable, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, expectedSize);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable, int) newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4,
            @Nonnull Iterable<Integer> elems5, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5, expectedSize);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterator, int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterator<Integer> elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Consumer, int) newImmutableSet(elementsSupplier, expectedSize)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.IntConsumer> elementsSupplier
            , int expectedSize) {
        return getDefaultFactory().newImmutableSet(elementsSupplier, expectedSize);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * int[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(@Nonnull int[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Integer[], int) newImmutableSet(elements, expectedSize)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * @param expectedSize the expected size of the returned set
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashIntSet newImmutableSet(@Nonnull Integer[] elements, int expectedSize) {
        return getDefaultFactory().newImmutableSet(elements, expectedSize);
    }

    
    
    

    

    /**
     * Constructs a new immutable set containing the elements in the specified iterable.
     *
     * 
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterable) newImmutableSet(elements)}.
     *
     * @param elements the iterable whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of the elements of the specified iterable
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterable, Iterable) newImmutableSet(elems1, elems2)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2) {
        return getDefaultFactory().newImmutableSet(elems1, elems2);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(Iterable,
     * Iterable, Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4);
    }

    /**
     * Constructs a new immutable set which merge the elements of the specified iterables.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(Iterable, Iterable, Iterable,
     * Iterable, Iterable) newImmutableSet(elems1, elems2, elems3, elems4, elems5)}.
     *
     * @param elems1 the first source of elements for the returned set
     * @param elems2 the second source of elements for the returned set
     * @param elems3 the third source of elements for the returned set
     * @param elems4 the fourth source of elements for the returned set
     * @param elems5 the fifth source of elements for the returned set
    * 
    
     * @return a new immutable set which merge the elements of the specified iterables
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterable<Integer> elems1,
            @Nonnull Iterable<Integer> elems2,
            @Nonnull Iterable<Integer> elems3,
            @Nonnull Iterable<Integer> elems4,
            @Nonnull Iterable<Integer> elems5) {
        return getDefaultFactory().newImmutableSet(elems1, elems2, elems3, elems4, elems5);
    }

    /**
     * Constructs a new immutable set containing the elements traversed by the specified iterator.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Iterator) newImmutableSet(elements)}.
     *
     * @param elements the iterator from which elements are to be placed into the returned set
    * 
    
     * @return a new immutable set containing the elements traversed by the specified iterator
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(
            @Nonnull Iterator<Integer> elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }

    /**
     * Constructs a new immutable set of elements consumed by the callback within the given closure.
     *
     * <p>Example: TODO
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Consumer) newImmutableSet(elementsSupplier)}.
     *
     * @param elementsSupplier the function which supply mappings for the returned set via
     *        the callback passed in
    * 
    
     * @return a new immutable set of elements consumed by the callback within the given closure
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(@Nonnull
            Consumer<net.openhft.koloboke.function.IntConsumer> elementsSupplier
            ) {
        return getDefaultFactory().newImmutableSet(elementsSupplier);
    }

    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * int[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     */
    @Nonnull
    public static  HashIntSet newImmutableSet(@Nonnull int[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }
    
    /**
     * Constructs a new immutable set of elements from the given array.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSet(
     * Integer[]) newImmutableSet(elements)}.
     *
     * @param elements the array whose elements are to be placed into the returned set
    * 
    
     * @return a new immutable set of elements from the given array
     * @throws NullPointerException if {@code elements} array contain {@code null} elements
     */
    @Nonnull
    public static HashIntSet newImmutableSet(@Nonnull Integer[] elements) {
        return getDefaultFactory().newImmutableSet(elements);
    }


    /**
     * Constructs a new immutable singleton set of the given element.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSetOf(
     * int) newImmutableSetOf(e1)}.
     *
     * @param e1 the sole element
    
     * @return a new immutable singleton set of the given element
     */
    @Nonnull
    public static  HashIntSet newImmutableSetOf(int e1) {
        return getDefaultFactory().newImmutableSetOf(e1);
    }

    /**
     * Constructs a new immutable set of the two specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSetOf(
     * int, int) newImmutableSetOf(e1, e2)}.
     *
     * @param e1 the first element
     * @param e2 the second element
    
     * @return a new immutable set of the two specified elements
     */
    @Nonnull
    public static  HashIntSet newImmutableSetOf(int e1, int e2) {
        return getDefaultFactory().newImmutableSetOf(e1, e2);
    }

    /**
     * Constructs a new immutable set of the three specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSetOf(
     * int, int, int) newImmutableSetOf(e1, e2, e3)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
    
     * @return a new immutable set of the three specified elements
     */
    @Nonnull
    public static  HashIntSet newImmutableSetOf(int e1, int e2, int e3) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3);
    }

    /**
     * Constructs a new immutable set of the four specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSetOf(
     * int, int, int, int) newImmutableSetOf(e1, e2, e3, e4)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
    
     * @return a new immutable set of the four specified elements
     */
    @Nonnull
    public static  HashIntSet newImmutableSetOf(int e1, int e2, int e3, int e4) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4);
    }

    /**
     * Constructs a new immutable set of the specified elements.
     *
     * <p>This method simply delegates to {@link #getDefaultFactory()
     * }<tt>.</tt>{@link HashIntSetFactory#newImmutableSetOf(int, int, int,
     * int, int, int...) newImmutableSetOf(e1, e2, e3, e4, e5, restElements)}.
     *
     * @param e1 the first element
     * @param e2 the second element
     * @param e3 the third element
     * @param e4 the fourth element
     * @param e5 the fifth element
     * @param restElements the rest elements to be placed into the set
    
     * @return a new immutable set of the specified elements
     */
    @Nonnull
    public static  HashIntSet newImmutableSetOf(int e1, int e2, int e3, int e4,
            int e5, int... restElements) {
        return getDefaultFactory().newImmutableSetOf(e1, e2, e3, e4, e5, restElements);
    }

    private HashIntSets() {}
}

