/* Copyright 2004, 2005 Acegi Technology Pty Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.acegisecurity.ui.webapp;

import org.acegisecurity.Authentication;
import org.acegisecurity.AuthenticationException;
import org.acegisecurity.providers.UsernamePasswordAuthenticationToken;
import org.acegisecurity.ui.AbstractProcessingFilter;
import org.acegisecurity.ui.WebAuthenticationDetails;

import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;


/**
 * Processes an authentication form.
 * 
 * <p>
 * Login forms must present two parameters to this filter: a username and
 * password. The parameter names to use are contained in the static fields
 * {@link #ACEGI_SECURITY_FORM_USERNAME_KEY} and {@link
 * #ACEGI_SECURITY_FORM_PASSWORD_KEY}.
 * </p>
 * 
 * <P>
 * <B>Do not use this class directly.</B> Instead configure
 * <code>web.xml</code> to use the {@link
 * org.acegisecurity.util.FilterToBeanProxy}.
 * </p>
 *
 * @author Ben Alex
 * @author Colin Sampaleanu
 * @version $Id: AuthenticationProcessingFilter.java,v 1.13 2005/11/17 00:55:50 benalex Exp $
 */
public class AuthenticationProcessingFilter extends AbstractProcessingFilter {
    //~ Static fields/initializers =============================================

    public static final String ACEGI_SECURITY_FORM_USERNAME_KEY = "j_username";
    public static final String ACEGI_SECURITY_FORM_PASSWORD_KEY = "j_password";
    public static final String ACEGI_SECURITY_LAST_USERNAME_KEY = "ACEGI_SECURITY_LAST_USERNAME";

    //~ Methods ================================================================

    /**
     * This filter by default responds to <code>/j_acegi_security_check</code>.
     *
     * @return the default
     */
    public String getDefaultFilterProcessesUrl() {
        return "/j_acegi_security_check";
    }

    public Authentication attemptAuthentication(HttpServletRequest request)
        throws AuthenticationException {
        String username = obtainUsername(request);
        String password = obtainPassword(request);

        if (username == null) {
            username = "";
        }

        if (password == null) {
            password = "";
        }

        UsernamePasswordAuthenticationToken authRequest = new UsernamePasswordAuthenticationToken(username,
                password);

        // Allow subclasses to set the "details" property
        setDetails(request, authRequest);

        // Place the last username attempted into HttpSession for views
        request.getSession().setAttribute(ACEGI_SECURITY_LAST_USERNAME_KEY,
            username);

        return this.getAuthenticationManager().authenticate(authRequest);
    }

    public void init(FilterConfig filterConfig) throws ServletException {}

    /**
     * Provided so that subclasses may configure what is put into the
     * authentication request's details property. The default implementation
     * simply constructs {@link WebAuthenticationDetails}.
     *
     * @param request that an authentication request is being created for
     * @param authRequest the authentication request object that should have
     *        its details set
     */
    protected void setDetails(HttpServletRequest request,
        UsernamePasswordAuthenticationToken authRequest) {
        authRequest.setDetails(new WebAuthenticationDetails(request));
    }

    /**
     * Enables subclasses to override the composition of the password, such as
     * by including additional values and a separator.
     * 
     * <p>
     * This might be used for example if a postcode/zipcode was required in
     * addition to the password. A delimiter such as a pipe (|) should be used
     * to separate the password and extended value(s). The
     * <code>AuthenticationDao</code> will need to generate the expected
     * password in a corresponding manner.
     * </p>
     *
     * @param request so that request attributes can be retrieved
     *
     * @return the password that will be presented in the
     *         <code>Authentication</code> request token to the
     *         <code>AuthenticationManager</code>
     */
    protected String obtainPassword(HttpServletRequest request) {
        return request.getParameter(ACEGI_SECURITY_FORM_PASSWORD_KEY);
    }

    /**
     * Enables subclasses to override the composition of the username, such as
     * by including additional values and a separator.
     *
     * @param request so that request attributes can be retrieved
     *
     * @return the username that will be presented in the
     *         <code>Authentication</code> request token to the
     *         <code>AuthenticationManager</code>
     */
    protected String obtainUsername(HttpServletRequest request) {
        return request.getParameter(ACEGI_SECURITY_FORM_USERNAME_KEY);
    }
}
