/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.io.buffer;

import io.netty.buffer.Unpooled;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.LockSupport;
import org.apache.activemq.artemis.api.core.ActiveMQBuffer;
import org.apache.activemq.artemis.api.core.ActiveMQInterruptedException;
import org.apache.activemq.artemis.core.buffers.impl.ChannelBufferWrapper;
import org.apache.activemq.artemis.core.io.IOCallback;
import org.apache.activemq.artemis.core.io.buffer.TimedBufferObserver;
import org.apache.activemq.artemis.core.journal.EncodingSupport;
import org.apache.activemq.artemis.journal.ActiveMQJournalLogger;

public final class TimedBuffer {
    private static final int MAX_CHECKS_ON_SLEEP = 20;
    private TimedBufferObserver bufferObserver;
    private final Semaphore spinLimiter = new Semaphore(1);
    private CheckTimer timerRunnable;
    private final int bufferSize;
    private final ActiveMQBuffer buffer;
    private int bufferLimit = 0;
    private List<IOCallback> callbacks;
    private final int timeout;
    private volatile boolean pendingSync = false;
    private Thread timerThread;
    private volatile boolean started;
    private boolean delayFlush;
    private final boolean logRates;
    private final AtomicLong bytesFlushed = new AtomicLong(0L);
    private final AtomicLong flushesDone = new AtomicLong(0L);
    private Timer logRatesTimer;
    private TimerTask logRatesTimerTask;
    private boolean useSleep = true;
    private boolean spinning = false;

    public TimedBuffer(int size, int timeout, boolean logRates) {
        this.bufferSize = size;
        this.logRates = logRates;
        if (logRates) {
            this.logRatesTimer = new Timer(true);
        }
        this.buffer = new ChannelBufferWrapper(Unpooled.wrappedBuffer((ByteBuffer)ByteBuffer.allocateDirect(size)));
        this.buffer.clear();
        this.bufferLimit = 0;
        this.callbacks = new ArrayList<IOCallback>();
        this.timeout = timeout;
    }

    public synchronized void start() {
        if (this.started) {
            return;
        }
        try {
            this.spinLimiter.acquire();
        }
        catch (InterruptedException e) {
            throw new ActiveMQInterruptedException((Throwable)e);
        }
        this.timerRunnable = new CheckTimer();
        this.timerThread = new Thread((Runnable)this.timerRunnable, "activemq-buffer-timeout");
        this.timerThread.start();
        if (this.logRates) {
            this.logRatesTimerTask = new LogRatesTimerTask();
            this.logRatesTimer.scheduleAtFixedRate(this.logRatesTimerTask, 2000L, 2000L);
        }
        this.started = true;
    }

    public void stop() {
        if (!this.started) {
            return;
        }
        this.flush();
        this.bufferObserver = null;
        this.timerRunnable.close();
        this.spinLimiter.release();
        if (this.logRates) {
            this.logRatesTimerTask.cancel();
        }
        while (this.timerThread.isAlive()) {
            try {
                this.timerThread.join();
            }
            catch (InterruptedException e) {
                throw new ActiveMQInterruptedException((Throwable)e);
            }
        }
        this.started = false;
    }

    public synchronized void setObserver(TimedBufferObserver observer) {
        if (this.bufferObserver != null) {
            this.flush();
        }
        this.bufferObserver = observer;
    }

    public synchronized boolean checkSize(int sizeChecked) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        if (sizeChecked > this.bufferSize) {
            throw new IllegalStateException("Can't write records bigger than the bufferSize(" + this.bufferSize + ") on the journal");
        }
        if (this.bufferLimit == 0 || this.buffer.writerIndex() + sizeChecked > this.bufferLimit) {
            this.flush();
            this.delayFlush = true;
            int remainingInFile = this.bufferObserver.getRemainingBytes();
            if (sizeChecked > remainingInFile) {
                return false;
            }
            this.bufferLimit = Math.min(remainingInFile, this.bufferSize);
            return true;
        }
        this.delayFlush = true;
        return true;
    }

    public synchronized void addBytes(ActiveMQBuffer bytes, boolean sync, IOCallback callback) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        this.delayFlush = false;
        int readableBytes = bytes.readableBytes();
        int writerIndex = this.buffer.writerIndex();
        this.buffer.setBytes(writerIndex, bytes, bytes.readerIndex(), readableBytes);
        this.buffer.writerIndex(writerIndex + readableBytes);
        this.callbacks.add(callback);
        if (sync) {
            this.pendingSync = true;
            this.startSpin();
        }
    }

    public synchronized void addBytes(EncodingSupport bytes, boolean sync, IOCallback callback) {
        if (!this.started) {
            throw new IllegalStateException("TimedBuffer is not started");
        }
        this.delayFlush = false;
        bytes.encode(this.buffer);
        this.callbacks.add(callback);
        if (sync) {
            this.pendingSync = true;
            this.startSpin();
        }
    }

    public void flush() {
        this.flush(false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush(boolean force) {
        TimedBuffer timedBuffer = this;
        synchronized (timedBuffer) {
            if (!this.started) {
                throw new IllegalStateException("TimedBuffer is not started");
            }
            if ((force || !this.delayFlush) && this.buffer.writerIndex() > 0) {
                int pos = this.buffer.writerIndex();
                if (this.logRates) {
                    this.bytesFlushed.addAndGet(pos);
                }
                ByteBuffer bufferToFlush = this.bufferObserver.newBuffer(this.bufferSize, pos);
                bufferToFlush.limit(pos);
                this.buffer.getBytes(0, bufferToFlush);
                this.bufferObserver.flushBuffer(bufferToFlush, this.pendingSync, this.callbacks);
                this.stopSpin();
                this.pendingSync = false;
                this.callbacks = new ArrayList<IOCallback>();
                this.buffer.clear();
                this.bufferLimit = 0;
                this.flushesDone.incrementAndGet();
            }
        }
    }

    protected void sleep(long sleepNanos) {
        LockSupport.parkNanos(sleepNanos);
    }

    protected void stopSpin() {
        if (this.spinning) {
            try {
                this.spinLimiter.acquire();
            }
            catch (InterruptedException e) {
                throw new ActiveMQInterruptedException((Throwable)e);
            }
            this.spinning = false;
        }
    }

    protected void startSpin() {
        if (!this.spinning) {
            this.spinLimiter.release();
            this.spinning = true;
        }
    }

    private class CheckTimer
    implements Runnable {
        private volatile boolean closed = false;
        int checks = 0;
        int failedChecks = 0;

        private CheckTimer() {
        }

        @Override
        public void run() {
            long lastFlushTime = System.nanoTime();
            while (!this.closed) {
                if (TimedBuffer.this.pendingSync) {
                    if (TimedBuffer.this.useSleep) {
                        lastFlushTime = System.nanoTime();
                        TimedBuffer.this.flush();
                    } else if (TimedBuffer.this.bufferObserver != null && System.nanoTime() > lastFlushTime + (long)TimedBuffer.this.timeout) {
                        lastFlushTime = System.nanoTime();
                        TimedBuffer.this.flush();
                    }
                }
                long timeFromTheLastFlush = System.nanoTime() - lastFlushTime;
                long timeToSleep = (long)TimedBuffer.this.timeout - timeFromTheLastFlush;
                if (timeToSleep > 0L) {
                    this.sleepIfPossible(timeToSleep);
                }
                try {
                    TimedBuffer.this.spinLimiter.acquire();
                    Thread.yield();
                    TimedBuffer.this.spinLimiter.release();
                }
                catch (InterruptedException e) {
                    throw new ActiveMQInterruptedException((Throwable)e);
                }
            }
        }

        private void sleepIfPossible(long nanosToSleep) {
            if (TimedBuffer.this.useSleep) {
                try {
                    long startSleep = System.nanoTime();
                    TimedBuffer.this.sleep(nanosToSleep);
                    long elapsedSleep = System.nanoTime() - startSleep;
                    if (this.checks < 20) {
                        if ((double)elapsedSleep > (double)nanosToSleep * 1.5) {
                            ++this.failedChecks;
                        }
                        if (++this.checks >= 20 && (double)this.failedChecks > 10.0) {
                            ActiveMQJournalLogger.LOGGER.debug("LockSupport.parkNanos with nano seconds is not working as expected, Your kernel possibly doesn't support real time. the Journal TimedBuffer will spin for timeouts");
                            TimedBuffer.this.useSleep = false;
                        }
                    }
                }
                catch (Exception e) {
                    TimedBuffer.this.useSleep = false;
                    ActiveMQJournalLogger.LOGGER.warn(e.getMessage() + ", disabling sleep on TimedBuffer, using spin now", e);
                }
            }
        }

        public void close() {
            this.closed = true;
        }
    }

    private class LogRatesTimerTask
    extends TimerTask {
        private boolean closed;
        private long lastExecution;
        private long lastBytesFlushed;
        private long lastFlushesDone;

        private LogRatesTimerTask() {
        }

        @Override
        public synchronized void run() {
            if (!this.closed) {
                long now = System.currentTimeMillis();
                long bytesF = TimedBuffer.this.bytesFlushed.get();
                long flushesD = TimedBuffer.this.flushesDone.get();
                if (this.lastExecution != 0L) {
                    double rate = 1000.0 * (double)(bytesF - this.lastBytesFlushed) / (double)(now - this.lastExecution);
                    ActiveMQJournalLogger.LOGGER.writeRate(rate, (long)(rate / 1048576.0));
                    double flushRate = 1000.0 * (double)(flushesD - this.lastFlushesDone) / (double)(now - this.lastExecution);
                    ActiveMQJournalLogger.LOGGER.flushRate(flushRate);
                }
                this.lastExecution = now;
                this.lastBytesFlushed = bytesF;
                this.lastFlushesDone = flushesD;
            }
        }

        @Override
        public synchronized boolean cancel() {
            this.closed = true;
            return super.cancel();
        }
    }
}

