/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.cxf.jaxrs;

import java.lang.ref.SoftReference;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.ws.rs.core.Response;

import org.apache.camel.CamelException;
import org.apache.camel.Exchange;
import org.apache.camel.Message;
import org.apache.camel.component.cxf.CxfConstants;
import org.apache.camel.component.cxf.CxfOperationException;
import org.apache.camel.component.cxf.util.CxfEndpointUtils;
import org.apache.camel.impl.DefaultProducer;
import org.apache.camel.util.LRUCache;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.cxf.jaxrs.JAXRSServiceFactoryBean;
import org.apache.cxf.jaxrs.client.Client;
import org.apache.cxf.jaxrs.client.JAXRSClientFactoryBean;
import org.apache.cxf.jaxrs.client.WebClient;

/**
 * CxfRsProducer binds a Camel exchange to a CXF exchange, acts as a CXF
 * JAXRS client, it will turn the normal Object invocation to a RESTful request
 * according to resource annotation.  Any response will be bound to Camel exchange.
 */
public class CxfRsProducer extends DefaultProducer {

    private static final Log LOG = LogFactory.getLog(CxfRsProducer.class);

    private boolean throwException;
    
    // using a cache of factory beans instead of setting the address of a single cfb
    // to avoid concurrent issues
    private ClientFactoryBeanCache clientFactoryBeanCache;
    
    public CxfRsProducer(CxfRsEndpoint endpoint) {
        super(endpoint);
        this.throwException = endpoint.isThrowExceptionOnFailure();
        clientFactoryBeanCache = new ClientFactoryBeanCache(endpoint.getMaxClientCacheSize());
    }

    public void process(Exchange exchange) throws Exception {
        if (LOG.isTraceEnabled()) {
            LOG.trace("Process exchange: " + exchange);
        }

        Message inMessage = exchange.getIn();
        Boolean httpClientAPI = inMessage.getHeader(CxfConstants.CAMEL_CXF_RS_USING_HTTP_API, Boolean.class);
        // set the value with endpoint's option
        if (httpClientAPI == null) {
            httpClientAPI = ((CxfRsEndpoint) getEndpoint()).isHttpClientAPI();
        }
        if (httpClientAPI.booleanValue()) {
            invokeHttpClient(exchange);
        } else {
            invokeProxyClient(exchange);
        }
    }

    @SuppressWarnings("unchecked")
    protected void invokeHttpClient(Exchange exchange) throws Exception {
        Message inMessage = exchange.getIn();
        JAXRSClientFactoryBean cfb = clientFactoryBeanCache.get(CxfEndpointUtils
            .getEffectiveAddress(exchange, ((CxfRsEndpoint)getEndpoint()).getAddress()));
        
        WebClient client = cfb.createWebClient();
        String httpMethod = inMessage.getHeader(Exchange.HTTP_METHOD, String.class);
        Class responseClass = inMessage.getHeader(CxfConstants.CAMEL_CXF_RS_RESPONSE_CLASS, Class.class);
        Type genericType = inMessage.getHeader(CxfConstants.CAMEL_CXF_RS_RESPONSE_GENERIC_TYPE, Type.class);
        String path = inMessage.getHeader(Exchange.HTTP_PATH, String.class);

        if (LOG.isTraceEnabled()) {
            LOG.trace("HTTP method = " + httpMethod);
            LOG.trace("path = " + path);
            LOG.trace("responseClass = " + responseClass);
        }

        // set the path
        if (path != null) {
            client.path(path);
        }

        CxfRsEndpoint cxfRsEndpoint = (CxfRsEndpoint) getEndpoint();
        // check if there is a query map in the message header
        Map<String, String> maps = inMessage.getHeader(CxfConstants.CAMEL_CXF_RS_QUERY_MAP, Map.class);
        if (maps == null) {
            maps = cxfRsEndpoint.getParameters();
        }
        if (maps != null) {
            for (Map.Entry<String, String> entry : maps.entrySet()) {
                client.query(entry.getKey(), entry.getValue());
            }
        }

        CxfRsBinding binding = cxfRsEndpoint.getBinding();

        // set the body
        Object body = null;
        if (!"GET".equals(httpMethod)) {
            // need to check the request object.           
            body = binding.bindCamelMessageBodyToRequestBody(inMessage, exchange);
            if (LOG.isTraceEnabled()) {
                LOG.trace("Request body = " + body);
            }
        }

        // set headers
        client.headers(binding.bindCamelHeadersToRequestHeaders(inMessage.getHeaders(), exchange));

        // invoke the client
        Object response = null;
        if (responseClass == null || Response.class.equals(responseClass)) {
            response = client.invoke(httpMethod, body);
        } else {
            if (Collection.class.isAssignableFrom(responseClass)) {
                if (genericType instanceof ParameterizedType) {
                    // Get the collection member type first
                    Type[] actualTypeArguments = ((ParameterizedType) genericType).getActualTypeArguments();
                    response = client.invokeAndGetCollection(httpMethod, body, (Class) actualTypeArguments[0]);
                } else {
                    throw new CamelException("Can't find the Collection member type");
                }
            } else {
                response = client.invoke(httpMethod, body, responseClass);
            }
        }
        //Throw exception on a response > 207
        //http://en.wikipedia.org/wiki/List_of_HTTP_status_codes
        if (throwException) {
            if (response instanceof Response) {
                Integer respCode = ((Response) response).getStatus();
                if (respCode > 207) {
                    throw populateCxfRsProducerException(exchange, (Response) response, respCode);
                }
            }
        }
        // set response
        if (exchange.getPattern().isOutCapable()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("Response body = " + response);
            }
            exchange.getOut().setBody(binding.bindResponseToCamelBody(response, exchange));
            exchange.getOut().setHeaders(binding.bindResponseHeadersToCamelHeaders(response, exchange));
        }
    }

    protected void invokeProxyClient(Exchange exchange) throws Exception {
        Message inMessage = exchange.getIn();
        Object[] varValues = inMessage.getHeader(CxfConstants.CAMEL_CXF_RS_VAR_VALUES, Object[].class);
        String methodName = inMessage.getHeader(CxfConstants.OPERATION_NAME, String.class);
        Client target = null;
        
        JAXRSClientFactoryBean cfb = clientFactoryBeanCache.get(CxfEndpointUtils
                                   .getEffectiveAddress(exchange, ((CxfRsEndpoint)getEndpoint()).getAddress()));
        
        if (varValues == null) {
            target = cfb.create();
        } else {
            target = cfb.createWithValues(varValues);
        }
        // find out the method which we want to invoke
        JAXRSServiceFactoryBean sfb = cfb.getServiceFactory();
        sfb.getResourceClasses();
        Object[] parameters = inMessage.getBody(Object[].class);
        // get the method
        Method method = findRightMethod(sfb.getResourceClasses(), methodName, getParameterTypes(parameters));
        // Will send out the message to
        // Need to deal with the sub resource class
        Object response = method.invoke(target, parameters);

        if (throwException) {
            if (response instanceof Response) {
                Integer respCode = ((Response) response).getStatus();
                if (respCode > 207) {
                    throw populateCxfRsProducerException(exchange, (Response) response, respCode);
                }
            }
        }
        if (exchange.getPattern().isOutCapable()) {
            exchange.getOut().setBody(response);
        }
    }

    private Method findRightMethod(List<Class<?>> resourceClasses, String methodName, Class[] parameterTypes) throws NoSuchMethodException {
        Method answer = null;
        for (Class<?> clazz : resourceClasses) {
            try {
                answer = clazz.getMethod(methodName, parameterTypes);
            } catch (NoSuchMethodException ex) {
                // keep looking 
            } catch (SecurityException ex) {
                // keep looking
            }
            if (answer != null) {
                return answer;
            }
        }
        throw new NoSuchMethodException("Can find the method " + methodName + "withe these parameter " + arrayToString(parameterTypes));
    }

    private Class<?>[] getParameterTypes(Object[] objects) {
        Class<?>[] answer = new Class[objects.length];
        int i = 0;
        for (Object obj : objects) {
            answer[i] = obj.getClass();
            i++;
        }
        return answer;
    }

    private String arrayToString(Object[] array) {
        StringBuilder buffer = new StringBuilder("[");
        for (Object obj : array) {
            if (buffer.length() > 2) {
                buffer.append(",");
            }
            buffer.append(obj.toString());
        }
        buffer.append("]");
        return buffer.toString();
    }

    protected CxfOperationException populateCxfRsProducerException(Exchange exchange, Response response, int responseCode) {
        CxfOperationException exception;
        String uri = exchange.getFromEndpoint().getEndpointUri();
        String statusText = Response.Status.fromStatusCode(responseCode).toString();
        Map<String, String> headers = parseResponseHeaders(response, exchange);
        String copy = response.toString();
        LOG.warn(headers);
        if (responseCode >= 300 && responseCode < 400) {
            String redirectLocation;
            if (response.getMetadata().getFirst("Location") != null) {
                redirectLocation = response.getMetadata().getFirst("location").toString();
                exception = new CxfOperationException(uri, responseCode, statusText, redirectLocation, headers, copy);
            } else {
                //no redirect location
                exception = new CxfOperationException(uri, responseCode, statusText, null, headers, copy);
            }
        } else {
            //internal server error(error code 500)
            exception = new CxfOperationException(uri, responseCode, statusText, null, headers, copy);
        }

        return exception;
    }

    protected Map<String, String> parseResponseHeaders(Object response, Exchange camelExchange) {

        Map<String, String> answer = new HashMap<String, String>();
        if (response instanceof Response) {

            for (Map.Entry<String, List<Object>> entry : ((Response) response).getMetadata().entrySet()) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Parse external header " + entry.getKey() + "=" + entry.getValue());
                }
                LOG.info("Parse external header " + entry.getKey() + "=" + entry.getValue());
                answer.put(entry.getKey(), entry.getValue().get(0).toString());
            }
        }

        return answer;
    }
    
    /**
     * Cache contains {@link org.apache.cxf.jaxrs.client.JAXRSClientFactoryBean}
     */
    private class ClientFactoryBeanCache {
        private LRUCache<String, SoftReference<JAXRSClientFactoryBean>> cache;    
        
        public ClientFactoryBeanCache(final int maxCacheSize) {
            this.cache = new LRUCache<String, SoftReference<JAXRSClientFactoryBean>>(maxCacheSize);
        }

        public JAXRSClientFactoryBean get(String address) throws Exception {
            JAXRSClientFactoryBean retval = null;
            synchronized (cache) {
                SoftReference<JAXRSClientFactoryBean> ref = cache.get(address);
                
                if (ref != null) {
                    retval = ref.get();
                }

                if (retval == null) {
                    retval = ((CxfRsEndpoint)getEndpoint()).createJAXRSClientFactoryBean(address);
                    
                    cache.put(address, new SoftReference<JAXRSClientFactoryBean>(retval));
                    
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Created client factory bean and add to cache for address '" + address + "'");
                    }
                    
                } else {
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Retrieved client factory bean from cache for address '" + address + "'");
                    }
                }
            }
            return retval;
        }
    }
}
