/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.tools;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.compaction.SSTableSplitter;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.SSTableDeletingTask;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.service.CassandraDaemon;
import org.apache.cassandra.tools.BulkLoader;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.ParseException;

public class StandaloneSplitter {
    public static final int DEFAULT_SSTABLE_SIZE = 50;
    private static final String TOOL_NAME = "sstablessplit";
    private static final String VERBOSE_OPTION = "verbose";
    private static final String DEBUG_OPTION = "debug";
    private static final String HELP_OPTION = "help";
    private static final String NO_SNAPSHOT_OPTION = "no-snapshot";
    private static final String SIZE_OPTION = "size";

    public static void main(String[] args) throws IOException {
        Options options = Options.parseArgs(args);
        try {
            DatabaseDescriptor.loadSchemas();
            String ksName = null;
            String cfName = null;
            HashMap<Descriptor, HashSet<Component>> parsedFilenames = new HashMap<Descriptor, HashSet<Component>>();
            for (String filename : options.filenames) {
                File file = new File(filename);
                if (!file.exists()) {
                    System.out.println("Skipping inexisting file " + file);
                    continue;
                }
                Pair<Descriptor, Component> pair = SSTable.tryComponentFromFilename(file.getParentFile(), file.getName());
                if (pair == null) {
                    System.out.println("Skipping non sstable file " + file);
                    continue;
                }
                Descriptor desc = (Descriptor)pair.left;
                if (ksName == null) {
                    ksName = desc.ksname;
                } else if (!ksName.equals(desc.ksname)) {
                    throw new IllegalArgumentException("All sstables must be part of the same keyspace");
                }
                if (cfName == null) {
                    cfName = desc.cfname;
                } else if (!cfName.equals(desc.cfname)) {
                    throw new IllegalArgumentException("All sstables must be part of the same column family");
                }
                HashSet<Component> components = new HashSet<Component>(Arrays.asList(Component.DATA, Component.PRIMARY_INDEX, Component.FILTER, Component.COMPRESSION_INFO, Component.STATS));
                Iterator iter = components.iterator();
                while (iter.hasNext()) {
                    Component component = (Component)iter.next();
                    if (new File(desc.filenameFor(component)).exists()) continue;
                    iter.remove();
                }
                parsedFilenames.put(desc, components);
            }
            if (ksName == null || cfName == null) {
                System.err.println("No valid sstables to split");
                System.exit(1);
            }
            Keyspace keyspace = Keyspace.openWithoutSSTables(ksName);
            ColumnFamilyStore cfs = keyspace.getColumnFamilyStore(cfName);
            String snapshotName = "pre-split-" + System.currentTimeMillis();
            ArrayList<SSTableReader> sstables = new ArrayList<SSTableReader>();
            for (Map.Entry fn : parsedFilenames.entrySet()) {
                try {
                    SSTableReader sstable = SSTableReader.openNoValidation((Descriptor)fn.getKey(), (Set)fn.getValue(), cfs.metadata);
                    sstables.add(sstable);
                    if (!options.snapshot) continue;
                    File snapshotDirectory = Directories.getSnapshotDirectory(sstable.descriptor, snapshotName);
                    sstable.createLinks(snapshotDirectory.getPath());
                }
                catch (Exception e) {
                    System.err.println(String.format("Error Loading %s: %s", fn.getKey(), e.getMessage()));
                    if (!options.debug) continue;
                    e.printStackTrace(System.err);
                }
            }
            if (options.snapshot) {
                System.out.println(String.format("Pre-split sstables snapshotted into snapshot %s", snapshotName));
            }
            cfs.getDataTracker().markCompacting(sstables);
            for (SSTableReader sstable : sstables) {
                try {
                    new SSTableSplitter(cfs, sstable, options.sizeInMB).split();
                    sstable.markObsolete();
                    sstable.releaseReference();
                }
                catch (Exception e) {
                    System.err.println(String.format("Error splitting %s: %s", sstable, e.getMessage()));
                    if (!options.debug) continue;
                    e.printStackTrace(System.err);
                }
            }
            SSTableDeletingTask.waitForDeletions();
            System.exit(0);
        }
        catch (Exception e) {
            System.err.println(e.getMessage());
            if (options.debug) {
                e.printStackTrace(System.err);
            }
            System.exit(1);
        }
    }

    static {
        CassandraDaemon.initLog4j();
    }

    private static class Options {
        public final List<String> filenames;
        public boolean debug;
        public boolean verbose;
        public boolean snapshot;
        public int sizeInMB;

        private Options(List<String> filenames) {
            this.filenames = filenames;
        }

        public static Options parseArgs(String[] cmdArgs) {
            GnuParser parser = new GnuParser();
            BulkLoader.CmdLineOptions options = Options.getCmdLineOptions();
            try {
                String[] args;
                CommandLine cmd = parser.parse((org.apache.commons.cli.Options)options, cmdArgs, false);
                if (cmd.hasOption(StandaloneSplitter.HELP_OPTION)) {
                    Options.printUsage(options);
                    System.exit(0);
                }
                if ((args = cmd.getArgs()).length == 0) {
                    System.err.println("No sstables to split");
                    Options.printUsage(options);
                    System.exit(1);
                }
                Options opts = new Options(Arrays.asList(args));
                opts.debug = cmd.hasOption(StandaloneSplitter.DEBUG_OPTION);
                opts.verbose = cmd.hasOption(StandaloneSplitter.VERBOSE_OPTION);
                opts.snapshot = !cmd.hasOption(StandaloneSplitter.NO_SNAPSHOT_OPTION);
                opts.sizeInMB = 50;
                if (cmd.hasOption(StandaloneSplitter.SIZE_OPTION)) {
                    opts.sizeInMB = Integer.valueOf(cmd.getOptionValue(StandaloneSplitter.SIZE_OPTION));
                }
                return opts;
            }
            catch (ParseException e) {
                Options.errorMsg(e.getMessage(), options);
                return null;
            }
        }

        private static void errorMsg(String msg, BulkLoader.CmdLineOptions options) {
            System.err.println(msg);
            Options.printUsage(options);
            System.exit(1);
        }

        private static BulkLoader.CmdLineOptions getCmdLineOptions() {
            BulkLoader.CmdLineOptions options = new BulkLoader.CmdLineOptions();
            options.addOption(null, StandaloneSplitter.DEBUG_OPTION, "display stack traces");
            options.addOption("v", StandaloneSplitter.VERBOSE_OPTION, "verbose output");
            options.addOption("h", StandaloneSplitter.HELP_OPTION, "display this help message");
            options.addOption(null, StandaloneSplitter.NO_SNAPSHOT_OPTION, "don't snapshot the sstables before splitting");
            options.addOption("s", StandaloneSplitter.SIZE_OPTION, StandaloneSplitter.SIZE_OPTION, "maximum size in MB for the output sstables (default: 50)");
            return options;
        }

        public static void printUsage(BulkLoader.CmdLineOptions options) {
            String usage = String.format("%s [options] <filename> [<filename>]*", StandaloneSplitter.TOOL_NAME);
            StringBuilder header = new StringBuilder();
            header.append("--\n");
            header.append("Split the provided sstables files in sstables of maximum provided file size (see option --size).");
            header.append("\n--\n");
            header.append("Options are:");
            new HelpFormatter().printHelp(usage, header.toString(), (org.apache.commons.cli.Options)options, "");
        }
    }
}

