/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.session.throttling;

import com.datastax.oss.driver.api.core.RequestThrottlingException;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverExecutionProfile;
import com.datastax.oss.driver.api.core.context.DriverContext;
import com.datastax.oss.driver.api.core.session.throttling.RequestThrottler;
import com.datastax.oss.driver.api.core.session.throttling.Throttled;
import com.datastax.oss.driver.shaded.guava.common.annotations.VisibleForTesting;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.util.Deque;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicInteger;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class ConcurrencyLimitingRequestThrottler
implements RequestThrottler {
    private static final Logger LOG = LoggerFactory.getLogger(ConcurrencyLimitingRequestThrottler.class);
    private final String logPrefix;
    private final int maxConcurrentRequests;
    private final int maxQueueSize;
    private final AtomicInteger concurrentRequests = new AtomicInteger(0);
    private final Deque<Throttled> queue = new ConcurrentLinkedDeque<Throttled>();
    private final AtomicInteger queueSize = new AtomicInteger(0);
    private volatile boolean closed = false;

    public ConcurrencyLimitingRequestThrottler(DriverContext context) {
        this.logPrefix = context.getSessionName();
        DriverExecutionProfile config = context.getConfig().getDefaultProfile();
        this.maxConcurrentRequests = config.getInt(DefaultDriverOption.REQUEST_THROTTLER_MAX_CONCURRENT_REQUESTS);
        this.maxQueueSize = config.getInt(DefaultDriverOption.REQUEST_THROTTLER_MAX_QUEUE_SIZE);
        LOG.debug("[{}] Initializing with maxConcurrentRequests = {}, maxQueueSize = {}", new Object[]{this.logPrefix, this.maxConcurrentRequests, this.maxQueueSize});
    }

    @Override
    public void register(@NonNull Throttled request) {
        int newQueueSize;
        if (this.closed) {
            LOG.trace("[{}] Rejecting request after shutdown", (Object)this.logPrefix);
            ConcurrencyLimitingRequestThrottler.fail(request, "The session is shutting down");
            return;
        }
        if (this.queueSize.get() == 0) {
            int newConcurrent = this.concurrentRequests.incrementAndGet();
            if (newConcurrent <= this.maxConcurrentRequests) {
                LOG.trace("[{}] Starting newly registered request", (Object)this.logPrefix);
                request.onThrottleReady(false);
                return;
            }
            this.concurrentRequests.decrementAndGet();
        }
        if ((newQueueSize = this.queueSize.incrementAndGet()) <= this.maxQueueSize) {
            LOG.trace("[{}] Enqueuing request", (Object)this.logPrefix);
            this.queue.offer(request);
            if (this.closed && this.queue.remove(request)) {
                this.queueSize.decrementAndGet();
                LOG.trace("[{}] Rejecting late request after shutdown", (Object)this.logPrefix);
                ConcurrencyLimitingRequestThrottler.fail(request, "The session is shutting down");
            }
        } else {
            LOG.trace("[{}] Rejecting request because of full queue", (Object)this.logPrefix);
            this.queueSize.decrementAndGet();
            ConcurrencyLimitingRequestThrottler.fail(request, String.format("The session has reached its maximum capacity (concurrent requests: %d, queue size: %d)", this.maxConcurrentRequests, this.maxQueueSize));
        }
    }

    @Override
    public void signalSuccess(@NonNull Throttled request) {
        Throttled nextRequest = this.onRequestDoneAndDequeNext();
        if (nextRequest != null) {
            nextRequest.onThrottleReady(true);
        }
    }

    @Override
    public void signalError(@NonNull Throttled request, @NonNull Throwable error) {
        this.signalSuccess(request);
    }

    @Override
    public void signalTimeout(@NonNull Throttled request) {
        Throttled nextRequest = null;
        if (!this.closed) {
            if (this.queue.remove(request)) {
                this.queueSize.decrementAndGet();
                LOG.trace("[{}] Removing timed out request from the queue", (Object)this.logPrefix);
            } else {
                nextRequest = this.onRequestDoneAndDequeNext();
            }
        }
        if (nextRequest != null) {
            nextRequest.onThrottleReady(true);
        }
    }

    @Override
    public void signalCancel(@NonNull Throttled request) {
        Throttled nextRequest = null;
        if (!this.closed) {
            if (this.queue.remove(request)) {
                this.queueSize.decrementAndGet();
                LOG.trace("[{}] Removing cancelled request from the queue", (Object)this.logPrefix);
            } else {
                nextRequest = this.onRequestDoneAndDequeNext();
            }
        }
        if (nextRequest != null) {
            nextRequest.onThrottleReady(true);
        }
    }

    @Nullable
    private Throttled onRequestDoneAndDequeNext() {
        if (!this.closed) {
            Throttled nextRequest = this.queue.poll();
            if (nextRequest == null) {
                this.concurrentRequests.decrementAndGet();
            } else {
                this.queueSize.decrementAndGet();
                LOG.trace("[{}] Starting dequeued request", (Object)this.logPrefix);
                return nextRequest;
            }
        }
        return null;
    }

    @Override
    public void close() {
        Throttled request;
        this.closed = true;
        LOG.debug("[{}] Rejecting {} queued requests after shutdown", (Object)this.logPrefix, (Object)this.queueSize.get());
        while ((request = this.queue.poll()) != null) {
            this.queueSize.decrementAndGet();
            ConcurrencyLimitingRequestThrottler.fail(request, "The session is shutting down");
        }
    }

    public int getQueueSize() {
        return this.queueSize.get();
    }

    @VisibleForTesting
    int getConcurrentRequests() {
        return this.concurrentRequests.get();
    }

    @VisibleForTesting
    Deque<Throttled> getQueue() {
        return this.queue;
    }

    private static void fail(Throttled request, String message) {
        request.onThrottleFailure(new RequestThrottlingException(message));
    }
}

