/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import java.util.Arrays;
import org.apache.commons.math3.exception.ConvergenceException;
import org.apache.commons.math3.exception.DimensionMismatchException;
import org.apache.commons.math3.exception.util.LocalizedFormats;
import org.apache.commons.math3.linear.ArrayRealVector;
import org.apache.commons.math3.linear.BlockRealMatrix;
import org.apache.commons.math3.linear.DecompositionSolver;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.linear.RealVector;
import org.apache.commons.math3.linear.SingularMatrixException;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.MathArrays;

public class PivotingQRDecomposition {
    private double[][] qr;
    private double[] rDiag;
    private RealMatrix cachedQ;
    private RealMatrix cachedQT;
    private RealMatrix cachedR;
    private RealMatrix cachedH;
    private int[] permutation;
    private int rank;
    private double[] beta;

    public boolean isSingular() {
        return this.rank != this.qr[0].length;
    }

    public int getRank() {
        return this.rank;
    }

    public int[] getOrder() {
        return MathArrays.copyOf(this.permutation);
    }

    public PivotingQRDecomposition(RealMatrix matrix) throws ConvergenceException {
        this(matrix, 1.0E-16, true);
    }

    public PivotingQRDecomposition(RealMatrix matrix, boolean allowPivot) throws ConvergenceException {
        this(matrix, 1.0E-16, allowPivot);
    }

    public PivotingQRDecomposition(RealMatrix matrix, double qrRankingThreshold, boolean allowPivot) throws ConvergenceException {
        int k;
        int rows = matrix.getRowDimension();
        int cols = matrix.getColumnDimension();
        this.qr = matrix.getData();
        this.rDiag = new double[cols];
        this.beta = new double[cols];
        this.permutation = new int[cols];
        this.cachedQ = null;
        this.cachedQT = null;
        this.cachedR = null;
        this.cachedH = null;
        for (k = 0; k < cols; ++k) {
            this.permutation[k] = k;
        }
        for (k = 0; k < cols; ++k) {
            double betak;
            int i;
            int nextColumn = -1;
            double ak2 = Double.NEGATIVE_INFINITY;
            if (allowPivot) {
                for (i = k; i < cols; ++i) {
                    double norm2 = 0.0;
                    for (int j = k; j < rows; ++j) {
                        double aki = this.qr[j][this.permutation[i]];
                        norm2 += aki * aki;
                    }
                    if (Double.isInfinite(norm2) || Double.isNaN(norm2)) {
                        throw new ConvergenceException(LocalizedFormats.UNABLE_TO_PERFORM_QR_DECOMPOSITION_ON_JACOBIAN, rows, cols);
                    }
                    if (!(norm2 > ak2)) continue;
                    nextColumn = i;
                    ak2 = norm2;
                }
            } else {
                nextColumn = k;
                ak2 = 0.0;
                for (int j = k; j < rows; ++j) {
                    double aki = this.qr[j][k];
                    ak2 += aki * aki;
                }
            }
            if (ak2 <= qrRankingThreshold) {
                for (i = this.rank = k; i < rows; ++i) {
                    for (int j = i + 1; j < cols; ++j) {
                        this.qr[i][this.permutation[j]] = 0.0;
                    }
                }
                return;
            }
            int pk = this.permutation[nextColumn];
            this.permutation[nextColumn] = this.permutation[k];
            this.permutation[k] = pk;
            double akk = this.qr[k][pk];
            double alpha = akk > 0.0 ? -FastMath.sqrt(ak2) : FastMath.sqrt(ak2);
            this.beta[pk] = betak = 1.0 / (ak2 - akk * alpha);
            this.rDiag[pk] = alpha;
            double[] dArray = this.qr[k];
            int n = pk;
            dArray[n] = dArray[n] - alpha;
            for (int dk = cols - 1 - k; dk > 0; --dk) {
                int j;
                double gamma = 0.0;
                for (j = k; j < rows; ++j) {
                    gamma += this.qr[j][pk] * this.qr[j][this.permutation[k + dk]];
                }
                gamma *= betak;
                for (j = k; j < rows; ++j) {
                    double[] dArray2 = this.qr[j];
                    int n2 = this.permutation[k + dk];
                    dArray2[n2] = dArray2[n2] - gamma * this.qr[j][pk];
                }
            }
        }
        this.rank = cols;
    }

    public RealMatrix getQ() {
        if (this.cachedQ == null) {
            this.cachedQ = this.getQT().transpose();
        }
        return this.cachedQ;
    }

    public RealMatrix getQT() {
        if (this.cachedQT == null) {
            int minor;
            int m = this.qr.length;
            this.cachedQT = MatrixUtils.createRealMatrix(m, m);
            for (minor = m - 1; minor >= this.rank; --minor) {
                this.cachedQT.setEntry(minor, minor, 1.0);
            }
            for (minor = this.rank - 1; minor >= 0; --minor) {
                int p_minor = this.permutation[minor];
                this.cachedQT.setEntry(minor, minor, 1.0);
                for (int col = minor; col < m; ++col) {
                    int row;
                    double alpha = 0.0;
                    for (row = minor; row < m; ++row) {
                        alpha -= this.cachedQT.getEntry(col, row) * this.qr[row][p_minor];
                    }
                    alpha /= this.rDiag[p_minor] * this.qr[minor][p_minor];
                    for (row = minor; row < m; ++row) {
                        this.cachedQT.addToEntry(col, row, -alpha * this.qr[row][p_minor]);
                    }
                }
            }
        }
        return this.cachedQT;
    }

    public RealMatrix getR() {
        if (this.cachedR == null) {
            int n = this.qr[0].length;
            int m = this.qr.length;
            this.cachedR = MatrixUtils.createRealMatrix(m, n);
            for (int row = this.rank - 1; row >= 0; --row) {
                this.cachedR.setEntry(row, row, this.rDiag[this.permutation[row]]);
                for (int col = row + 1; col < n; ++col) {
                    this.cachedR.setEntry(row, col, this.qr[row][this.permutation[col]]);
                }
            }
        }
        return this.cachedR;
    }

    public RealMatrix getH() {
        if (this.cachedH == null) {
            int n = this.qr[0].length;
            int m = this.qr.length;
            this.cachedH = MatrixUtils.createRealMatrix(m, n);
            for (int i = 0; i < m; ++i) {
                for (int j = 0; j < FastMath.min(i + 1, n); ++j) {
                    int p_j = this.permutation[j];
                    this.cachedH.setEntry(i, j, this.qr[i][p_j] / -this.rDiag[p_j]);
                }
            }
        }
        return this.cachedH;
    }

    public RealMatrix getPermutationMatrix() {
        RealMatrix rm = MatrixUtils.createRealMatrix(this.qr[0].length, this.qr[0].length);
        for (int i = 0; i < this.qr[0].length; ++i) {
            rm.setEntry(this.permutation[i], i, 1.0);
        }
        return rm;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.qr, this.rDiag, this.permutation, this.rank);
    }

    private static class Solver
    implements DecompositionSolver {
        private final double[][] qr;
        private final double[] rDiag;
        private final int rank;
        private final int[] perm;

        private Solver(double[][] qr, double[] rDiag, int[] perm, int rank) {
            this.qr = qr;
            this.rDiag = rDiag;
            this.perm = perm;
            this.rank = rank;
        }

        public boolean isNonSingular() {
            if (this.qr.length >= this.qr[0].length) {
                return this.rank == this.qr[0].length;
            }
            return this.rank == this.qr.length;
        }

        public RealVector solve(RealVector b) {
            int n = this.qr[0].length;
            int m = this.qr.length;
            if (b.getDimension() != m) {
                throw new DimensionMismatchException(b.getDimension(), m);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            double[] x = new double[n];
            double[] y = b.toArray();
            for (int minor = 0; minor < this.rank; ++minor) {
                int row;
                int m_idx = this.perm[minor];
                double dotProduct = 0.0;
                for (row = minor; row < m; ++row) {
                    dotProduct += y[row] * this.qr[row][m_idx];
                }
                dotProduct /= this.rDiag[m_idx] * this.qr[minor][m_idx];
                for (row = minor; row < m; ++row) {
                    int n2 = row;
                    y[n2] = y[n2] + dotProduct * this.qr[row][m_idx];
                }
            }
            for (int row = this.rank - 1; row >= 0; --row) {
                double yRow;
                int m_row = this.perm[row];
                int n3 = row;
                y[n3] = y[n3] / this.rDiag[m_row];
                x[this.perm[row]] = yRow = y[row];
                for (int i = 0; i < row; ++i) {
                    int n4 = i;
                    y[n4] = y[n4] - yRow * this.qr[i][m_row];
                }
            }
            return new ArrayRealVector(x, false);
        }

        public RealMatrix solve(RealMatrix b) {
            int cols = this.qr[0].length;
            int rows = this.qr.length;
            if (b.getRowDimension() != rows) {
                throw new DimensionMismatchException(b.getRowDimension(), rows);
            }
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int columns = b.getColumnDimension();
            int blockSize = 52;
            int cBlocks = (columns + 52 - 1) / 52;
            double[][] xBlocks = BlockRealMatrix.createBlocksLayout(cols, columns);
            double[][] y = new double[b.getRowDimension()][52];
            double[] alpha = new double[52];
            for (int kBlock = 0; kBlock < cBlocks; ++kBlock) {
                int k;
                int kStart = kBlock * 52;
                int kEnd = FastMath.min(kStart + 52, columns);
                int kWidth = kEnd - kStart;
                b.copySubMatrix(0, rows - 1, kStart, kEnd - 1, y);
                for (int minor = 0; minor < this.rank; ++minor) {
                    double[] yRow;
                    double d;
                    int row;
                    int m_idx = this.perm[minor];
                    double factor = 1.0 / (this.rDiag[m_idx] * this.qr[minor][m_idx]);
                    Arrays.fill(alpha, 0, kWidth, 0.0);
                    for (row = minor; row < rows; ++row) {
                        d = this.qr[row][m_idx];
                        yRow = y[row];
                        for (k = 0; k < kWidth; ++k) {
                            int n = k;
                            alpha[n] = alpha[n] + d * yRow[k];
                        }
                    }
                    int k2 = 0;
                    while (k2 < kWidth) {
                        int n = k2++;
                        alpha[n] = alpha[n] * factor;
                    }
                    for (row = minor; row < rows; ++row) {
                        d = this.qr[row][m_idx];
                        yRow = y[row];
                        for (k = 0; k < kWidth; ++k) {
                            int n = k;
                            yRow[n] = yRow[n] + alpha[k] * d;
                        }
                    }
                }
                for (int j = this.rank - 1; j >= 0; --j) {
                    int jBlock = this.perm[j] / 52;
                    int jStart = jBlock * 52;
                    double factor = 1.0 / this.rDiag[this.perm[j]];
                    double[] yJ = y[j];
                    double[] xBlock = xBlocks[jBlock * cBlocks + kBlock];
                    int index = (this.perm[j] - jStart) * kWidth;
                    for (k = 0; k < kWidth; ++k) {
                        int n = k;
                        yJ[n] = yJ[n] * factor;
                        xBlock[index++] = yJ[k];
                    }
                    for (int i = 0; i < j; ++i) {
                        double rIJ = this.qr[i][this.perm[j]];
                        double[] yI = y[i];
                        for (int k3 = 0; k3 < kWidth; ++k3) {
                            int n = k3;
                            yI[n] = yI[n] - yJ[k3] * rIJ;
                        }
                    }
                }
            }
            return new BlockRealMatrix(cols, columns, xBlocks, false);
        }

        public RealMatrix getInverse() {
            return this.solve(MatrixUtils.createRealIdentityMatrix(this.rDiag.length));
        }
    }
}

