/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.metrics;

import java.util.TimerTask;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.flink.annotation.Internal;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.metrics.Metric;
import org.apache.flink.metrics.groups.AbstractMetricGroup;
import org.apache.flink.metrics.groups.scope.ScopeFormat;
import org.apache.flink.metrics.groups.scope.ScopeFormats;
import org.apache.flink.metrics.reporter.JMXReporter;
import org.apache.flink.metrics.reporter.MetricReporter;
import org.apache.flink.metrics.reporter.Scheduled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class MetricRegistry {
    static final Logger LOG = LoggerFactory.getLogger(MetricRegistry.class);
    private final MetricReporter reporter;
    private final ScheduledExecutorService executor;
    private final ScopeFormats scopeFormats;
    private final char delimiter;

    public MetricRegistry(Configuration config) {
        int delim;
        ScopeFormats scopeFormats;
        try {
            scopeFormats = MetricRegistry.createScopeConfig(config);
        }
        catch (Exception e) {
            LOG.warn("Failed to parse scope format, using default scope formats", (Throwable)e);
            scopeFormats = new ScopeFormats();
        }
        this.scopeFormats = scopeFormats;
        try {
            delim = config.getString("metrics.scope.delimiter", ".").charAt(0);
        }
        catch (Exception e) {
            LOG.warn("Failed to parse delimiter, using default delimiter.", (Throwable)e);
            delim = 46;
        }
        this.delimiter = (char)delim;
        String className = config.getString("metrics.reporter.class", null);
        if (className == null) {
            LOG.info("No metrics reporter configured, exposing metrics via JMX");
            this.reporter = MetricRegistry.startJmxReporter(config);
            this.executor = null;
        } else {
            MetricReporter reporter;
            ScheduledExecutorService executor = null;
            try {
                String configuredPeriod = config.getString("metrics.reporter.interval", null);
                TimeUnit timeunit = TimeUnit.SECONDS;
                long period = 10L;
                if (configuredPeriod != null) {
                    try {
                        String[] interval = configuredPeriod.split(" ");
                        period = Long.parseLong(interval[0]);
                        timeunit = TimeUnit.valueOf(interval[1]);
                    }
                    catch (Exception e) {
                        LOG.error("Cannot parse report interval from config: " + configuredPeriod + " - please use values like '10 SECONDS' or '500 MILLISECONDS'. " + "Using default reporting interval.");
                    }
                }
                Configuration reporterConfig = MetricRegistry.createReporterConfig(config, timeunit, period);
                Class<?> reporterClass = Class.forName(className);
                reporter = (MetricReporter)reporterClass.newInstance();
                reporter.open(reporterConfig);
                if (reporter instanceof Scheduled) {
                    executor = Executors.newSingleThreadScheduledExecutor();
                    LOG.info("Periodically reporting metrics in intervals of {} {}", (Object)period, (Object)timeunit.name());
                    executor.scheduleWithFixedDelay(new ReporterTask((Scheduled)((Object)reporter)), period, period, timeunit);
                }
            }
            catch (Throwable t) {
                this.shutdownExecutor();
                LOG.error("Could not instantiate custom metrics reporter. Defaulting to JMX metrics export.", t);
                reporter = MetricRegistry.startJmxReporter(config);
            }
            this.reporter = reporter;
            this.executor = executor;
        }
    }

    public char getDelimiter() {
        return this.delimiter;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static JMXReporter startJmxReporter(Configuration config) {
        JMXReporter reporter = null;
        try {
            Configuration reporterConfig = new Configuration();
            String portRange = config.getString("metrics.jmx.port", null);
            if (portRange != null) {
                reporterConfig.setString("metrics.jmx.port", portRange);
            }
            reporter = new JMXReporter();
            reporter.open(reporterConfig);
        }
        catch (Exception e) {
            LOG.error("Failed to instantiate JMX reporter.", (Throwable)e);
        }
        finally {
            return reporter;
        }
    }

    public void shutdown() {
        if (this.reporter != null) {
            try {
                this.reporter.close();
            }
            catch (Throwable t) {
                LOG.warn("Metrics reporter did not shut down cleanly", t);
            }
        }
        this.shutdownExecutor();
    }

    private void shutdownExecutor() {
        if (this.executor != null) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(1L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
            }
        }
    }

    public ScopeFormats getScopeFormats() {
        return this.scopeFormats;
    }

    public void register(Metric metric, String metricName, AbstractMetricGroup group) {
        try {
            if (this.reporter != null) {
                this.reporter.notifyOfAddedMetric(metric, metricName, group);
            }
        }
        catch (Exception e) {
            LOG.error("Error while registering metric.", (Throwable)e);
        }
    }

    public void unregister(Metric metric, String metricName, AbstractMetricGroup group) {
        try {
            if (this.reporter != null) {
                this.reporter.notifyOfRemovedMetric(metric, metricName, group);
            }
        }
        catch (Exception e) {
            LOG.error("Error while registering metric.", (Throwable)e);
        }
    }

    static Configuration createReporterConfig(Configuration config, TimeUnit timeunit, long period) {
        Configuration reporterConfig = new Configuration();
        reporterConfig.setLong("period", period);
        reporterConfig.setString("timeunit", timeunit.name());
        String[] arguments = config.getString("metrics.reporter.arguments", "").split(" ");
        if (arguments.length > 1) {
            for (int x = 0; x < arguments.length; x += 2) {
                reporterConfig.setString(arguments[x].replace("--", ""), arguments[x + 1]);
            }
        }
        return reporterConfig;
    }

    static ScopeFormats createScopeConfig(Configuration config) {
        String jmFormat = config.getString("metrics.scope.jm", ScopeFormat.DEFAULT_SCOPE_JOBMANAGER_GROUP);
        String jmJobFormat = config.getString("metrics.scope.jm.job", ScopeFormat.DEFAULT_SCOPE_JOBMANAGER_JOB_GROUP);
        String tmFormat = config.getString("metrics.scope.tm", ScopeFormat.DEFAULT_SCOPE_TASKMANAGER_GROUP);
        String tmJobFormat = config.getString("metrics.scope.tm.job", ScopeFormat.DEFAULT_SCOPE_TASKMANAGER_JOB_GROUP);
        String taskFormat = config.getString("metrics.scope.task", ScopeFormat.DEFAULT_SCOPE_TASK_GROUP);
        String operatorFormat = config.getString("metrics.scope.operator", ScopeFormat.DEFAULT_SCOPE_OPERATOR_GROUP);
        return new ScopeFormats(jmFormat, jmJobFormat, tmFormat, tmJobFormat, taskFormat, operatorFormat);
    }

    private static final class ReporterTask
    extends TimerTask {
        private final Scheduled reporter;

        private ReporterTask(Scheduled reporter) {
            this.reporter = reporter;
        }

        @Override
        public void run() {
            try {
                this.reporter.report();
            }
            catch (Throwable t) {
                LOG.warn("Error while reporting metrics", t);
            }
        }
    }
}

