/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.configuration;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.ConfigOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.GlobalConfiguration;
import org.apache.flink.core.io.IOReadableWritable;
import org.apache.flink.testutils.junit.extensions.parameterized.Parameter;
import org.apache.flink.testutils.junit.extensions.parameterized.ParameterizedTestExtension;
import org.apache.flink.testutils.junit.extensions.parameterized.Parameters;
import org.apache.flink.util.ExceptionUtils;
import org.apache.flink.util.InstantiationUtil;
import org.assertj.core.api.AbstractCollectionAssert;
import org.assertj.core.api.AbstractStringAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ThrowingConsumer;
import org.assertj.core.data.Offset;
import org.junit.jupiter.api.TestTemplate;
import org.junit.jupiter.api.extension.ExtendWith;

@ExtendWith(value={ParameterizedTestExtension.class})
public class ConfigurationTest {
    @Parameter
    public boolean standardYaml;
    private static final ConfigOption<String> STRING_OPTION = ConfigOptions.key((String)"test-string-key").stringType().noDefaultValue();
    private static final ConfigOption<List<String>> LIST_STRING_OPTION = ConfigOptions.key((String)"test-list-key").stringType().asList().noDefaultValue();
    private static final ConfigOption<Map<String, String>> MAP_OPTION = ConfigOptions.key((String)"test-map-key").mapType().noDefaultValue();
    private static final ConfigOption<Duration> DURATION_OPTION = ConfigOptions.key((String)"test-duration-key").durationType().noDefaultValue();
    private static final Map<String, String> PROPERTIES_MAP = new HashMap<String, String>();
    private static final String MAP_PROPERTY_1;
    private static final String MAP_PROPERTY_2;

    @Parameters(name="standardYaml: {0}")
    public static Collection<Boolean> parameters() {
        return Arrays.asList(true, false);
    }

    @TestTemplate
    void testConfigurationSerializationAndGetters() {
        try {
            Configuration orig = new Configuration(this.standardYaml);
            orig.setString("mykey", "myvalue");
            orig.setInteger("mynumber", 100);
            orig.setLong("longvalue", 478236947162389746L);
            orig.setFloat("PI", 3.1415925f);
            orig.setDouble("E", Math.E);
            orig.setBoolean("shouldbetrue", true);
            orig.setBytes("bytes sequence", new byte[]{1, 2, 3, 4, 5});
            orig.setClass("myclass", this.getClass());
            Configuration copy = (Configuration)InstantiationUtil.createCopyWritable((IOReadableWritable)orig);
            Assertions.assertThat((String)"myvalue").isEqualTo(copy.getString("mykey", "null"));
            Assertions.assertThat((int)copy.getInteger("mynumber", 0)).isEqualTo(100);
            Assertions.assertThat((long)478236947162389746L).isEqualTo(copy.getLong("longvalue", 0L));
            Assertions.assertThat((float)3.1415925f).isCloseTo(copy.getFloat("PI", 3.1415925f), Offset.offset((Number)Float.valueOf(0.0f)));
            Assertions.assertThat((double)Math.E).isCloseTo(copy.getDouble("E", 0.0), Offset.offset((Number)0.0));
            Assertions.assertThat((boolean)copy.getBoolean("shouldbetrue", false)).isTrue();
            Assertions.assertThat((byte[])new byte[]{1, 2, 3, 4, 5}).isEqualTo((Object)copy.getBytes("bytes sequence", null));
            Assertions.assertThat(this.getClass()).isEqualTo((Object)copy.getClass("myclass", null, this.getClass().getClassLoader()));
            Assertions.assertThat((Object)orig).isEqualTo((Object)copy);
            Assertions.assertThat((Collection)orig.keySet()).isEqualTo((Object)copy.keySet());
            Assertions.assertThat((int)orig.hashCode()).isEqualTo(copy.hashCode());
        }
        catch (Exception e) {
            e.printStackTrace();
            org.junit.jupiter.api.Assertions.fail((String)e.getMessage());
        }
    }

    @TestTemplate
    void testCopyConstructor() {
        try {
            String key = "theKey";
            Configuration cfg1 = new Configuration(this.standardYaml);
            cfg1.setString("theKey", "value");
            Configuration cfg2 = new Configuration(cfg1);
            cfg2.setString("theKey", "another value");
            Assertions.assertThat((String)"value").isEqualTo(cfg1.getString("theKey", ""));
        }
        catch (Exception e) {
            e.printStackTrace();
            org.junit.jupiter.api.Assertions.fail((String)e.getMessage());
        }
    }

    @TestTemplate
    void testOptionWithDefault() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setInteger("int-key", 11);
        cfg.setString("string-key", "abc");
        ConfigOption presentStringOption = ConfigOptions.key((String)"string-key").stringType().defaultValue((Object)"my-beautiful-default");
        ConfigOption presentIntOption = ConfigOptions.key((String)"int-key").intType().defaultValue((Object)87);
        Assertions.assertThat((String)"abc").isEqualTo(cfg.getString(presentStringOption));
        Assertions.assertThat((String)"abc").isEqualTo(cfg.getValue(presentStringOption));
        Assertions.assertThat((int)11).isEqualTo(cfg.getInteger(presentIntOption));
        Assertions.assertThat((String)"11").isEqualTo(cfg.getValue(presentIntOption));
        ConfigOption stringOption = ConfigOptions.key((String)"test").stringType().defaultValue((Object)"my-beautiful-default");
        ConfigOption intOption = ConfigOptions.key((String)"test2").intType().defaultValue((Object)87);
        Assertions.assertThat((String)"my-beautiful-default").isEqualTo(cfg.getValue(stringOption));
        Assertions.assertThat((String)"my-beautiful-default").isEqualTo(cfg.getString(stringOption));
        Assertions.assertThat((String)"override").isEqualTo(cfg.getString(stringOption, "override"));
        Assertions.assertThat((int)87).isEqualTo(cfg.getInteger(intOption));
        Assertions.assertThat((String)"87").isEqualTo(cfg.getValue(intOption));
    }

    @TestTemplate
    void testOptionWithNoDefault() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setInteger("int-key", 11);
        cfg.setString("string-key", "abc");
        ConfigOption presentStringOption = ConfigOptions.key((String)"string-key").stringType().noDefaultValue();
        Assertions.assertThat((String)"abc").isEqualTo(cfg.getString(presentStringOption));
        Assertions.assertThat((String)"abc").isEqualTo(cfg.getValue(presentStringOption));
        ConfigOption stringOption = ConfigOptions.key((String)"test").stringType().noDefaultValue();
        Assertions.assertThat((String)cfg.getValue(stringOption)).isNull();
        Assertions.assertThat((String)cfg.getString(stringOption)).isNull();
        Assertions.assertThat((String)"override").isEqualTo(cfg.getString(stringOption, "override"));
    }

    @TestTemplate
    void testDeprecatedKeys() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setInteger("the-key", 11);
        cfg.setInteger("old-key", 12);
        cfg.setInteger("older-key", 13);
        ConfigOption matchesFirst = ConfigOptions.key((String)"the-key").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{"old-key", "older-key"});
        ConfigOption matchesSecond = ConfigOptions.key((String)"does-not-exist").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{"old-key", "older-key"});
        ConfigOption matchesThird = ConfigOptions.key((String)"does-not-exist").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{"foo", "older-key"});
        ConfigOption notContained = ConfigOptions.key((String)"does-not-exist").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{"not-there", "also-not-there"});
        Assertions.assertThat((int)11).isEqualTo(cfg.getInteger(matchesFirst));
        Assertions.assertThat((int)12).isEqualTo(cfg.getInteger(matchesSecond));
        Assertions.assertThat((int)13).isEqualTo(cfg.getInteger(matchesThird));
        Assertions.assertThat((int)-1).isEqualTo(cfg.getInteger(notContained));
    }

    @TestTemplate
    void testFallbackKeys() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setInteger("the-key", 11);
        cfg.setInteger("old-key", 12);
        cfg.setInteger("older-key", 13);
        ConfigOption matchesFirst = ConfigOptions.key((String)"the-key").intType().defaultValue((Object)-1).withFallbackKeys(new String[]{"old-key", "older-key"});
        ConfigOption matchesSecond = ConfigOptions.key((String)"does-not-exist").intType().defaultValue((Object)-1).withFallbackKeys(new String[]{"old-key", "older-key"});
        ConfigOption matchesThird = ConfigOptions.key((String)"does-not-exist").intType().defaultValue((Object)-1).withFallbackKeys(new String[]{"foo", "older-key"});
        ConfigOption notContained = ConfigOptions.key((String)"does-not-exist").intType().defaultValue((Object)-1).withFallbackKeys(new String[]{"not-there", "also-not-there"});
        Assertions.assertThat((int)11).isEqualTo(cfg.getInteger(matchesFirst));
        Assertions.assertThat((int)12).isEqualTo(cfg.getInteger(matchesSecond));
        Assertions.assertThat((int)13).isEqualTo(cfg.getInteger(matchesThird));
        Assertions.assertThat((int)-1).isEqualTo(cfg.getInteger(notContained));
    }

    @TestTemplate
    void testFallbackAndDeprecatedKeys() {
        ConfigOption fallback = ConfigOptions.key((String)"fallback").intType().defaultValue((Object)-1);
        ConfigOption deprecated = ConfigOptions.key((String)"deprecated").intType().defaultValue((Object)-1);
        ConfigOption mainOption = ConfigOptions.key((String)"main").intType().defaultValue((Object)-1).withFallbackKeys(new String[]{fallback.key()}).withDeprecatedKeys(new String[]{deprecated.key()});
        Configuration fallbackCfg = new Configuration(this.standardYaml);
        fallbackCfg.setInteger(fallback, 1);
        Assertions.assertThat((int)fallbackCfg.getInteger(mainOption)).isOne();
        Configuration deprecatedCfg = new Configuration(this.standardYaml);
        deprecatedCfg.setInteger(deprecated, 2);
        Assertions.assertThat((int)deprecatedCfg.getInteger(mainOption)).isEqualTo(2);
        ConfigOption reversedMainOption = ConfigOptions.key((String)"main").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{deprecated.key()}).withFallbackKeys(new String[]{fallback.key()});
        Configuration deprecatedAndFallBackConfig = new Configuration(this.standardYaml);
        deprecatedAndFallBackConfig.setInteger(fallback, 1);
        deprecatedAndFallBackConfig.setInteger(deprecated, 2);
        Assertions.assertThat((int)deprecatedAndFallBackConfig.getInteger(mainOption)).isOne();
        Assertions.assertThat((int)deprecatedAndFallBackConfig.getInteger(reversedMainOption)).isOne();
    }

    @TestTemplate
    void testRemove() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setInteger("a", 1);
        cfg.setInteger("b", 2);
        ConfigOption validOption = ConfigOptions.key((String)"a").intType().defaultValue((Object)-1);
        ConfigOption deprecatedOption = ConfigOptions.key((String)"c").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{"d", "b"});
        ConfigOption unexistedOption = ConfigOptions.key((String)"e").intType().defaultValue((Object)-1).withDeprecatedKeys(new String[]{"f", "g", "j"});
        ((AbstractCollectionAssert)Assertions.assertThat((Collection)cfg.keySet()).hasSize(2)).as("Wrong expectation about size", new Object[0]);
        Assertions.assertThat((boolean)cfg.removeConfig(validOption)).isTrue().as("Expected 'validOption' is removed", new Object[0]);
        ((AbstractCollectionAssert)Assertions.assertThat((Collection)cfg.keySet()).hasSize(1)).as("Wrong expectation about size", new Object[0]);
        Assertions.assertThat((boolean)cfg.removeConfig(deprecatedOption)).isTrue().as("Expected 'existedOption' is removed", new Object[0]);
        ((AbstractCollectionAssert)Assertions.assertThat((Collection)cfg.keySet()).hasSize(0)).as("Wrong expectation about size", new Object[0]);
        Assertions.assertThat((boolean)cfg.removeConfig(unexistedOption)).isFalse().as("Expected 'unexistedOption' is not removed", new Object[0]);
    }

    @TestTemplate
    void testRemoveKey() {
        Configuration cfg = new Configuration(this.standardYaml);
        String key1 = "a.b";
        String key2 = "c.d";
        cfg.setInteger(key1, 42);
        cfg.setInteger(key2, 44);
        cfg.setInteger(key2 + ".f1", 44);
        cfg.setInteger(key2 + ".f2", 44);
        cfg.setInteger("e.f", 1337);
        Assertions.assertThat((boolean)cfg.removeKey("not-existing-key")).isFalse();
        Assertions.assertThat((boolean)cfg.removeKey(key1)).isTrue();
        Assertions.assertThat((boolean)cfg.containsKey(key1)).isFalse();
        Assertions.assertThat((boolean)cfg.removeKey(key2)).isTrue();
        Assertions.assertThat((Collection)cfg.keySet()).containsExactlyInAnyOrder((Object[])new String[]{"e.f"});
    }

    @TestTemplate
    void testShouldParseValidStringToEnum() {
        Configuration configuration = new Configuration(this.standardYaml);
        configuration.setString(STRING_OPTION.key(), TestEnum.VALUE1.toString());
        TestEnum parsedEnumValue = (TestEnum)configuration.getEnum(TestEnum.class, STRING_OPTION);
        Assertions.assertThat((Comparable)((Object)TestEnum.VALUE1)).isEqualTo((Object)parsedEnumValue);
    }

    @TestTemplate
    void testShouldParseValidStringToEnumIgnoringCase() {
        Configuration configuration = new Configuration(this.standardYaml);
        configuration.setString(STRING_OPTION.key(), TestEnum.VALUE1.toString().toLowerCase());
        TestEnum parsedEnumValue = (TestEnum)configuration.getEnum(TestEnum.class, STRING_OPTION);
        Assertions.assertThat((Comparable)((Object)TestEnum.VALUE1)).isEqualTo((Object)parsedEnumValue);
    }

    @TestTemplate
    void testThrowsExceptionIfTryingToParseInvalidStringForEnum() {
        Configuration configuration = new Configuration(this.standardYaml);
        String invalidValueForTestEnum = "InvalidValueForTestEnum";
        configuration.setString(STRING_OPTION.key(), "InvalidValueForTestEnum");
        try {
            configuration.getEnum(TestEnum.class, STRING_OPTION);
            org.junit.jupiter.api.Assertions.fail((String)"Expected exception not thrown");
        }
        catch (IllegalArgumentException e) {
            String expectedMessage = "Value for config option " + STRING_OPTION.key() + " must be one of [VALUE1, VALUE2] (was " + "InvalidValueForTestEnum" + ")";
            Assertions.assertThat((String)e.getMessage()).contains(new CharSequence[]{expectedMessage});
        }
    }

    @TestTemplate
    void testToMap() {
        Configuration configuration = new Configuration(this.standardYaml);
        String listValues = "value1;value2;value3";
        String yamlListValues = "[value1, value2, value3]";
        configuration.set(LIST_STRING_OPTION, Arrays.asList("value1;value2;value3".split(";")));
        String mapValues = "key1:value1,key2:value2";
        String yamlMapValues = "{key1: value1, key2: value2}";
        configuration.set(MAP_OPTION, Arrays.stream("key1:value1,key2:value2".split(",")).collect(Collectors.toMap(e -> e.split(":")[0], e -> e.split(":")[1])));
        Duration duration = Duration.ofMillis(3000L);
        configuration.set(DURATION_OPTION, (Object)duration);
        if (this.standardYaml) {
            Assertions.assertThat((String)"[value1, value2, value3]").isEqualTo((String)configuration.toMap().get(LIST_STRING_OPTION.key()));
            Assertions.assertThat((String)"{key1: value1, key2: value2}").isEqualTo((String)configuration.toMap().get(MAP_OPTION.key()));
        } else {
            Assertions.assertThat((String)"value1;value2;value3").isEqualTo((String)configuration.toMap().get(LIST_STRING_OPTION.key()));
            Assertions.assertThat((String)"key1:value1,key2:value2").isEqualTo((String)configuration.toMap().get(MAP_OPTION.key()));
        }
        Assertions.assertThat((String)"3 s").isEqualTo((String)configuration.toMap().get(DURATION_OPTION.key()));
    }

    @TestTemplate
    void testToFileWritableMap() {
        Configuration configuration = new Configuration(this.standardYaml);
        String listValues = "value1;value2;value3";
        String yamlListValues = "[value1, value2, value3]";
        configuration.set(LIST_STRING_OPTION, Arrays.asList("value1;value2;value3".split(";")));
        String mapValues = "key1:value1,key2:value2";
        String yamlMapValues = "{key1: value1, key2: value2}";
        configuration.set(MAP_OPTION, Arrays.stream("key1:value1,key2:value2".split(",")).collect(Collectors.toMap(e -> e.split(":")[0], e -> e.split(":")[1])));
        Duration duration = Duration.ofMillis(3000L);
        configuration.set(DURATION_OPTION, (Object)duration);
        String strValues = "*";
        String yamlStrValues = "'*'";
        configuration.set(STRING_OPTION, (Object)"*");
        if (this.standardYaml) {
            Assertions.assertThat((String)"[value1, value2, value3]").isEqualTo((String)configuration.toFileWritableMap().get(LIST_STRING_OPTION.key()));
            Assertions.assertThat((String)"{key1: value1, key2: value2}").isEqualTo((String)configuration.toFileWritableMap().get(MAP_OPTION.key()));
            Assertions.assertThat((String)"'*'").isEqualTo((String)configuration.toFileWritableMap().get(STRING_OPTION.key()));
        } else {
            Assertions.assertThat((String)"value1;value2;value3").isEqualTo((String)configuration.toFileWritableMap().get(LIST_STRING_OPTION.key()));
            Assertions.assertThat((String)"key1:value1,key2:value2").isEqualTo((String)configuration.toFileWritableMap().get(MAP_OPTION.key()));
            Assertions.assertThat((String)"*").isEqualTo((String)configuration.toFileWritableMap().get(STRING_OPTION.key()));
        }
        Assertions.assertThat((String)"3 s").isEqualTo((String)configuration.toMap().get(DURATION_OPTION.key()));
    }

    @TestTemplate
    void testMapNotContained() {
        Configuration cfg = new Configuration(this.standardYaml);
        Assertions.assertThat((boolean)cfg.getOptional(MAP_OPTION).isPresent()).isFalse();
        Assertions.assertThat((boolean)cfg.contains(MAP_OPTION)).isFalse();
    }

    @TestTemplate
    void testMapWithPrefix() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setString(MAP_PROPERTY_1, "value1");
        cfg.setInteger(MAP_PROPERTY_2, 12);
        Assertions.assertThat((Map)((Map)cfg.get(MAP_OPTION))).isEqualTo(PROPERTIES_MAP);
        Assertions.assertThat((boolean)cfg.contains(MAP_OPTION)).isTrue();
    }

    @TestTemplate
    void testMapWithoutPrefix() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.set(MAP_OPTION, PROPERTIES_MAP);
        Assertions.assertThat((Map)((Map)cfg.get(MAP_OPTION))).isEqualTo(PROPERTIES_MAP);
        Assertions.assertThat((boolean)cfg.contains(MAP_OPTION)).isTrue();
    }

    @TestTemplate
    void testMapNonPrefixHasPrecedence() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.set(MAP_OPTION, PROPERTIES_MAP);
        cfg.setString(MAP_PROPERTY_1, "value1");
        cfg.setInteger(MAP_PROPERTY_2, 99999);
        Assertions.assertThat((Map)((Map)cfg.get(MAP_OPTION))).isEqualTo(PROPERTIES_MAP);
        Assertions.assertThat((boolean)cfg.contains(MAP_OPTION)).isTrue();
        Assertions.assertThat((boolean)cfg.containsKey(MAP_PROPERTY_1)).isTrue();
    }

    @TestTemplate
    void testMapThatOverwritesPrefix() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setString(MAP_PROPERTY_1, "value1");
        cfg.setInteger(MAP_PROPERTY_2, 99999);
        cfg.set(MAP_OPTION, PROPERTIES_MAP);
        Assertions.assertThat((Map)((Map)cfg.get(MAP_OPTION))).isEqualTo(PROPERTIES_MAP);
        Assertions.assertThat((boolean)cfg.contains(MAP_OPTION)).isTrue();
        Assertions.assertThat((boolean)cfg.containsKey(MAP_PROPERTY_1)).isFalse();
    }

    @TestTemplate
    void testMapRemovePrefix() {
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setString(MAP_PROPERTY_1, "value1");
        cfg.setInteger(MAP_PROPERTY_2, 99999);
        cfg.removeConfig(MAP_OPTION);
        Assertions.assertThat((boolean)cfg.contains(MAP_OPTION)).isFalse();
        Assertions.assertThat((boolean)cfg.containsKey(MAP_PROPERTY_1)).isFalse();
        Assertions.assertThat((boolean)cfg.containsKey(MAP_PROPERTY_2)).isFalse();
    }

    @TestTemplate
    void testListParserErrorDoesNotLeakSensitiveData() {
        ConfigOption secret = ConfigOptions.key((String)"secret").stringType().asList().noDefaultValue();
        Assertions.assertThat((boolean)GlobalConfiguration.isSensitive((String)secret.key())).isTrue();
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setString(secret.key(), "'secret_value");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
            List cfr_ignored_0 = (List)cfg.get(secret);
        }).isInstanceOf(IllegalArgumentException.class)).satisfies(new ThrowingConsumer[]{e -> {
            AbstractStringAssert cfr_ignored_0 = (AbstractStringAssert)Assertions.assertThat((String)ExceptionUtils.stringifyException((Throwable)e)).doesNotContain(new CharSequence[]{"secret_value"});
        }});
    }

    @TestTemplate
    void testMapParserErrorDoesNotLeakSensitiveData() {
        ConfigOption secret = ConfigOptions.key((String)"secret").mapType().noDefaultValue();
        Assertions.assertThat((boolean)GlobalConfiguration.isSensitive((String)secret.key())).isTrue();
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setString(secret.key(), "secret_value");
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> {
            Map cfr_ignored_0 = (Map)cfg.get(secret);
        }).isInstanceOf(IllegalArgumentException.class)).satisfies(new ThrowingConsumer[]{e -> {
            AbstractStringAssert cfr_ignored_0 = (AbstractStringAssert)Assertions.assertThat((String)ExceptionUtils.stringifyException((Throwable)e)).doesNotContain(new CharSequence[]{"secret_value"});
        }});
    }

    @TestTemplate
    void testToStringDoesNotLeakSensitiveData() {
        ConfigOption secret = ConfigOptions.key((String)"secret").mapType().noDefaultValue();
        Assertions.assertThat((boolean)GlobalConfiguration.isSensitive((String)secret.key())).isTrue();
        Configuration cfg = new Configuration(this.standardYaml);
        cfg.setString(secret.key(), "secret_value");
        Assertions.assertThat((String)cfg.toString()).doesNotContain(new CharSequence[]{"secret_value"});
    }

    @TestTemplate
    void testGetWithOverrideDefault() {
        Configuration conf = new Configuration(this.standardYaml);
        ConfigOption integerOption0 = ConfigOptions.key((String)"integer.key0").intType().noDefaultValue();
        Assertions.assertThat((Integer)((Integer)conf.get(integerOption0, (Object)2))).isEqualTo(2);
        conf.set(integerOption0, (Object)3);
        Assertions.assertThat((Integer)((Integer)conf.get(integerOption0, (Object)2))).isEqualTo(3);
        ConfigOption integerOption1 = ConfigOptions.key((String)"integer.key1").intType().defaultValue((Object)4);
        Assertions.assertThat((Integer)((Integer)conf.get(integerOption1, (Object)5))).isEqualTo(5);
        conf.set(integerOption1, (Object)6);
        Assertions.assertThat((Integer)((Integer)conf.get(integerOption1, (Object)5))).isEqualTo(6);
        ConfigOption stringOption0 = ConfigOptions.key((String)"string.key0").stringType().noDefaultValue();
        Assertions.assertThat((String)((String)conf.get(stringOption0, (Object)"a"))).isEqualTo("a");
        conf.set(stringOption0, (Object)"b");
        Assertions.assertThat((String)((String)conf.get(stringOption0, (Object)"a"))).isEqualTo("b");
        ConfigOption stringOption1 = ConfigOptions.key((String)"string.key1").stringType().defaultValue((Object)"c");
        Assertions.assertThat((String)((String)conf.get(stringOption1, (Object)"d"))).isEqualTo("d");
        conf.set(stringOption1, (Object)"e");
        Assertions.assertThat((String)((String)conf.get(stringOption1, (Object)"d"))).isEqualTo("e");
    }

    static {
        PROPERTIES_MAP.put("prop1", "value1");
        PROPERTIES_MAP.put("prop2", "12");
        MAP_PROPERTY_1 = MAP_OPTION.key() + ".prop1";
        MAP_PROPERTY_2 = MAP_OPTION.key() + ".prop2";
    }

    static enum TestEnum {
        VALUE1,
        VALUE2;

    }
}

