/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.buffer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Random;
import java.util.stream.Stream;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferPool;
import org.apache.flink.runtime.io.network.buffer.NetworkBufferPool;
import org.assertj.core.api.AbstractIntegerAssert;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

class BufferPoolFactoryTest {
    private static final int numBuffers = 1024;
    private static final int memorySegmentSize = 128;
    private NetworkBufferPool networkBufferPool;

    BufferPoolFactoryTest() {
    }

    @BeforeEach
    void setupNetworkBufferPool() {
        this.networkBufferPool = new NetworkBufferPool(1024, 128);
    }

    @AfterEach
    void verifyAllBuffersReturned() {
        try {
            ((AbstractIntegerAssert)Assertions.assertThat((int)this.networkBufferPool.getNumberOfAvailableMemorySegments()).withFailMessage("Did not return all buffers to network buffer pool after test.", new Object[0])).isEqualTo(1024);
        }
        finally {
            this.networkBufferPool.destroyAllBufferPools();
            this.networkBufferPool.destroy();
        }
    }

    @Test
    void testRequireMoreThanPossible1() {
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.networkBufferPool.createBufferPool(this.networkBufferPool.getTotalNumberOfMemorySegments() + 1, Integer.MAX_VALUE)).isInstanceOf(IOException.class)).hasMessageContaining("Insufficient number of network buffers");
    }

    @Test
    void testRequireMoreThanPossible2() throws IOException {
        BufferPool bufferPool = null;
        try {
            bufferPool = this.networkBufferPool.createBufferPool(513, 1024);
            ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> this.networkBufferPool.createBufferPool(513, 1024)).isInstanceOf(IOException.class)).hasMessageContaining("Insufficient number of network buffers");
        }
        finally {
            if (bufferPool != null) {
                bufferPool.lazyDestroy();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testOverprovisioned() throws IOException {
        int buffersToTakeFromPool1 = 513;
        int buffersToTakeFromPool2 = 1024 - buffersToTakeFromPool1;
        ArrayList<Buffer> buffers = new ArrayList<Buffer>(1024);
        BufferPool bufferPool1 = null;
        BufferPool bufferPool2 = null;
        try {
            Buffer buffer;
            int i;
            bufferPool1 = this.networkBufferPool.createBufferPool(buffersToTakeFromPool2, 1024);
            for (i = 0; i < buffersToTakeFromPool1; ++i) {
                buffer = bufferPool1.requestBuffer();
                Assertions.assertThat((Object)buffer).isNotNull();
                buffers.add(buffer);
            }
            Assertions.assertThat((int)bufferPool1.bestEffortGetNumOfUsedBuffers()).isEqualTo(buffersToTakeFromPool1);
            Assertions.assertThat((int)bufferPool1.getNumBuffers()).isEqualTo(1024);
            bufferPool2 = this.networkBufferPool.createBufferPool(buffersToTakeFromPool1, 1024);
            Assertions.assertThat((int)bufferPool2.getNumBuffers()).isEqualTo(bufferPool2.getNumberOfRequiredMemorySegments());
            Assertions.assertThat((int)bufferPool1.getNumBuffers()).isEqualTo(bufferPool1.getNumberOfRequiredMemorySegments());
            Assertions.assertThat((Object)bufferPool1.requestBuffer()).isNull();
            for (i = 0; i < buffersToTakeFromPool2; ++i) {
                buffer = bufferPool2.requestBuffer();
                Assertions.assertThat((Object)buffer).isNotNull();
                buffers.add(buffer);
            }
            Assertions.assertThat((int)bufferPool2.bestEffortGetNumOfUsedBuffers()).isEqualTo(buffersToTakeFromPool2);
            Assertions.assertThat((Object)bufferPool2.requestBuffer()).isNull();
            ((Buffer)buffers.remove(0)).recycleBuffer();
            Assertions.assertThat((int)this.networkBufferPool.getNumberOfAvailableMemorySegments()).isZero();
            Assertions.assertThat((int)(bufferPool1.bestEffortGetNumOfUsedBuffers() + bufferPool1.getNumberOfAvailableMemorySegments())).isEqualTo(buffersToTakeFromPool1 - 1);
            Assertions.assertThat((int)(bufferPool2.bestEffortGetNumOfUsedBuffers() + bufferPool2.getNumberOfAvailableMemorySegments())).isEqualTo(buffersToTakeFromPool2 + 1);
        }
        finally {
            for (Buffer buffer : buffers) {
                buffer.recycleBuffer();
            }
            if (bufferPool1 != null) {
                bufferPool1.lazyDestroy();
            }
            if (bufferPool2 != null) {
                bufferPool2.lazyDestroy();
            }
        }
    }

    @Test
    void testBoundedPools() throws IOException {
        BufferPool bufferPool1 = this.networkBufferPool.createBufferPool(1, 1);
        Assertions.assertThat((int)bufferPool1.getNumBuffers()).isOne();
        BufferPool bufferPool2 = this.networkBufferPool.createBufferPool(1, 2);
        Assertions.assertThat((int)bufferPool2.getNumBuffers()).isEqualTo(2);
        bufferPool1.lazyDestroy();
        bufferPool2.lazyDestroy();
    }

    @Test
    void testSingleManagedPoolGetsAll() throws IOException {
        BufferPool bufferPool = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
        Assertions.assertThat((int)bufferPool.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments());
        bufferPool.lazyDestroy();
    }

    @Test
    void testSingleManagedPoolGetsAllExceptFixedOnes() throws IOException {
        BufferPool fixedBufferPool = this.networkBufferPool.createBufferPool(24, 24);
        BufferPool flexibleBufferPool = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
        Assertions.assertThat((int)fixedBufferPool.getNumBuffers()).isEqualTo(24);
        Assertions.assertThat((int)flexibleBufferPool.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() - fixedBufferPool.getNumBuffers());
        fixedBufferPool.lazyDestroy();
        flexibleBufferPool.lazyDestroy();
    }

    @Test
    void testUniformDistribution() throws IOException {
        BufferPool first = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments());
        BufferPool second = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        first.lazyDestroy();
        second.lazyDestroy();
    }

    @Test
    void testUniformDistributionAllBuffers() throws IOException {
        BufferPool first = this.networkBufferPool.createBufferPool(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2, Integer.MAX_VALUE);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments());
        BufferPool second = this.networkBufferPool.createBufferPool(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2, Integer.MAX_VALUE);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        first.lazyDestroy();
        second.lazyDestroy();
    }

    @Test
    void testUniformDistributionBounded1() throws IOException {
        BufferPool first = this.networkBufferPool.createBufferPool(1, this.networkBufferPool.getTotalNumberOfMemorySegments());
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments());
        BufferPool second = this.networkBufferPool.createBufferPool(1, this.networkBufferPool.getTotalNumberOfMemorySegments());
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        first.lazyDestroy();
        second.lazyDestroy();
    }

    @Test
    void testUniformDistributionBounded2() throws IOException {
        BufferPool first = this.networkBufferPool.createBufferPool(1, 10);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(10);
        BufferPool second = this.networkBufferPool.createBufferPool(1, 10);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(10);
        Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(10);
        first.lazyDestroy();
        second.lazyDestroy();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testUniformDistributionBounded3() throws IOException {
        NetworkBufferPool globalPool = new NetworkBufferPool(3, 128);
        try {
            BufferPool first = globalPool.createBufferPool(1, 10);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(3);
            BufferPool second = globalPool.createBufferPool(1, 10);
            Assertions.assertThat((int)(first.getNumBuffers() + second.getNumBuffers())).isEqualTo(3);
            Assertions.assertThat((int)first.getNumBuffers()).isNotEqualTo(3);
            Assertions.assertThat((int)second.getNumBuffers()).isNotEqualTo(3);
            BufferPool third = globalPool.createBufferPool(1, 10);
            Assertions.assertThat((int)first.getNumBuffers()).isOne();
            Assertions.assertThat((int)second.getNumBuffers()).isOne();
            Assertions.assertThat((int)third.getNumBuffers()).isOne();
            String msg = "Wrong number of available segments after creating buffer pools.";
            ((AbstractIntegerAssert)Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).withFailMessage(msg, new Object[0])).isZero();
        }
        finally {
            globalPool.destroyAllBufferPools();
            globalPool.destroy();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    void testUniformDistributionBounded4() throws IOException {
        NetworkBufferPool globalPool = new NetworkBufferPool(10, 128);
        try {
            BufferPool first = globalPool.createBufferPool(1, 10);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(10);
            List segmentList1 = globalPool.requestUnpooledMemorySegments(2);
            Assertions.assertThat((List)segmentList1).hasSize(2);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(8);
            BufferPool second = globalPool.createBufferPool(1, 10);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(4);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(4);
            List segmentList2 = globalPool.requestUnpooledMemorySegments(2);
            Assertions.assertThat((List)segmentList2).hasSize(2);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(3);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(3);
            List segmentList3 = globalPool.requestUnpooledMemorySegments(2);
            Assertions.assertThat((List)segmentList3).hasSize(2);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(2);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(2);
            String msg = "Wrong number of available segments after creating buffer pools and requesting segments.";
            ((AbstractIntegerAssert)Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).withFailMessage(msg, new Object[0])).isEqualTo(2);
            globalPool.recycleUnpooledMemorySegments((Collection)segmentList1);
            ((AbstractIntegerAssert)Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).withFailMessage(msg, new Object[0])).isEqualTo(4);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(3);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(3);
            globalPool.recycleUnpooledMemorySegments((Collection)segmentList2);
            ((AbstractIntegerAssert)Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).withFailMessage(msg, new Object[0])).isEqualTo(6);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(4);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(4);
            globalPool.recycleUnpooledMemorySegments((Collection)segmentList3);
            ((AbstractIntegerAssert)Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).withFailMessage(msg, new Object[0])).isEqualTo(8);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(5);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(5);
            first.lazyDestroy();
            ((AbstractIntegerAssert)Assertions.assertThat((int)globalPool.getNumberOfAvailableMemorySegments()).withFailMessage(msg, new Object[0])).isEqualTo(9);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(10);
        }
        finally {
            globalPool.destroyAllBufferPools();
            globalPool.destroy();
        }
    }

    @Test
    void testBufferRedistributionMixed1() throws IOException {
        for (int i = 0; i < 1000; ++i) {
            BufferPool first = this.networkBufferPool.createBufferPool(1, 10);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(10);
            BufferPool second = this.networkBufferPool.createBufferPool(1, 10);
            Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(10);
            Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(10);
            BufferPool third = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
            for (BufferPool bp : new BufferPool[]{first, second, third}) {
                int avail = 1021;
                int size = 1021 * Math.min(1021, bp.getMaxNumberOfMemorySegments() - 1) / 1039 + 1;
                ((AbstractIntegerAssert)Assertions.assertThat((int)bp.getNumBuffers()).withFailMessage("Wrong buffer pool size after redistribution", new Object[0])).isIn(new Object[]{size, size + 1});
            }
            BufferPool fourth = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
            for (BufferPool bp : new BufferPool[]{first, second, third, fourth}) {
                int avail = 1020;
                int size = 1020 * Math.min(1020, bp.getMaxNumberOfMemorySegments() - 1) / 2058 + 1;
                ((AbstractIntegerAssert)Assertions.assertThat((int)bp.getNumBuffers()).withFailMessage("Wrong buffer pool size after redistribution", new Object[0])).isIn(new Object[]{size, size + 1});
            }
            Stream.of(first, second, third, fourth).forEach(BufferPool::lazyDestroy);
            this.verifyAllBuffersReturned();
            this.setupNetworkBufferPool();
        }
    }

    @Test
    void testAllDistributed() throws IOException {
        for (int i = 0; i < 1000; ++i) {
            Random random = new Random();
            ArrayList<BufferPool> pools = new ArrayList<BufferPool>();
            int numPools = 32;
            long maxTotalUsed = 0L;
            for (int j = 0; j < numPools; ++j) {
                int numRequiredBuffers = random.nextInt(7) + 1;
                int maxUsedBuffers = random.nextBoolean() ? Integer.MAX_VALUE : Math.max(1, random.nextInt(10) + numRequiredBuffers);
                pools.add(this.networkBufferPool.createBufferPool(numRequiredBuffers, maxUsedBuffers));
                maxTotalUsed = Math.min(1024L, maxTotalUsed + (long)maxUsedBuffers);
                int numDistributedBuffers = 0;
                for (BufferPool pool : pools) {
                    numDistributedBuffers += pool.getNumBuffers();
                }
                Assertions.assertThat((int)numDistributedBuffers).isEqualTo(maxTotalUsed);
            }
            pools.forEach(BufferPool::lazyDestroy);
            this.verifyAllBuffersReturned();
            this.setupNetworkBufferPool();
        }
    }

    @Test
    void testCreateDestroy() throws IOException {
        BufferPool first = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments());
        BufferPool second = this.networkBufferPool.createBufferPool(1, Integer.MAX_VALUE);
        Assertions.assertThat((int)first.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments() / 2);
        first.lazyDestroy();
        Assertions.assertThat((int)second.getNumBuffers()).isEqualTo(this.networkBufferPool.getTotalNumberOfMemorySegments());
        second.lazyDestroy();
    }
}

