/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Lists;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.hdfs.server.common.Storage;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogFileOutputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.EditLogOutputStream;
import org.apache.hadoop.hdfs.server.namenode.FSEditLogLoader;
import org.apache.hadoop.hdfs.server.namenode.JournalManager;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.server.namenode.NNStorageRetentionManager;
import org.apache.hadoop.hdfs.server.protocol.RemoteEditLog;

class FileJournalManager
implements JournalManager {
    private static final Log LOG = LogFactory.getLog(FileJournalManager.class);
    private final Storage.StorageDirectory sd;
    private int outputBufferCapacity = 524288;
    private static final Pattern EDITS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS.getName() + "_(\\d+)-(\\d+)");
    private static final Pattern EDITS_INPROGRESS_REGEX = Pattern.compile(NNStorage.NameNodeFile.EDITS_INPROGRESS.getName() + "_(\\d+)");
    @VisibleForTesting
    NNStorageRetentionManager.StoragePurger purger = new NNStorageRetentionManager.DeletionStoragePurger();

    public FileJournalManager(Storage.StorageDirectory sd) {
        this.sd = sd;
    }

    @Override
    public EditLogOutputStream startLogSegment(long txid) throws IOException {
        File newInProgress = NNStorage.getInProgressEditsFile(this.sd, txid);
        EditLogFileOutputStream stm = new EditLogFileOutputStream(newInProgress, this.outputBufferCapacity);
        ((EditLogOutputStream)stm).create();
        return stm;
    }

    @Override
    public void finalizeLogSegment(long firstTxId, long lastTxId) throws IOException {
        File inprogressFile = NNStorage.getInProgressEditsFile(this.sd, firstTxId);
        File dstFile = NNStorage.getFinalizedEditsFile(this.sd, firstTxId, lastTxId);
        LOG.debug((Object)("Finalizing edits file " + inprogressFile + " -> " + dstFile));
        Preconditions.checkState((!dstFile.exists() ? 1 : 0) != 0, (Object)("Can't finalize edits file " + inprogressFile + " since finalized file " + "already exists"));
        if (!inprogressFile.renameTo(dstFile)) {
            throw new IOException("Unable to finalize edits file " + inprogressFile);
        }
    }

    @VisibleForTesting
    public Storage.StorageDirectory getStorageDirectory() {
        return this.sd;
    }

    public String toString() {
        return "FileJournalManager for storage directory " + this.sd;
    }

    @Override
    public void setOutputBufferCapacity(int size) {
        this.outputBufferCapacity = size;
    }

    @Override
    public void purgeLogsOlderThan(long minTxIdToKeep) throws IOException {
        File[] files = FileUtil.listFiles((File)this.sd.getCurrentDir());
        List<EditLogFile> editLogs = FileJournalManager.matchEditLogs(files);
        for (EditLogFile log : editLogs) {
            if (log.getFirstTxId() >= minTxIdToKeep || log.getLastTxId() >= minTxIdToKeep) continue;
            this.purger.purgeLog(log);
        }
    }

    @Override
    public EditLogInputStream getInProgressInputStream(long segmentStartsAtTxId) throws IOException {
        File f = NNStorage.getInProgressEditsFile(this.sd, segmentStartsAtTxId);
        return new EditLogFileInputStream(f);
    }

    List<RemoteEditLog> getRemoteEditLogs(long firstTxId) throws IOException {
        File currentDir = this.sd.getCurrentDir();
        List<EditLogFile> allLogFiles = FileJournalManager.matchEditLogs(FileUtil.listFiles((File)currentDir));
        ArrayList ret = Lists.newArrayListWithCapacity((int)allLogFiles.size());
        for (EditLogFile elf : allLogFiles) {
            if (elf.isCorrupt() || elf.isInProgress()) continue;
            if (elf.getFirstTxId() >= firstTxId) {
                ret.add(new RemoteEditLog(elf.firstTxId, elf.lastTxId));
                continue;
            }
            if (firstTxId <= elf.getFirstTxId() || firstTxId > elf.getLastTxId()) continue;
            throw new IOException("Asked for firstTxId " + firstTxId + " which is in the middle of file " + elf.file);
        }
        return ret;
    }

    static List<EditLogFile> matchEditLogs(File[] filesInStorage) {
        ArrayList ret = Lists.newArrayList();
        for (File f : filesInStorage) {
            Matcher inProgressEditsMatch;
            String name = f.getName();
            Matcher editsMatch = EDITS_REGEX.matcher(name);
            if (editsMatch.matches()) {
                try {
                    long startTxId = Long.valueOf(editsMatch.group(1));
                    long endTxId = Long.valueOf(editsMatch.group(2));
                    ret.add(new EditLogFile(f, startTxId, endTxId));
                }
                catch (NumberFormatException nfe) {
                    LOG.error((Object)("Edits file " + f + " has improperly formatted " + "transaction ID"));
                }
            }
            if (!(inProgressEditsMatch = EDITS_INPROGRESS_REGEX.matcher(name)).matches()) continue;
            try {
                long startTxId = Long.valueOf(inProgressEditsMatch.group(1));
                ret.add(new EditLogFile(f, startTxId, -1L));
            }
            catch (NumberFormatException nfe) {
                LOG.error((Object)("In-progress edits file " + f + " has improperly " + "formatted transaction ID"));
            }
        }
        return ret;
    }

    static class EditLogFile {
        private File file;
        private final long firstTxId;
        private long lastTxId;
        private FSEditLogLoader.EditLogValidation cachedValidation = null;
        private boolean isCorrupt = false;
        static final long UNKNOWN_END = -1L;
        static final Comparator<EditLogFile> COMPARE_BY_START_TXID = new Comparator<EditLogFile>(){

            @Override
            public int compare(EditLogFile a, EditLogFile b) {
                return ComparisonChain.start().compare(a.getFirstTxId(), b.getFirstTxId()).compare(a.getLastTxId(), b.getLastTxId()).result();
            }
        };

        EditLogFile(File file, long firstTxId, long lastTxId) {
            assert (lastTxId == -1L || lastTxId >= firstTxId);
            assert (firstTxId > 0L);
            assert (file != null);
            this.firstTxId = firstTxId;
            this.lastTxId = lastTxId;
            this.file = file;
        }

        public void finalizeLog() throws IOException {
            long numTransactions = this.validateLog().numTransactions;
            long lastTxId = this.firstTxId + numTransactions - 1L;
            File dst = new File(this.file.getParentFile(), NNStorage.getFinalizedEditsFileName(this.firstTxId, lastTxId));
            LOG.info((Object)("Finalizing edits log " + this.file + " by renaming to " + dst.getName()));
            if (!this.file.renameTo(dst)) {
                throw new IOException("Couldn't finalize log " + this.file + " to " + dst);
            }
            this.lastTxId = lastTxId;
            this.file = dst;
        }

        long getFirstTxId() {
            return this.firstTxId;
        }

        long getLastTxId() {
            return this.lastTxId;
        }

        FSEditLogLoader.EditLogValidation validateLog() throws IOException {
            if (this.cachedValidation == null) {
                this.cachedValidation = EditLogFileInputStream.validateEditLog(this.file);
            }
            return this.cachedValidation;
        }

        boolean isInProgress() {
            return this.lastTxId == -1L;
        }

        File getFile() {
            return this.file;
        }

        void markCorrupt() {
            this.isCorrupt = true;
        }

        boolean isCorrupt() {
            return this.isCorrupt;
        }

        void moveAsideCorruptFile() throws IOException {
            assert (this.isCorrupt);
            File src = this.file;
            File dst = new File(src.getParent(), src.getName() + ".corrupt");
            boolean success = src.renameTo(dst);
            if (!success) {
                throw new IOException("Couldn't rename corrupt log " + src + " to " + dst);
            }
            this.file = dst;
        }

        public String toString() {
            return String.format("EditLogFile(file=%s,first=%019d,last=%019d,inProgress=%b,corrupt=%b)", this.file.toString(), this.firstTxId, this.lastTxId, this.isInProgress(), this.isCorrupt);
        }
    }
}

