/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.io.Writer;
import java.nio.channels.ClosedChannelException;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.DF;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.protocol.Block;
import org.apache.hadoop.hdfs.protocol.BlockListAsLongs;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.datanode.BlockMetadataHeader;
import org.apache.hadoop.hdfs.server.datanode.DatanodeUtil;
import org.apache.hadoop.hdfs.server.datanode.FileIoProvider;
import org.apache.hadoop.hdfs.server.datanode.checker.VolumeCheckResult;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.DataNodeVolumeMetrics;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeReference;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.BlockPoolSlice;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsDatasetImpl;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.RamDiskReplicaTracker;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.ReplicaMap;
import org.apache.hadoop.hdfs.server.protocol.DatanodeStorage;
import org.apache.hadoop.util.AutoCloseableLock;
import org.apache.hadoop.util.CloseableReferenceCount;
import org.apache.hadoop.util.DataChecksum;
import org.apache.hadoop.util.DiskChecker;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Time;
import org.apache.hadoop.util.Timer;
import org.codehaus.jackson.annotate.JsonProperty;
import org.codehaus.jackson.map.ObjectMapper;
import org.codehaus.jackson.map.ObjectReader;
import org.codehaus.jackson.map.ObjectWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@VisibleForTesting
public class FsVolumeImpl
implements FsVolumeSpi {
    public static final Logger LOG = LoggerFactory.getLogger(FsVolumeImpl.class);
    private static final ObjectWriter WRITER = new ObjectMapper().writerWithDefaultPrettyPrinter();
    private static final ObjectReader READER = new ObjectMapper().reader(BlockIteratorState.class);
    private final FsDatasetImpl dataset;
    private final String storageID;
    private final StorageType storageType;
    private final Map<String, BlockPoolSlice> bpSlices = new ConcurrentHashMap<String, BlockPoolSlice>();
    private final File currentDir;
    private final DF usage;
    private final long reserved;
    private CloseableReferenceCount reference = new CloseableReferenceCount();
    private AtomicLong reservedForReplicas;
    private long recentReserved = 0L;
    protected volatile long configuredCapacity;
    private final FileIoProvider fileIoProvider;
    private final DataNodeVolumeMetrics metrics;
    protected ThreadPoolExecutor cacheExecutor;

    FsVolumeImpl(FsDatasetImpl dataset, String storageID, File currentDir, Configuration conf, StorageType storageType) throws IOException {
        this.dataset = dataset;
        this.storageID = storageID;
        this.reserved = conf.getLong("dfs.datanode.du.reserved." + StringUtils.toLowerCase((String)storageType.toString()), conf.getLong("dfs.datanode.du.reserved", 0L));
        this.reservedForReplicas = new AtomicLong(0L);
        this.currentDir = currentDir;
        File parent = currentDir.getParentFile();
        this.usage = new DF(parent, conf);
        this.storageType = storageType;
        this.configuredCapacity = -1L;
        this.fileIoProvider = dataset.datanode != null ? dataset.datanode.getFileIoProvider() : new FileIoProvider(conf, dataset.datanode);
        this.cacheExecutor = this.initializeCacheExecutor(parent);
        this.metrics = DataNodeVolumeMetrics.create(conf, parent.getAbsolutePath());
    }

    protected ThreadPoolExecutor initializeCacheExecutor(File parent) {
        if (this.storageType.isTransient()) {
            return null;
        }
        if (this.dataset.datanode == null) {
            return null;
        }
        int maxNumThreads = this.dataset.datanode.getConf().getInt("dfs.datanode.fsdatasetcache.max.threads.per.volume", 4);
        ThreadFactory workerFactory = new ThreadFactoryBuilder().setDaemon(true).setNameFormat("FsVolumeImplWorker-" + parent.toString() + "-%d").build();
        ThreadPoolExecutor executor = new ThreadPoolExecutor(1, maxNumThreads, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), workerFactory);
        executor.allowCoreThreadTimeOut(true);
        return executor;
    }

    private void printReferenceTraceInfo(String op) {
        StackTraceElement[] stack;
        for (StackTraceElement ste : stack = Thread.currentThread().getStackTrace()) {
            switch (ste.getMethodName()) {
                case "getDfsUsed": 
                case "getBlockPoolUsed": 
                case "getAvailable": 
                case "getVolumeMap": {
                    return;
                }
            }
        }
        FsDatasetImpl.LOG.trace("Reference count: " + op + " " + this + ": " + this.reference.getReferenceCount());
        FsDatasetImpl.LOG.trace(Joiner.on((String)"\n").join((Object[])Thread.currentThread().getStackTrace()));
    }

    private void reference() throws ClosedChannelException {
        this.reference.reference();
        if (FsDatasetImpl.LOG.isTraceEnabled()) {
            this.printReferenceTraceInfo("incr");
        }
    }

    private void unreference() {
        if (FsDatasetImpl.LOG.isTraceEnabled()) {
            this.printReferenceTraceInfo("desc");
        }
        if (FsDatasetImpl.LOG.isDebugEnabled() && this.reference.getReferenceCount() <= 0) {
            FsDatasetImpl.LOG.debug("Decrease reference count <= 0 on " + this + Joiner.on((String)"\n").join((Object[])Thread.currentThread().getStackTrace()));
        }
        this.checkReference();
        this.reference.unreference();
    }

    @Override
    public FsVolumeReference obtainReference() throws ClosedChannelException {
        return new FsVolumeReferenceImpl(this);
    }

    private void checkReference() {
        Preconditions.checkState((this.reference.getReferenceCount() > 0 ? 1 : 0) != 0);
    }

    @VisibleForTesting
    int getReferenceCount() {
        return this.reference.getReferenceCount();
    }

    void setClosed() throws IOException {
        try {
            this.reference.setClosed();
            this.dataset.stopAllDataxceiverThreads(this);
        }
        catch (ClosedChannelException e) {
            throw new IOException("The volume has already closed.", e);
        }
    }

    boolean checkClosed() {
        if (this.reference.getReferenceCount() > 0) {
            if (FsDatasetImpl.LOG.isDebugEnabled()) {
                FsDatasetImpl.LOG.debug(String.format("The reference count for %s is %d, wait to be 0.", this, this.reference.getReferenceCount()));
            }
            return false;
        }
        return true;
    }

    File getCurrentDir() {
        return this.currentDir;
    }

    File getRbwDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getRbwDir();
    }

    File getLazyPersistDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getLazypersistDir();
    }

    File getTmpDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getTmpDir();
    }

    void onBlockFileDeletion(String bpid, long value) {
        this.decDfsUsedAndNumBlocks(bpid, value, true);
        if (this.isTransientStorage()) {
            this.dataset.releaseLockedMemory(value, true);
        }
    }

    void onMetaFileDeletion(String bpid, long value) {
        this.decDfsUsedAndNumBlocks(bpid, value, false);
    }

    private void decDfsUsedAndNumBlocks(String bpid, long value, boolean blockFileDeleted) {
        try (AutoCloseableLock lock = this.dataset.acquireDatasetLock();){
            BlockPoolSlice bp = this.bpSlices.get(bpid);
            if (bp != null) {
                bp.decDfsUsed(value);
                if (blockFileDeleted) {
                    bp.decrNumBlocks();
                }
            }
        }
    }

    void incDfsUsedAndNumBlocks(String bpid, long value) {
        try (AutoCloseableLock lock = this.dataset.acquireDatasetLock();){
            BlockPoolSlice bp = this.bpSlices.get(bpid);
            if (bp != null) {
                bp.incDfsUsed(value);
                bp.incrNumBlocks();
            }
        }
    }

    void incDfsUsed(String bpid, long value) {
        try (AutoCloseableLock lock = this.dataset.acquireDatasetLock();){
            BlockPoolSlice bp = this.bpSlices.get(bpid);
            if (bp != null) {
                bp.incDfsUsed(value);
            }
        }
    }

    @VisibleForTesting
    public long getDfsUsed() throws IOException {
        long dfsUsed = 0L;
        try (AutoCloseableLock lock = this.dataset.acquireDatasetLock();){
            for (BlockPoolSlice s : this.bpSlices.values()) {
                dfsUsed += s.getDfsUsed();
            }
        }
        return dfsUsed;
    }

    long getBlockPoolUsed(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getDfsUsed();
    }

    @VisibleForTesting
    public long getCapacity() {
        if (this.configuredCapacity < 0L) {
            long remaining = this.usage.getCapacity() - this.reserved;
            return remaining > 0L ? remaining : 0L;
        }
        return this.configuredCapacity;
    }

    @VisibleForTesting
    public void setCapacityForTesting(long capacity) {
        this.configuredCapacity = capacity;
    }

    @Override
    public long getAvailable() throws IOException {
        long available;
        long remaining = this.getCapacity() - this.getDfsUsed() - this.getReservedForReplicas();
        if (remaining > (available = this.usage.getAvailable() - this.getRemainingReserved() - this.getReservedForReplicas())) {
            remaining = available;
        }
        return remaining > 0L ? remaining : 0L;
    }

    long getActualNonDfsUsed() throws IOException {
        return this.usage.getUsed() - this.getDfsUsed();
    }

    private long getRemainingReserved() throws IOException {
        long actualNonDfsUsed = this.getActualNonDfsUsed();
        if (actualNonDfsUsed < this.reserved) {
            return this.reserved - actualNonDfsUsed;
        }
        return 0L;
    }

    public long getNonDfsUsed() throws IOException {
        long actualNonDfsUsed = this.getActualNonDfsUsed();
        if (actualNonDfsUsed < this.reserved) {
            return 0L;
        }
        return actualNonDfsUsed - this.reserved;
    }

    @VisibleForTesting
    long getDfAvailable() {
        return this.usage.getAvailable();
    }

    @VisibleForTesting
    public long getReservedForReplicas() {
        return this.reservedForReplicas.get();
    }

    @VisibleForTesting
    long getRecentReserved() {
        return this.recentReserved;
    }

    long getReserved() {
        return this.reserved;
    }

    BlockPoolSlice getBlockPoolSlice(String bpid) throws IOException {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp == null) {
            throw new IOException("block pool " + bpid + " is not found");
        }
        return bp;
    }

    @Override
    public String getBasePath() {
        return this.currentDir.getParent();
    }

    @Override
    public boolean isTransientStorage() {
        return this.storageType.isTransient();
    }

    @Override
    public String getPath(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getDirectory().getAbsolutePath();
    }

    @Override
    public File getFinalizedDir(String bpid) throws IOException {
        return this.getBlockPoolSlice(bpid).getFinalizedDir();
    }

    @Override
    public String[] getBlockPoolList() {
        return this.bpSlices.keySet().toArray(new String[this.bpSlices.keySet().size()]);
    }

    File createTmpFile(String bpid, Block b) throws IOException {
        this.checkReference();
        this.reserveSpaceForReplica(b.getNumBytes());
        try {
            return this.getBlockPoolSlice(bpid).createTmpFile(b);
        }
        catch (IOException exception) {
            this.releaseReservedSpace(b.getNumBytes());
            throw exception;
        }
    }

    @Override
    public void reserveSpaceForReplica(long bytesToReserve) {
        if (bytesToReserve != 0L) {
            this.reservedForReplicas.addAndGet(bytesToReserve);
            this.recentReserved = bytesToReserve;
        }
    }

    @Override
    public void releaseReservedSpace(long bytesToRelease) {
        if (bytesToRelease != 0L) {
            long newReservation;
            long oldReservation;
            do {
                if ((newReservation = (oldReservation = this.reservedForReplicas.get()) - bytesToRelease) >= 0L) continue;
                newReservation = 0L;
            } while (!this.reservedForReplicas.compareAndSet(oldReservation, newReservation));
        }
    }

    @Override
    public void releaseLockedMemory(long bytesToRelease) {
        if (this.isTransientStorage()) {
            this.dataset.releaseLockedMemory(bytesToRelease, false);
        }
    }

    @VisibleForTesting
    public static String nextSorted(List<String> arr, String prev) {
        int res = 0;
        if (prev != null) {
            res = Collections.binarySearch(arr, prev);
            res = res < 0 ? -1 - res : ++res;
        }
        if (res >= arr.size()) {
            return null;
        }
        return arr.get(res);
    }

    @Override
    public FsVolumeSpi.BlockIterator newBlockIterator(String bpid, String name) {
        return new BlockIteratorImpl(bpid, name);
    }

    @Override
    public FsVolumeSpi.BlockIterator loadBlockIterator(String bpid, String name) throws IOException {
        BlockIteratorImpl iter = new BlockIteratorImpl(bpid, name);
        iter.load();
        return iter;
    }

    @Override
    public FsDatasetSpi<? extends FsVolumeSpi> getDataset() {
        return this.dataset;
    }

    File createRbwFile(String bpid, Block b) throws IOException {
        this.checkReference();
        this.reserveSpaceForReplica(b.getNumBytes());
        try {
            return this.getBlockPoolSlice(bpid).createRbwFile(b);
        }
        catch (IOException exception) {
            this.releaseReservedSpace(b.getNumBytes());
            throw exception;
        }
    }

    File addFinalizedBlock(String bpid, Block b, File f, long bytesReserved) throws IOException {
        this.releaseReservedSpace(bytesReserved);
        return this.getBlockPoolSlice(bpid).addBlock(b, f);
    }

    Executor getCacheExecutor() {
        return this.cacheExecutor;
    }

    @Override
    public VolumeCheckResult check(FsVolumeSpi.VolumeCheckContext ignored) throws DiskChecker.DiskErrorException {
        for (BlockPoolSlice s : this.bpSlices.values()) {
            s.checkDirs();
        }
        return VolumeCheckResult.HEALTHY;
    }

    void getVolumeMap(ReplicaMap volumeMap, RamDiskReplicaTracker ramDiskReplicaMap) throws IOException {
        for (BlockPoolSlice s : this.bpSlices.values()) {
            s.getVolumeMap(volumeMap, ramDiskReplicaMap);
        }
    }

    void getVolumeMap(String bpid, ReplicaMap volumeMap, RamDiskReplicaTracker ramDiskReplicaMap) throws IOException {
        this.getBlockPoolSlice(bpid).getVolumeMap(volumeMap, ramDiskReplicaMap);
    }

    long getNumBlocks() {
        long numBlocks = 0L;
        for (BlockPoolSlice s : this.bpSlices.values()) {
            numBlocks += s.getNumOfBlocks();
        }
        return numBlocks;
    }

    public String toString() {
        return this.currentDir.getAbsolutePath();
    }

    void shutdown() {
        if (this.cacheExecutor != null) {
            this.cacheExecutor.shutdown();
        }
        Set<Map.Entry<String, BlockPoolSlice>> set = this.bpSlices.entrySet();
        for (Map.Entry<String, BlockPoolSlice> entry : set) {
            entry.getValue().shutdown(null);
        }
        if (this.metrics != null) {
            this.metrics.unRegister();
        }
    }

    void addBlockPool(String bpid, Configuration conf) throws IOException {
        this.addBlockPool(bpid, conf, null);
    }

    void addBlockPool(String bpid, Configuration conf, Timer timer) throws IOException {
        File bpdir = new File(this.currentDir, bpid);
        BlockPoolSlice bp = timer == null ? new BlockPoolSlice(bpid, this, bpdir, conf, new Timer()) : new BlockPoolSlice(bpid, this, bpdir, conf, timer);
        this.bpSlices.put(bpid, bp);
    }

    void shutdownBlockPool(String bpid, BlockListAsLongs blocksListsAsLongs) {
        BlockPoolSlice bp = this.bpSlices.get(bpid);
        if (bp != null) {
            bp.shutdown(blocksListsAsLongs);
        }
        this.bpSlices.remove(bpid);
    }

    boolean isBPDirEmpty(String bpid) throws IOException {
        File volumeCurrentDir = this.getCurrentDir();
        File bpDir = new File(volumeCurrentDir, bpid);
        File bpCurrentDir = new File(bpDir, "current");
        File finalizedDir = new File(bpCurrentDir, "finalized");
        File rbwDir = new File(bpCurrentDir, "rbw");
        if (this.fileIoProvider.exists(this, finalizedDir) && !DatanodeUtil.dirNoFilesRecursive(this, finalizedDir, this.fileIoProvider)) {
            return false;
        }
        return !this.fileIoProvider.exists(this, rbwDir) || this.fileIoProvider.list(this, rbwDir).length == 0;
    }

    void deleteBPDirectories(String bpid, boolean force) throws IOException {
        File volumeCurrentDir = this.getCurrentDir();
        File bpDir = new File(volumeCurrentDir, bpid);
        if (!bpDir.isDirectory()) {
            return;
        }
        File tmpDir = new File(bpDir, "tmp");
        File bpCurrentDir = new File(bpDir, "current");
        File finalizedDir = new File(bpCurrentDir, "finalized");
        File lazypersistDir = new File(bpCurrentDir, "lazypersist");
        File rbwDir = new File(bpCurrentDir, "rbw");
        if (force) {
            this.fileIoProvider.fullyDelete(this, bpDir);
        } else {
            if (!this.fileIoProvider.delete(this, rbwDir)) {
                throw new IOException("Failed to delete " + rbwDir);
            }
            if (!DatanodeUtil.dirNoFilesRecursive(this, finalizedDir, this.fileIoProvider) || !this.fileIoProvider.fullyDelete(this, finalizedDir)) {
                throw new IOException("Failed to delete " + finalizedDir);
            }
            if (!(!lazypersistDir.exists() || DatanodeUtil.dirNoFilesRecursive(this, lazypersistDir, this.fileIoProvider) && this.fileIoProvider.fullyDelete(this, lazypersistDir))) {
                throw new IOException("Failed to delete " + lazypersistDir);
            }
            this.fileIoProvider.fullyDelete(this, tmpDir);
            for (File f : this.fileIoProvider.listFiles(this, bpCurrentDir)) {
                if (this.fileIoProvider.delete(this, f)) continue;
                throw new IOException("Failed to delete " + f);
            }
            if (!this.fileIoProvider.delete(this, bpCurrentDir)) {
                throw new IOException("Failed to delete " + bpCurrentDir);
            }
            for (File f : this.fileIoProvider.listFiles(this, bpDir)) {
                if (this.fileIoProvider.delete(this, f)) continue;
                throw new IOException("Failed to delete " + f);
            }
            if (!this.fileIoProvider.delete(this, bpDir)) {
                throw new IOException("Failed to delete " + bpDir);
            }
        }
    }

    @Override
    public String getStorageID() {
        return this.storageID;
    }

    @Override
    public StorageType getStorageType() {
        return this.storageType;
    }

    DatanodeStorage toDatanodeStorage() {
        return new DatanodeStorage(this.storageID, DatanodeStorage.State.NORMAL, this.storageType);
    }

    @Override
    public byte[] loadLastPartialChunkChecksum(File blockFile, File metaFile) throws IOException {
        DataChecksum dcs;
        try (FileInputStream fis = this.fileIoProvider.getFileInputStream(this, metaFile);){
            dcs = BlockMetadataHeader.readHeader((FileInputStream)fis).getChecksum();
        }
        int checksumSize = dcs.getChecksumSize();
        long onDiskLen = blockFile.length();
        int bytesPerChecksum = dcs.getBytesPerChecksum();
        if (onDiskLen % (long)bytesPerChecksum == 0L) {
            return null;
        }
        long offsetInChecksum = (long)BlockMetadataHeader.getHeaderSize() + onDiskLen / (long)bytesPerChecksum * (long)checksumSize;
        byte[] lastChecksum = new byte[checksumSize];
        try (RandomAccessFile raf = this.fileIoProvider.getRandomAccessFile(this, metaFile, "r");){
            raf.seek(offsetInChecksum);
            int readBytes = raf.read(lastChecksum, 0, checksumSize);
            if (readBytes == -1) {
                throw new IOException("Expected to read " + checksumSize + " bytes from offset " + offsetInChecksum + " but reached end of file.");
            }
            if (readBytes != checksumSize) {
                throw new IOException("Expected to read " + checksumSize + " bytes from offset " + offsetInChecksum + " but read " + readBytes + " bytes.");
            }
        }
        return lastChecksum;
    }

    @Override
    public FileIoProvider getFileIoProvider() {
        return this.fileIoProvider;
    }

    @Override
    public DataNodeVolumeMetrics getMetrics() {
        return this.metrics;
    }

    private class BlockIteratorImpl
    implements FsVolumeSpi.BlockIterator {
        private final File bpidDir;
        private final String name;
        private final String bpid;
        private long maxStalenessMs = 0L;
        private List<String> cache;
        private long cacheMs;
        private BlockIteratorState state;

        BlockIteratorImpl(String bpid, String name) {
            this.bpidDir = new File(FsVolumeImpl.this.currentDir, bpid);
            this.name = name;
            this.bpid = bpid;
            this.rewind();
        }

        private String getNextSubDir(String prev, File dir) throws IOException {
            List<String> children = FsVolumeImpl.this.fileIoProvider.listDirectory(FsVolumeImpl.this, dir, SubdirFilter.INSTANCE);
            this.cache = null;
            this.cacheMs = 0L;
            if (children.size() == 0) {
                LOG.trace("getNextSubDir({}, {}): no subdirectories found in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, dir.getAbsolutePath()});
                return null;
            }
            Collections.sort(children);
            String nextSubDir = FsVolumeImpl.nextSorted(children, prev);
            if (nextSubDir == null) {
                LOG.trace("getNextSubDir({}, {}): no more subdirectories found in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, dir.getAbsolutePath()});
            } else {
                LOG.trace("getNextSubDir({}, {}): picking next subdirectory {} within {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, nextSubDir, dir.getAbsolutePath()});
            }
            return nextSubDir;
        }

        private String getNextFinalizedDir() throws IOException {
            File dir = Paths.get(this.bpidDir.getAbsolutePath(), "current", "finalized").toFile();
            return this.getNextSubDir(this.state.curFinalizedDir, dir);
        }

        private String getNextFinalizedSubDir() throws IOException {
            if (this.state.curFinalizedDir == null) {
                return null;
            }
            File dir = Paths.get(this.bpidDir.getAbsolutePath(), "current", "finalized", this.state.curFinalizedDir).toFile();
            return this.getNextSubDir(this.state.curFinalizedSubDir, dir);
        }

        private List<String> getSubdirEntries() throws IOException {
            if (this.state.curFinalizedSubDir == null) {
                return null;
            }
            long now = Time.monotonicNow();
            if (this.cache != null) {
                long delta = now - this.cacheMs;
                if (delta < this.maxStalenessMs) {
                    return this.cache;
                }
                LOG.trace("getSubdirEntries({}, {}): purging entries cache for {} after {} ms.", new Object[]{FsVolumeImpl.this.storageID, this.bpid, this.state.curFinalizedSubDir, delta});
                this.cache = null;
            }
            File dir = Paths.get(this.bpidDir.getAbsolutePath(), "current", "finalized", this.state.curFinalizedDir, this.state.curFinalizedSubDir).toFile();
            List<String> entries = FsVolumeImpl.this.fileIoProvider.listDirectory(FsVolumeImpl.this, dir, BlockFileFilter.INSTANCE);
            if (entries.size() == 0) {
                entries = null;
            } else {
                Collections.sort(entries);
            }
            if (entries == null) {
                LOG.trace("getSubdirEntries({}, {}): no entries found in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, dir.getAbsolutePath()});
            } else {
                LOG.trace("getSubdirEntries({}, {}): listed {} entries in {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, entries.size(), dir.getAbsolutePath()});
            }
            this.cache = entries;
            this.cacheMs = now;
            return this.cache;
        }

        @Override
        public ExtendedBlock nextBlock() throws IOException {
            if (this.state.atEnd) {
                return null;
            }
            try {
                while (true) {
                    List<String> entries;
                    if ((entries = this.getSubdirEntries()) != null) {
                        this.state.curEntry = FsVolumeImpl.nextSorted(entries, this.state.curEntry);
                        if (this.state.curEntry == null) {
                            LOG.trace("nextBlock({}, {}): advancing from {} to next subdirectory.", new Object[]{FsVolumeImpl.this.storageID, this.bpid, this.state.curFinalizedSubDir});
                        } else {
                            ExtendedBlock block = new ExtendedBlock(this.bpid, Block.filename2id((String)this.state.curEntry));
                            File expectedBlockDir = DatanodeUtil.idToBlockDir(new File("."), block.getBlockId());
                            File actualBlockDir = Paths.get(".", this.state.curFinalizedDir, this.state.curFinalizedSubDir).toFile();
                            if (!expectedBlockDir.equals(actualBlockDir)) {
                                LOG.error("nextBlock({}, {}): block id {} found in invalid directory.  Expected directory: {}.  Actual directory: {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, block.getBlockId(), expectedBlockDir.getPath(), actualBlockDir.getPath()});
                                continue;
                            }
                            LOG.trace("nextBlock({}, {}): advancing to {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, block});
                            return block;
                        }
                    }
                    this.state.curFinalizedSubDir = this.getNextFinalizedSubDir();
                    if (this.state.curFinalizedSubDir != null) continue;
                    this.state.curFinalizedDir = this.getNextFinalizedDir();
                    if (this.state.curFinalizedDir == null) break;
                }
                this.state.atEnd = true;
                return null;
            }
            catch (IOException e) {
                this.state.atEnd = true;
                LOG.error("nextBlock({}, {}): I/O error", new Object[]{FsVolumeImpl.this.storageID, this.bpid, e});
                throw e;
            }
        }

        @Override
        public boolean atEnd() {
            return this.state.atEnd;
        }

        @Override
        public void rewind() {
            this.cache = null;
            this.cacheMs = 0L;
            this.state = new BlockIteratorState();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void save() throws IOException {
            this.state.lastSavedMs = Time.now();
            boolean success = false;
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)FsVolumeImpl.this.fileIoProvider.getFileOutputStream((FsVolumeSpi)FsVolumeImpl.this, this.getTempSaveFile()), "UTF-8"));){
                WRITER.writeValue((Writer)writer, (Object)this.state);
                success = true;
            }
            finally {
                if (!success) {
                    FsVolumeImpl.this.fileIoProvider.delete(FsVolumeImpl.this, this.getTempSaveFile());
                }
            }
            FsVolumeImpl.this.fileIoProvider.move(FsVolumeImpl.this, this.getTempSaveFile().toPath(), this.getSaveFile().toPath(), StandardCopyOption.ATOMIC_MOVE);
            if (LOG.isTraceEnabled()) {
                LOG.trace("save({}, {}): saved {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, WRITER.writeValueAsString((Object)this.state)});
            }
        }

        public void load() throws IOException {
            File file = this.getSaveFile();
            this.state = (BlockIteratorState)READER.readValue(file);
            LOG.trace("load({}, {}): loaded iterator {} from {}: {}", new Object[]{FsVolumeImpl.this.storageID, this.bpid, this.name, file.getAbsoluteFile(), WRITER.writeValueAsString((Object)this.state)});
        }

        File getSaveFile() {
            return new File(this.bpidDir, this.name + ".cursor");
        }

        File getTempSaveFile() {
            return new File(this.bpidDir, this.name + ".cursor.tmp");
        }

        @Override
        public void setMaxStalenessMs(long maxStalenessMs) {
            this.maxStalenessMs = maxStalenessMs;
        }

        @Override
        public void close() throws IOException {
        }

        @Override
        public long getIterStartMs() {
            return this.state.iterStartMs;
        }

        @Override
        public long getLastSavedMs() {
            return this.state.lastSavedMs;
        }

        @Override
        public String getBlockPoolId() {
            return this.bpid;
        }
    }

    private static class BlockIteratorState {
        @JsonProperty
        private long lastSavedMs;
        @JsonProperty
        private long iterStartMs;
        @JsonProperty
        private String curFinalizedDir;
        @JsonProperty
        private String curFinalizedSubDir;
        @JsonProperty
        private String curEntry;
        @JsonProperty
        private boolean atEnd;

        BlockIteratorState() {
            this.lastSavedMs = this.iterStartMs = Time.now();
            this.curFinalizedDir = null;
            this.curFinalizedSubDir = null;
            this.curEntry = null;
            this.atEnd = false;
        }
    }

    private static enum BlockFileFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return !name.endsWith(".meta") && name.startsWith("blk_");
        }
    }

    private static enum SubdirFilter implements FilenameFilter
    {
        INSTANCE;


        @Override
        public boolean accept(File dir, String name) {
            return name.startsWith("subdir");
        }
    }

    private static class FsVolumeReferenceImpl
    implements FsVolumeReference {
        private FsVolumeImpl volume;

        FsVolumeReferenceImpl(FsVolumeImpl volume) throws ClosedChannelException {
            this.volume = volume;
            volume.reference();
        }

        @Override
        public void close() throws IOException {
            if (this.volume != null) {
                this.volume.unreference();
                this.volume = null;
            }
        }

        @Override
        public FsVolumeSpi getVolume() {
            return this.volume;
        }
    }
}

