/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public class ProcfsBasedProcessTree {
    static final Log LOG = LogFactory.getLog(ProcfsBasedProcessTree.class);
    private static final String PROCFS = "/proc/";
    private static final Pattern PROCFS_STAT_FILE_FORMAT = Pattern.compile("^([0-9-]+)\\s([^\\s]+)\\s[^\\s]\\s([0-9-]+)\\s([0-9-]+)\\s([0-9-]+)\\s([0-9-]+\\s){7}([0-9]+)\\s([0-9]+)\\s([0-9-]+\\s){7}([0-9]+)\\s([0-9]+)(\\s[0-9-]+){15}");
    public static final String PROCFS_STAT_FILE = "stat";
    public static final String PROCFS_CMDLINE_FILE = "cmdline";
    public static final long PAGE_SIZE;
    public static final long JIFFY_LENGTH_IN_MILLIS;
    private String procfsDir;
    protected final Integer pid;
    private Long cpuTime = 0L;
    private boolean setsidUsed = false;
    protected Map<Integer, ProcessInfo> processTree = new HashMap<Integer, ProcessInfo>();
    private static final String PROCESSTREE_DUMP_FORMAT = "\t|- %d %d %d %d %s %d %d %d %d %s\n";

    public ProcfsBasedProcessTree(String pid) {
        this(pid, false);
    }

    public ProcfsBasedProcessTree(String pid, boolean setsidUsed) {
        this(pid, setsidUsed, PROCFS);
    }

    public ProcfsBasedProcessTree(String pid, boolean setsidUsed, String procfsDir) {
        this.pid = ProcfsBasedProcessTree.getValidPID(pid);
        this.setsidUsed = setsidUsed;
        this.procfsDir = procfsDir;
    }

    public static boolean isAvailable() {
        try {
            String osName = System.getProperty("os.name");
            if (!osName.startsWith("Linux")) {
                LOG.info((Object)"ProcfsBasedProcessTree currently is supported only on Linux.");
                return false;
            }
        }
        catch (SecurityException se) {
            LOG.warn((Object)("Failed to get Operating System name. " + se));
            return false;
        }
        return true;
    }

    public ProcfsBasedProcessTree getProcessTree() {
        if (this.pid != -1) {
            List<Integer> processList = this.getProcessList();
            HashMap<Integer, ProcessInfo> allProcessInfo = new HashMap<Integer, ProcessInfo>();
            HashMap<Integer, ProcessInfo> oldProcs = new HashMap<Integer, ProcessInfo>(this.processTree);
            this.processTree.clear();
            ProcessInfo me = null;
            for (Integer n : processList) {
                ProcessInfo pInfo = new ProcessInfo(n);
                if (ProcfsBasedProcessTree.constructProcessInfo(pInfo, this.procfsDir) == null) continue;
                allProcessInfo.put(n, pInfo);
                if (!n.equals(this.pid)) continue;
                me = pInfo;
                this.processTree.put(n, pInfo);
            }
            if (me == null) {
                return this;
            }
            for (Map.Entry entry : allProcessInfo.entrySet()) {
                ProcessInfo pInfo;
                ProcessInfo parentPInfo;
                Integer pID = (Integer)entry.getKey();
                if (pID == 1 || (parentPInfo = (ProcessInfo)allProcessInfo.get((pInfo = (ProcessInfo)entry.getValue()).getPpid())) == null) continue;
                parentPInfo.addChild(pInfo);
            }
            LinkedList<ProcessInfo> pInfoQueue = new LinkedList<ProcessInfo>();
            pInfoQueue.addAll(me.getChildren());
            while (!pInfoQueue.isEmpty()) {
                ProcessInfo processInfo = (ProcessInfo)pInfoQueue.remove();
                if (!this.processTree.containsKey(processInfo.getPid())) {
                    this.processTree.put(processInfo.getPid(), processInfo);
                }
                pInfoQueue.addAll(processInfo.getChildren());
            }
            for (Map.Entry<Integer, ProcessInfo> procs : this.processTree.entrySet()) {
                ProcessInfo oldInfo = (ProcessInfo)oldProcs.get(procs.getKey());
                if (procs.getValue() == null) continue;
                procs.getValue().updateJiffy(oldInfo);
                if (oldInfo == null) continue;
                procs.getValue().updateAge(oldInfo);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)this.toString());
            }
        }
        return this;
    }

    public boolean checkPidPgrpidForMatch() {
        return ProcfsBasedProcessTree.checkPidPgrpidForMatch(this.pid, PROCFS);
    }

    public static boolean checkPidPgrpidForMatch(int _pid, String procfs) {
        ProcessInfo pInfo = new ProcessInfo(_pid);
        return (pInfo = ProcfsBasedProcessTree.constructProcessInfo(pInfo, procfs)) == null || pInfo.getPgrpId().equals(_pid);
    }

    public List<Integer> getCurrentProcessIDs() {
        ArrayList<Integer> currentPIDs = new ArrayList<Integer>();
        currentPIDs.addAll(this.processTree.keySet());
        return currentPIDs;
    }

    public String getProcessTreeDump() {
        StringBuilder ret = new StringBuilder();
        ret.append(String.format("\t|- PID PPID PGRPID SESSID CMD_NAME USER_MODE_TIME(MILLIS) SYSTEM_TIME(MILLIS) VMEM_USAGE(BYTES) RSSMEM_USAGE(PAGES) FULL_CMD_LINE\n", new Object[0]));
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            ret.append(String.format(PROCESSTREE_DUMP_FORMAT, p.getPid(), p.getPpid(), p.getPgrpId(), p.getSessionId(), p.getName(), p.getUtime(), p.getStime(), p.getVmem(), p.getRssmemPage(), p.getCmdLine(this.procfsDir)));
        }
        return ret.toString();
    }

    public long getCumulativeVmem() {
        return this.getCumulativeVmem(0);
    }

    public long getCumulativeRssmem() {
        return this.getCumulativeRssmem(0);
    }

    public long getCumulativeVmem(int olderThanAge) {
        long total = 0L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null || p.getAge() <= olderThanAge) continue;
            total += p.getVmem().longValue();
        }
        return total;
    }

    public long getCumulativeRssmem(int olderThanAge) {
        if (PAGE_SIZE < 0L) {
            return 0L;
        }
        long totalPages = 0L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null || p.getAge() <= olderThanAge) continue;
            totalPages += p.getRssmemPage().longValue();
        }
        return totalPages * PAGE_SIZE;
    }

    public long getCumulativeCpuTime() {
        if (JIFFY_LENGTH_IN_MILLIS < 0L) {
            return 0L;
        }
        long incJiffies = 0L;
        for (ProcessInfo p : this.processTree.values()) {
            if (p == null) continue;
            incJiffies += p.dtime.longValue();
        }
        this.cpuTime = this.cpuTime + incJiffies * JIFFY_LENGTH_IN_MILLIS;
        return this.cpuTime;
    }

    private static Integer getValidPID(String pid) {
        Integer retPid = -1;
        try {
            retPid = Integer.parseInt(pid);
            if (retPid <= 0) {
                retPid = -1;
            }
        }
        catch (NumberFormatException nfe) {
            retPid = -1;
        }
        return retPid;
    }

    private List<Integer> getProcessList() {
        String[] processDirs = new File(this.procfsDir).list();
        ArrayList<Integer> processList = new ArrayList<Integer>();
        for (String dir : processDirs) {
            try {
                int pd = Integer.parseInt(dir);
                if (!new File(this.procfsDir, dir).isDirectory()) continue;
                processList.add(pd);
            }
            catch (NumberFormatException n) {
            }
            catch (SecurityException s) {
                // empty catch block
            }
        }
        return processList;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ProcessInfo constructProcessInfo(ProcessInfo pinfo, String procfsDir) {
        ProcessInfo ret = null;
        BufferedReader in = null;
        FileReader fReader = null;
        try {
            File pidDir = new File(procfsDir, String.valueOf(pinfo.getPid()));
            fReader = new FileReader(new File(pidDir, PROCFS_STAT_FILE));
            in = new BufferedReader(fReader);
        }
        catch (FileNotFoundException f) {
            LOG.warn((Object)("The process " + pinfo.getPid() + " may have finished in the interim."));
            return ret;
        }
        ret = pinfo;
        try {
            String str = in.readLine();
            Matcher m = PROCFS_STAT_FILE_FORMAT.matcher(str);
            boolean mat = m.find();
            if (mat) {
                pinfo.updateProcessInfo(m.group(2), Integer.parseInt(m.group(3)), Integer.parseInt(m.group(4)), Integer.parseInt(m.group(5)), Long.parseLong(m.group(7)), Long.parseLong(m.group(8)), Long.parseLong(m.group(10)), Long.parseLong(m.group(11)));
            } else {
                LOG.warn((Object)("Unexpected: procfs stat file is not in the expected format for process with pid " + pinfo.getPid()));
                ret = null;
            }
        }
        catch (IOException io) {
            LOG.warn((Object)("Error reading the stream " + io));
            ret = null;
        }
        finally {
            try {
                fReader.close();
                try {
                    in.close();
                }
                catch (IOException i) {
                    LOG.warn((Object)("Error closing the stream " + in));
                }
            }
            catch (IOException i) {
                LOG.warn((Object)("Error closing the stream " + fReader));
            }
        }
        return ret;
    }

    public String toString() {
        StringBuffer pTree = new StringBuffer("[ ");
        for (Integer p : this.processTree.keySet()) {
            pTree.append(p);
            pTree.append(" ");
        }
        return pTree.substring(0, pTree.length()) + "]";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static {
        Shell.ShellCommandExecutor shellExecutor = new Shell.ShellCommandExecutor(new String[]{"getconf", "PAGESIZE"});
        long pageSize = -1L;
        try {
            shellExecutor.execute();
            pageSize = Long.parseLong(shellExecutor.getOutput().replace("\n", ""));
        }
        catch (IOException e) {
            LOG.error((Object)StringUtils.stringifyException((Throwable)e));
        }
        finally {
            PAGE_SIZE = pageSize;
        }
        shellExecutor = new Shell.ShellCommandExecutor(new String[]{"getconf", "CLK_TCK"});
        long jiffiesPerSecond = -1L;
        try {
            shellExecutor.execute();
            jiffiesPerSecond = Long.parseLong(shellExecutor.getOutput().replace("\n", ""));
            JIFFY_LENGTH_IN_MILLIS = jiffiesPerSecond != -1L ? Math.round(1000.0 / (double)jiffiesPerSecond) : -1L;
        }
        catch (IOException e) {
            try {
                LOG.error((Object)StringUtils.stringifyException((Throwable)e));
                JIFFY_LENGTH_IN_MILLIS = jiffiesPerSecond != -1L ? Math.round(1000.0 / (double)jiffiesPerSecond) : -1L;
            }
            catch (Throwable throwable) {
                JIFFY_LENGTH_IN_MILLIS = jiffiesPerSecond != -1L ? Math.round(1000.0 / (double)jiffiesPerSecond) : -1L;
                throw throwable;
            }
        }
    }

    private static class ProcessInfo {
        private Integer pid;
        private String name;
        private Integer pgrpId;
        private Integer ppid;
        private Integer sessionId;
        private Long vmem;
        private Long rssmemPage;
        private Long utime = 0L;
        private Long stime = 0L;
        private int age;
        private Long dtime = 0L;
        private List<ProcessInfo> children = new ArrayList<ProcessInfo>();

        public ProcessInfo(int pid) {
            this.pid = pid;
            this.age = 1;
        }

        public Integer getPid() {
            return this.pid;
        }

        public String getName() {
            return this.name;
        }

        public Integer getPgrpId() {
            return this.pgrpId;
        }

        public Integer getPpid() {
            return this.ppid;
        }

        public Integer getSessionId() {
            return this.sessionId;
        }

        public Long getVmem() {
            return this.vmem;
        }

        public Long getUtime() {
            return this.utime;
        }

        public Long getStime() {
            return this.stime;
        }

        public Long getDtime() {
            return this.dtime;
        }

        public Long getRssmemPage() {
            return this.rssmemPage;
        }

        public int getAge() {
            return this.age;
        }

        public boolean isParent(ProcessInfo p) {
            return this.pid.equals(p.getPpid());
        }

        public void updateProcessInfo(String name, Integer ppid, Integer pgrpId, Integer sessionId, Long utime, Long stime, Long vmem, Long rssmem) {
            this.name = name;
            this.ppid = ppid;
            this.pgrpId = pgrpId;
            this.sessionId = sessionId;
            this.utime = utime;
            this.stime = stime;
            this.vmem = vmem;
            this.rssmemPage = rssmem;
        }

        public void updateJiffy(ProcessInfo oldInfo) {
            this.dtime = oldInfo == null ? this.utime + this.stime : this.utime + this.stime - (oldInfo.utime + oldInfo.stime);
        }

        public void updateAge(ProcessInfo oldInfo) {
            this.age = oldInfo.age + 1;
        }

        public boolean addChild(ProcessInfo p) {
            return this.children.add(p);
        }

        public List<ProcessInfo> getChildren() {
            return this.children;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public String getCmdLine(String procfsDir) {
            String ret = "N/A";
            if (this.pid == null) {
                return ret;
            }
            BufferedReader in = null;
            FileReader fReader = null;
            try {
                fReader = new FileReader(new File(new File(procfsDir, this.pid.toString()), ProcfsBasedProcessTree.PROCFS_CMDLINE_FILE));
            }
            catch (FileNotFoundException f) {
                return ret;
            }
            in = new BufferedReader(fReader);
            try {
                ret = in.readLine();
                if (ret == null) {
                    ret = "N/A";
                } else if ((ret = ret.replace('\u0000', ' ')).equals("")) {
                    ret = "N/A";
                }
            }
            catch (IOException io) {
                LOG.warn((Object)("Error reading the stream " + io));
                ret = "N/A";
            }
            finally {
                try {
                    fReader.close();
                    try {
                        in.close();
                    }
                    catch (IOException i) {
                        LOG.warn((Object)("Error closing the stream " + in));
                    }
                }
                catch (IOException i) {
                    LOG.warn((Object)("Error closing the stream " + fReader));
                }
            }
            return ret;
        }
    }
}

