/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.mahout.fpm.pfpgrowth;

import java.io.IOException;

import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.mahout.common.Parameters;
import org.apache.mahout.fpm.pfpgrowth.convertors.string.TopKStringPatterns;

/**
 * 
 *  groups all Frequent Patterns containing an item and outputs the top K patterns
 * containing that particular item
 * 
 */
public class AggregatorReducer extends Reducer<Text,TopKStringPatterns,Text,TopKStringPatterns> {
  
  private int maxHeapSize = 50;
  
  @Override
  protected void reduce(Text key, Iterable<TopKStringPatterns> values, Context context) throws IOException,
                                                                                       InterruptedException {
    TopKStringPatterns patterns = new TopKStringPatterns();
    for (TopKStringPatterns value : values) {
      context.setStatus("Aggregator Reducer: Selecting TopK patterns for: " + key);
      patterns = patterns.merge(value, maxHeapSize);
    }
    context.write(key, patterns);
    
  }
  
  @Override
  protected void setup(Context context) throws IOException, InterruptedException {
    super.setup(context);
    Parameters params = new Parameters(context.getConfiguration().get("pfp.parameters", ""));
    maxHeapSize = Integer.valueOf(params.get("maxHeapSize", "50"));
    
  }
}
