/*
 * Copyright 2002-2021 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.pulsar.shade.org.apache.avro.util.springframework;

import org.apache.avro.reflect.Nullable;
import org.apache.pulsar.shade.org.apache.avro.util.ClassUtils;

import java.util.Arrays;

/**
 * Miscellaneous object utility methods.
 *
 * <p>
 * Mainly for internal use within the framework.
 *
 * <p>
 * Thanks to Alex Ruiz for contributing several enhancements to this class!
 *
 * @author Juergen Hoeller
 * @author Keith Donald
 * @author Rod Johnson
 * @author Rob Harrop
 * @author Chris Beams
 * @author Sam Brannen
 * @see ClassUtils see CollectionUtils see StringUtils
 * @since 19.03.2004
 */
class ObjectUtils {
  private ObjectUtils() {
  }

  private static final int INITIAL_HASH = 7;
  private static final int MULTIPLIER = 31;

  /**
   * Determine whether the given array is empty: i.e. {@code null} or of zero
   * length.
   *
   * @param array the array to check
   */
  public static boolean isEmpty(@Nullable Object[] array) {
    return (array == null || array.length == 0);
  }

  // ---------------------------------------------------------------------
  // Convenience methods for content-based equality/hash-code handling
  // ---------------------------------------------------------------------

  /**
   * Determine if the given objects are equal, returning {@code true} if both are
   * {@code null} or {@code false} if only one is {@code null}.
   * <p>
   * Compares arrays with {@code Arrays.equals}, performing an equality check
   * based on the array elements rather than the array reference.
   *
   * @param o1 first Object to compare
   * @param o2 second Object to compare
   * @return whether the given objects are equal
   * @see Object#equals(Object)
   * @see Arrays#equals
   */
  public static boolean nullSafeEquals(@Nullable Object o1, @Nullable Object o2) {
    if (o1 == o2) {
      return true;
    }
    if (o1 == null || o2 == null) {
      return false;
    }
    if (o1.equals(o2)) {
      return true;
    }
    if (o1.getClass().isArray() && o2.getClass().isArray()) {
      return arrayEquals(o1, o2);
    }
    return false;
  }

  /**
   * Compare the given arrays with {@code Arrays.equals}, performing an equality
   * check based on the array elements rather than the array reference.
   *
   * @param o1 first array to compare
   * @param o2 second array to compare
   * @return whether the given objects are equal
   * @see #nullSafeEquals(Object, Object)
   * @see Arrays#equals
   */
  private static boolean arrayEquals(Object o1, Object o2) {
    if (o1 instanceof Object[] && o2 instanceof Object[]) {
      return Arrays.equals((Object[]) o1, (Object[]) o2);
    }
    if (o1 instanceof boolean[] && o2 instanceof boolean[]) {
      return Arrays.equals((boolean[]) o1, (boolean[]) o2);
    }
    if (o1 instanceof byte[] && o2 instanceof byte[]) {
      return Arrays.equals((byte[]) o1, (byte[]) o2);
    }
    if (o1 instanceof char[] && o2 instanceof char[]) {
      return Arrays.equals((char[]) o1, (char[]) o2);
    }
    if (o1 instanceof double[] && o2 instanceof double[]) {
      return Arrays.equals((double[]) o1, (double[]) o2);
    }
    if (o1 instanceof float[] && o2 instanceof float[]) {
      return Arrays.equals((float[]) o1, (float[]) o2);
    }
    if (o1 instanceof int[] && o2 instanceof int[]) {
      return Arrays.equals((int[]) o1, (int[]) o2);
    }
    if (o1 instanceof long[] && o2 instanceof long[]) {
      return Arrays.equals((long[]) o1, (long[]) o2);
    }
    if (o1 instanceof short[] && o2 instanceof short[]) {
      return Arrays.equals((short[]) o1, (short[]) o2);
    }
    return false;
  }

  /**
   * Return as hash code for the given object; typically the value of
   * {@code Object#hashCode()}}. If the object is an array, this method will
   * delegate to any of the {@code nullSafeHashCode} methods for arrays in this
   * class. If the object is {@code null}, this method returns 0.
   *
   * @see Object#hashCode()
   * @see #nullSafeHashCode(Object[])
   * @see #nullSafeHashCode(boolean[])
   * @see #nullSafeHashCode(byte[])
   * @see #nullSafeHashCode(char[])
   * @see #nullSafeHashCode(double[])
   * @see #nullSafeHashCode(float[])
   * @see #nullSafeHashCode(int[])
   * @see #nullSafeHashCode(long[])
   * @see #nullSafeHashCode(short[])
   */
  public static int nullSafeHashCode(@Nullable Object obj) {
    if (obj == null) {
      return 0;
    }
    if (obj.getClass().isArray()) {
      if (obj instanceof Object[]) {
        return nullSafeHashCode((Object[]) obj);
      }
      if (obj instanceof boolean[]) {
        return nullSafeHashCode((boolean[]) obj);
      }
      if (obj instanceof byte[]) {
        return nullSafeHashCode((byte[]) obj);
      }
      if (obj instanceof char[]) {
        return nullSafeHashCode((char[]) obj);
      }
      if (obj instanceof double[]) {
        return nullSafeHashCode((double[]) obj);
      }
      if (obj instanceof float[]) {
        return nullSafeHashCode((float[]) obj);
      }
      if (obj instanceof int[]) {
        return nullSafeHashCode((int[]) obj);
      }
      if (obj instanceof long[]) {
        return nullSafeHashCode((long[]) obj);
      }
      if (obj instanceof short[]) {
        return nullSafeHashCode((short[]) obj);
      }
    }
    return obj.hashCode();
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable Object[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (Object element : array) {
      hash = MULTIPLIER * hash + nullSafeHashCode(element);
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable boolean[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (boolean element : array) {
      hash = MULTIPLIER * hash + Boolean.hashCode(element);
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable byte[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (byte element : array) {
      hash = MULTIPLIER * hash + element;
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable char[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (char element : array) {
      hash = MULTIPLIER * hash + element;
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable double[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (double element : array) {
      hash = MULTIPLIER * hash + Double.hashCode(element);
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable float[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (float element : array) {
      hash = MULTIPLIER * hash + Float.hashCode(element);
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable int[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (int element : array) {
      hash = MULTIPLIER * hash + element;
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable long[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (long element : array) {
      hash = MULTIPLIER * hash + Long.hashCode(element);
    }
    return hash;
  }

  /**
   * Return a hash code based on the contents of the specified array. If
   * {@code array} is {@code null}, this method returns 0.
   */
  public static int nullSafeHashCode(@Nullable short[] array) {
    if (array == null) {
      return 0;
    }
    int hash = INITIAL_HASH;
    for (short element : array) {
      hash = MULTIPLIER * hash + element;
    }
    return hash;
  }
}
