/*
 * Decompiled with CFR 0.152.
 */
package org.apache.qpid.server.security;

import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import org.apache.qpid.server.logging.EventLogger;
import org.apache.qpid.server.logging.messages.KeyStoreMessages;
import org.apache.qpid.server.model.AbstractConfigurationChangeListener;
import org.apache.qpid.server.model.AbstractConfiguredObject;
import org.apache.qpid.server.model.Broker;
import org.apache.qpid.server.model.ConfiguredObject;
import org.apache.qpid.server.model.IntegrityViolationException;
import org.apache.qpid.server.model.KeyStore;
import org.apache.qpid.server.model.Port;
import org.apache.qpid.server.model.State;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractKeyStore<X extends AbstractKeyStore<X>>
extends AbstractConfiguredObject<X>
implements KeyStore<X> {
    private static Logger LOGGER = LoggerFactory.getLogger(AbstractKeyStore.class);
    protected static final long ONE_DAY = 86400000L;
    private final Broker<?> _broker;
    private final EventLogger _eventLogger;
    private ScheduledFuture<?> _checkExpiryTaskFuture;

    public AbstractKeyStore(Map<String, Object> attributes, Broker<?> broker) {
        super(broker, attributes);
        this._broker = broker;
        this._eventLogger = broker.getEventLogger();
        this._eventLogger.message(KeyStoreMessages.CREATE(this.getName()));
    }

    public final Broker<?> getBroker() {
        return this._broker;
    }

    final EventLogger getEventLogger() {
        return this._eventLogger;
    }

    @Override
    protected ListenableFuture<Void> onClose() {
        if (this._checkExpiryTaskFuture != null) {
            this._checkExpiryTaskFuture.cancel(false);
            this._checkExpiryTaskFuture = null;
        }
        return Futures.immediateFuture(null);
    }

    @Override
    protected void logOperation(String operation) {
        this._broker.getEventLogger().message(KeyStoreMessages.OPERATION(operation));
    }

    protected void initializeExpiryChecking() {
        int checkFrequency = this.getCertificateExpiryCheckFrequency();
        if (this.getBroker().getState() == State.ACTIVE) {
            this._checkExpiryTaskFuture = this.getBroker().scheduleHouseKeepingTask(checkFrequency, TimeUnit.DAYS, new Runnable(){

                @Override
                public void run() {
                    AbstractKeyStore.this.checkCertificateExpiry();
                }
            });
        } else {
            final int frequency = checkFrequency;
            this.getBroker().addChangeListener(new AbstractConfigurationChangeListener(){

                @Override
                public void stateChanged(ConfiguredObject<?> object, State oldState, State newState) {
                    if (newState == State.ACTIVE) {
                        AbstractKeyStore.this._checkExpiryTaskFuture = AbstractKeyStore.this.getBroker().scheduleHouseKeepingTask(frequency, TimeUnit.DAYS, () -> AbstractKeyStore.this.checkCertificateExpiry());
                        AbstractKeyStore.this.getBroker().removeChangeListener(this);
                    }
                }
            });
        }
    }

    protected final ListenableFuture<Void> deleteIfNotInUse() {
        String storeName = this.getName();
        ArrayList ports = new ArrayList(this.getBroker().getPorts());
        for (Port port : ports) {
            if (port.getKeyStore() != this) continue;
            throw new IntegrityViolationException("Key store '" + storeName + "' can't be deleted as it is in use by a port:" + port.getName());
        }
        this.deleted();
        this.setState(State.DELETED);
        this.getEventLogger().message(KeyStoreMessages.DELETE(this.getName()));
        return Futures.immediateFuture(null);
    }

    protected abstract void checkCertificateExpiry();

    protected void checkCertificatesExpiry(long currentTime, Date expiryTestDate, X509Certificate[] chain) {
        if (chain != null) {
            for (X509Certificate cert : chain) {
                try {
                    cert.checkValidity(expiryTestDate);
                }
                catch (CertificateExpiredException e) {
                    long timeToExpiry = cert.getNotAfter().getTime() - currentTime;
                    int days = Math.max(0, (int)(timeToExpiry / 86400000L));
                    this.getEventLogger().message(KeyStoreMessages.EXPIRING(this.getName(), String.valueOf(days), cert.getSubjectDN().toString()));
                }
                catch (CertificateNotYetValidException certificateNotYetValidException) {
                    // empty catch block
                }
            }
        }
    }

    @Override
    public final int getCertificateExpiryWarnPeriod() {
        try {
            return this.getContextValue(Integer.class, "qpid.keystore.certificateExpiryWarnPeriod");
        }
        catch (IllegalArgumentException | NullPointerException e) {
            LOGGER.warn("The value of the context variable '{}' for keystore {} cannot be converted to an integer. The value {} will be used as a default", new Object[]{"qpid.keystore.certificateExpiryWarnPeriod", this.getName(), 30});
            return 30;
        }
    }

    @Override
    public int getCertificateExpiryCheckFrequency() {
        int checkFrequency;
        try {
            checkFrequency = this.getContextValue(Integer.class, "qpid.keystore.certificateExpiryCheckFrequency");
        }
        catch (IllegalArgumentException | NullPointerException e) {
            LOGGER.warn("Cannot parse the context variable {} ", (Object)"qpid.keystore.certificateExpiryCheckFrequency", (Object)e);
            checkFrequency = 1;
        }
        return checkFrequency;
    }
}

