/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 */
package org.apache.shindig.gadgets.rewrite;


import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.config.ContainerConfig;
import org.apache.shindig.gadgets.GadgetContext;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.GadgetSpecFactory;
import org.apache.shindig.gadgets.http.HttpRequest;
import org.apache.shindig.gadgets.spec.GadgetSpec;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Factory for content rewriter features
 */
@Singleton
public class ContentRewriterFeatureFactory {

  private final GadgetSpecFactory specFactory;
  private Map<String, ContentRewriterFeature> contentRewriters;
  
  static final String CONTENT_REWRITE_KEY = "gadgets.content-rewrite";
  static final String INCLUDE_TAGS_KEY = "include-tags";
  static final String INCLUDE_URLS_KEY = "include-urls";
  static final String EXCLUDE_TAGS_KEY = "exclude-urls";
  static final String EXPIRES_KEY = "expires";
  
  private final ContainerConfig config;

  @Inject
  public ContentRewriterFeatureFactory(GadgetSpecFactory specFactory, ContainerConfig config) { 
    this.specFactory = specFactory;
    this.config = config;
    contentRewriters = new HashMap<String, ContentRewriterFeature>();
  }

  public ContentRewriterFeature getDefault(String container) {
    if (!contentRewriters.containsKey(container)) {
      contentRewriters.put(container, createContentRewriterFeature(null, container));
    }
    return contentRewriters.get(container);
   }

  public ContentRewriterFeature get(HttpRequest request) {
    final Uri gadgetUri = request.getGadget();
    GadgetSpec spec;
    if (gadgetUri != null) {
      try {
        GadgetContext context = new GadgetContext() {
          @Override
          public Uri getUrl() {
            return gadgetUri;
          }
        };

        spec = specFactory.getGadgetSpec(context);
        if (spec != null) {
          return get(spec, request.getContainer());
        }
      } catch (GadgetException ge) {
      }
    }
    return getDefault(request.getContainer());
  }
  
  @SuppressWarnings("unchecked")
  private ContentRewriterFeature createContentRewriterFeature(GadgetSpec spec, String container) {
      Object object = config.getProperty(container, CONTENT_REWRITE_KEY);
      Set<String> tags = new HashSet<String>();
      if(object instanceof JSONObject) {
        try {
          JSONObject contentRewrite = (JSONObject)object;
          JSONArray jsonTags = contentRewrite.getJSONArray(INCLUDE_TAGS_KEY);
          for (int i = 0, j = jsonTags.length(); i < j; ++i) {
            tags.add(jsonTags.getString(i).toLowerCase());
          }
        
          return new ContentRewriterFeature(spec,
                                          contentRewrite.getString(INCLUDE_URLS_KEY),
                                          contentRewrite.getString(EXCLUDE_TAGS_KEY),
                                          contentRewrite.getString(EXPIRES_KEY),
                                          tags);
        }catch(JSONException e){
          return null;
        }
      } else {
        ImmutableMap contentRewrite = (ImmutableMap) object;
        ImmutableList<String> tagsList = (ImmutableList<String>) contentRewrite.get(INCLUDE_TAGS_KEY);
        for (Object tag : tagsList) {
          tags.add(tag.toString().toLowerCase());
        }
        return new ContentRewriterFeature(spec,
                                          (String)contentRewrite.get(INCLUDE_URLS_KEY),
                                          (String)contentRewrite.get(EXCLUDE_TAGS_KEY),
                                          (String)contentRewrite.get(EXPIRES_KEY),
                                          tags);
      }
  }

  public ContentRewriterFeature get(GadgetSpec spec, String container) {
    ContentRewriterFeature rewriterFeature =
        (ContentRewriterFeature)spec.getAttribute("content-rewriter");
    if (rewriterFeature != null) return rewriterFeature;
    rewriterFeature = createContentRewriterFeature(spec,  container);
    spec.setAttribute("content-rewriter", rewriterFeature);
    return rewriterFeature;
  }

  /**
   * Create a rewriter feature that allows all URIs to be rewritten.
   */
  public ContentRewriterFeature createRewriteAllFeature(int ttl) {
    return new ContentRewriterFeature(null,
        ".*", "", (ttl == -1) ? "HTTP" : Integer.toString(ttl),
        Collections.<String>emptySet());
  }
}
