/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shindig.social.sample.spi;

import org.apache.shindig.auth.SecurityToken;
import org.apache.shindig.common.testing.FakeGadgetToken;
import org.apache.shindig.protocol.DataCollection;
import org.apache.shindig.protocol.ProtocolException;
import org.apache.shindig.protocol.RestfulCollection;
import org.apache.shindig.protocol.model.FilterOperation;
import org.apache.shindig.protocol.model.SortOrder;
import org.apache.shindig.social.SocialApiTestsGuiceModule;
import org.apache.shindig.social.opensocial.model.Activity;
import org.apache.shindig.social.opensocial.model.Person;
import org.apache.shindig.social.opensocial.spi.CollectionOptions;
import org.apache.shindig.social.opensocial.spi.GroupId;
import org.apache.shindig.social.opensocial.spi.PersonService;
import org.apache.shindig.social.opensocial.spi.UserId;

import java.util.Collections;

import javax.servlet.http.HttpServletResponse;

import junit.framework.TestCase;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.inject.Guice;
import com.google.inject.Injector;

/**
 * Test the JSONOpensocialService
 */
public class JsonDbOpensocialServiceTest extends TestCase {
  private JsonDbOpensocialService db;

  private static final UserId CANON_USER = new UserId(UserId.Type.userId, "canonical");
  private static final UserId JOHN_DOE = new UserId(UserId.Type.userId, "john.doe");
  private static final UserId JANE_DOE = new UserId(UserId.Type.userId, "jane.doe");

  private static final GroupId SELF_GROUP = new GroupId(GroupId.Type.self, null);
  private static final String APP_ID = "1";
  private static final String CANONICAL_USER_ID = "canonical";

  private SecurityToken token = new FakeGadgetToken();

  @Override
  protected void setUp() throws Exception {
    super.setUp();
    Injector injector = Guice.createInjector(new SocialApiTestsGuiceModule());
    db = injector.getInstance(JsonDbOpensocialService.class);
  }

  public void testGetPersonDefaultFields() throws Exception {
    Person person = db
        .getPerson(CANON_USER, Person.Field.DEFAULT_FIELDS, token).get();

    assertNotNull("Canonical user not found", person);
    assertNotNull("Canonical user has no id", person.getId());
    assertNotNull("Canonical user has no name", person.getName());
    assertNotNull("Canonical user has no thumbnail",
        person.getThumbnailUrl());
  }

  public void testGetPersonAllFields() throws Exception {
    Person person = db
        .getPerson(CANON_USER, Person.Field.ALL_FIELDS, token).get();
    assertNotNull("Canonical user not found", person);
  }

  public void testGetPersonAllAppData() throws Exception {
    Person person = db
        .getPerson(CANON_USER, ImmutableSet.of("id", "appData"), token).get();

    assertNotNull("Canonical user not found", person);
    assertEquals("Canonical user has wrong id", "canonical", person.getId());
    assertEquals("Canonical user has wrong app data",
        ImmutableMap.of("count", "2", "size", "100"), person.getAppData());
  }

  public void testGetPersonOneAppDataField() throws Exception {
    Person person = db
        .getPerson(CANON_USER, ImmutableSet.of("id", "appData.size"), token).get();

    assertNotNull("Canonical user not found", person);
    assertEquals("Canonical user has wrong id", "canonical", person.getId());
    assertEquals("Canonical user has wrong app data",
        ImmutableMap.of("size", "100"), person.getAppData());
  }

  public void testGetPersonMultipleAppDataFields() throws Exception {
    Person person = db
        .getPerson(CANON_USER,
            ImmutableSet.of("id", "appData.size", "appData.count", "appData.bogus"),
            token).get();

    assertNotNull("Canonical user not found", person);
    assertEquals("Canonical user has wrong id", "canonical", person.getId());
    assertEquals("Canonical user has wrong app data",
        ImmutableMap.of("count", "2", "size", "100"), person.getAppData());
  }

  public void testGetExpectedFriends() throws Exception {
    CollectionOptions options = new CollectionOptions();
    options.setSortBy(PersonService.TOP_FRIENDS_SORT);
    options.setSortOrder(SortOrder.ascending);
    options.setFilter(null);
    options.setFilterOperation(FilterOperation.contains);
    options.setFilterValue("");
    options.setFirst(0);
    options.setMax(20);

    RestfulCollection<Person> responseItem = db.getPeople(
        ImmutableSet.of(CANON_USER), new GroupId(GroupId.Type.friends, null),
        options, Collections.<String>emptySet(), token).get();
    assertNotNull(responseItem);
    assertEquals(4, responseItem.getTotalResults());
    // Test a couple of users
    assertEquals("john.doe", responseItem.getEntry().get(0).getId());
    assertEquals("jane.doe", responseItem.getEntry().get(1).getId());
  }

  public void testGetExpectedUsersForPlural() throws Exception {
    CollectionOptions options = new CollectionOptions();
    options.setSortBy(PersonService.TOP_FRIENDS_SORT);
    options.setSortOrder(SortOrder.ascending);
    options.setFilter(null);
    options.setFilterOperation(FilterOperation.contains);
    options.setFilterValue("");
    options.setFirst(0);
    options.setMax(20);

    RestfulCollection<Person> responseItem = db.getPeople(
        ImmutableSet.of(JOHN_DOE, JANE_DOE), new GroupId(GroupId.Type.friends, null),
       options, Collections.<String>emptySet(), token).get();
    assertNotNull(responseItem);
    assertEquals(4, responseItem.getTotalResults());
    // Test a couple of users
    assertEquals("john.doe", responseItem.getEntry().get(0).getId());
    assertEquals("jane.doe", responseItem.getEntry().get(1).getId());
  }

  public void testGetExpectedActivities() throws Exception {
    RestfulCollection<Activity> responseItem = db.getActivities(
        ImmutableSet.of(CANON_USER), SELF_GROUP, APP_ID, Collections.<String>emptySet(), null,
        new FakeGadgetToken()).get();
    assertSame(2, responseItem.getTotalResults());
  }

  public void testGetExpectedActivitiesForPlural() throws Exception {
    RestfulCollection<Activity> responseItem = db.getActivities(
        ImmutableSet.of(CANON_USER, JOHN_DOE), SELF_GROUP, APP_ID, Collections.<String>emptySet(), null,
        new FakeGadgetToken()).get();
    assertSame(3, responseItem.getTotalResults());
  }

  public void testGetExpectedActivity() throws Exception {
    Activity activity = db.getActivity(
        CANON_USER, SELF_GROUP, APP_ID,
        ImmutableSet.of("appId", "body", "mediaItems"), APP_ID, new FakeGadgetToken()).get();
    assertNotNull(activity);
    // Check that some fields are fetched and others are not
    assertNotNull(activity.getBody());
    assertNull(activity.getBodyId());
  }

  public void testDeleteExpectedActivity() throws Exception {
    db.deleteActivities(CANON_USER, SELF_GROUP, APP_ID, ImmutableSet.of(APP_ID),
        new FakeGadgetToken());

    // Try to fetch the activity
    try {
      db.getActivity(
          CANON_USER, SELF_GROUP, APP_ID,
          ImmutableSet.of("appId", "body", "mediaItems"), APP_ID, new FakeGadgetToken()).get();
      fail();
    } catch (ProtocolException sse) {
      assertEquals(HttpServletResponse.SC_BAD_REQUEST, sse.getCode());
    }
  }

  public void testGetExpectedAppData() throws Exception {
    DataCollection responseItem = db.getPersonData(
        ImmutableSet.of(CANON_USER), SELF_GROUP, APP_ID, Collections.<String>emptySet(),
        new FakeGadgetToken()).get();
    assertFalse(responseItem.getEntry().isEmpty());
    assertFalse(responseItem.getEntry().get(CANONICAL_USER_ID).isEmpty());

    assertSame(2, responseItem.getEntry().get(CANONICAL_USER_ID).size());
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("count"));
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("size"));
  }

  public void testGetExpectedAppDataForPlural() throws Exception {
    DataCollection responseItem = db.getPersonData(
        ImmutableSet.of(CANON_USER, JOHN_DOE), SELF_GROUP, APP_ID, Collections.<String>emptySet(),
        new FakeGadgetToken()).get();
    assertFalse(responseItem.getEntry().isEmpty());
    assertFalse(responseItem.getEntry().get(CANONICAL_USER_ID).isEmpty());

    assertSame(2, responseItem.getEntry().get(CANONICAL_USER_ID).size());
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("count"));
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("size"));

    assertFalse(responseItem.getEntry().get(JOHN_DOE.getUserId()).isEmpty());
    assertSame(1, responseItem.getEntry().get(JOHN_DOE.getUserId()).size());
    assertTrue(responseItem.getEntry().get(JOHN_DOE.getUserId()).containsKey("count"));
  }

  public void testDeleteExpectedAppData() throws Exception {
    // Delete the data
    db.deletePersonData(CANON_USER, SELF_GROUP, APP_ID,
        ImmutableSet.of("count"), new FakeGadgetToken());

    // Fetch the remaining and test
    DataCollection responseItem = db.getPersonData(
        ImmutableSet.of(CANON_USER), SELF_GROUP, APP_ID, Collections.<String>emptySet(),
        new FakeGadgetToken()).get();
    assertFalse(responseItem.getEntry().isEmpty());
    assertFalse(responseItem.getEntry().get(CANONICAL_USER_ID).isEmpty());

    assertSame(1, responseItem.getEntry().get(CANONICAL_USER_ID).size());
    assertFalse(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("count"));
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("size"));
  }

  public void testUpdateExpectedAppData() throws Exception {
    // Delete the data
    db.updatePersonData(CANON_USER, SELF_GROUP, APP_ID,
        null, ImmutableMap.of("count", "10", "newvalue", "20"), new FakeGadgetToken());

    // Fetch the remaining and test
    DataCollection responseItem = db.getPersonData(
        ImmutableSet.of(CANON_USER), SELF_GROUP, APP_ID, Collections.<String>emptySet(),
        new FakeGadgetToken()).get();

    assertFalse(responseItem.getEntry().isEmpty());
    assertFalse(responseItem.getEntry().get(CANONICAL_USER_ID).isEmpty());

    assertSame(3, responseItem.getEntry().get(CANONICAL_USER_ID).size());
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("count"));
    assertEquals("10", responseItem.getEntry().get(CANONICAL_USER_ID).get("count"));
    assertTrue(responseItem.getEntry().get(CANONICAL_USER_ID).containsKey("newvalue"));
    assertEquals("20", responseItem.getEntry().get(CANONICAL_USER_ID).get("newvalue"));
  }
}
