/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.crypto.sodium;

import com.google.common.base.Preconditions;
import java.util.Arrays;
import java.util.Objects;
import javax.annotation.Nullable;
import javax.security.auth.Destroyable;
import jnr.ffi.Pointer;
import jnr.ffi.byref.LongLongByReference;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.crypto.sodium.Allocated;
import org.apache.tuweni.crypto.sodium.DefaultDetachedEncryptionResult;
import org.apache.tuweni.crypto.sodium.DetachedEncryptionResult;
import org.apache.tuweni.crypto.sodium.Sodium;
import org.apache.tuweni.crypto.sodium.SodiumException;

public final class AES256GCM
implements AutoCloseable {
    private static final byte[] EMPTY_BYTES = new byte[0];
    private Pointer ctx;

    public static boolean isAvailable() {
        try {
            return Sodium.crypto_aead_aes256gcm_is_available() != 0;
        }
        catch (LinkageError e) {
            return false;
        }
    }

    private static void assertAvailable() {
        if (!AES256GCM.isAvailable()) {
            throw new UnsupportedOperationException("Sodium AES256-GCM is not available");
        }
    }

    private AES256GCM(Key key) {
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        this.ctx = Sodium.malloc(Sodium.crypto_aead_aes256gcm_statebytes());
        try {
            int rc = Sodium.crypto_aead_aes256gcm_beforenm(this.ctx, key.value.pointer());
            if (rc != 0) {
                throw new SodiumException("crypto_aead_aes256gcm_beforenm: failed with result " + rc);
            }
        }
        catch (Throwable e) {
            Sodium.sodium_free(this.ctx);
            this.ctx = null;
            throw e;
        }
    }

    public static AES256GCM forKey(Key key) {
        Objects.requireNonNull(key);
        AES256GCM.assertAvailable();
        return new AES256GCM(key);
    }

    public static Bytes encrypt(Bytes message, Key key, Nonce nonce) {
        return Bytes.wrap((byte[])AES256GCM.encrypt(message.toArrayUnsafe(), key, nonce));
    }

    public static byte[] encrypt(byte[] message, Key key, Nonce nonce) {
        return AES256GCM.encrypt(message, EMPTY_BYTES, key, nonce);
    }

    public static Bytes encrypt(Bytes message, Bytes data, Key key, Nonce nonce) {
        return Bytes.wrap((byte[])AES256GCM.encrypt(message.toArrayUnsafe(), data.toArrayUnsafe(), key, nonce));
    }

    public static byte[] encrypt(byte[] message, byte[] data, Key key, Nonce nonce) {
        AES256GCM.assertAvailable();
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        byte[] cipherText = new byte[AES256GCM.maxCombinedCypherTextLength(message)];
        LongLongByReference cipherTextLen = new LongLongByReference();
        int rc = Sodium.crypto_aead_aes256gcm_encrypt(cipherText, cipherTextLen, message, message.length, data, data.length, null, nonce.value.pointer(), key.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_encrypt: failed with result " + rc);
        }
        return AES256GCM.maybeSliceResult(cipherText, cipherTextLen, "crypto_aead_aes256gcm_encrypt");
    }

    public Bytes encrypt(Bytes message, Nonce nonce) {
        return Bytes.wrap((byte[])this.encrypt(message.toArrayUnsafe(), nonce));
    }

    public byte[] encrypt(byte[] message, Nonce nonce) {
        return this.encrypt(message, EMPTY_BYTES, nonce);
    }

    public Bytes encrypt(Bytes message, Bytes data, Nonce nonce) {
        return Bytes.wrap((byte[])this.encrypt(message.toArrayUnsafe(), data.toArrayUnsafe(), nonce));
    }

    public byte[] encrypt(byte[] message, byte[] data, Nonce nonce) {
        this.assertOpen();
        byte[] cipherText = new byte[AES256GCM.maxCombinedCypherTextLength(message)];
        LongLongByReference cipherTextLen = new LongLongByReference();
        int rc = Sodium.crypto_aead_aes256gcm_encrypt_afternm(cipherText, cipherTextLen, message, message.length, data, data.length, null, nonce.value.pointer(), this.ctx);
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_encrypt_afternm: failed with result " + rc);
        }
        return AES256GCM.maybeSliceResult(cipherText, cipherTextLen, "crypto_aead_aes256gcm_encrypt_afternm");
    }

    private static int maxCombinedCypherTextLength(byte[] message) {
        long abytes = Sodium.crypto_aead_aes256gcm_abytes();
        if (abytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_aead_aes256gcm_abytes: " + abytes + " is too large");
        }
        return (int)abytes + message.length;
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, Key key, Nonce nonce) {
        return AES256GCM.encryptDetached(message.toArrayUnsafe(), key, nonce);
    }

    public static DetachedEncryptionResult encryptDetached(byte[] message, Key key, Nonce nonce) {
        return AES256GCM.encryptDetached(message, EMPTY_BYTES, key, nonce);
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, Bytes data, Key key, Nonce nonce) {
        return AES256GCM.encryptDetached(message.toArrayUnsafe(), data.toArrayUnsafe(), key, nonce);
    }

    public static DetachedEncryptionResult encryptDetached(byte[] message, byte[] data, Key key, Nonce nonce) {
        AES256GCM.assertAvailable();
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        byte[] cipherText = new byte[message.length];
        long abytes = Sodium.crypto_aead_aes256gcm_abytes();
        if (abytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_aead_aes256gcm_abytes: " + abytes + " is too large");
        }
        byte[] mac = new byte[(int)abytes];
        LongLongByReference macLen = new LongLongByReference();
        int rc = Sodium.crypto_aead_aes256gcm_encrypt_detached(cipherText, mac, macLen, message, message.length, data, data.length, null, nonce.value.pointer(), key.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_encrypt_detached: failed with result " + rc);
        }
        return new DefaultDetachedEncryptionResult(cipherText, AES256GCM.maybeSliceResult(mac, macLen, "crypto_aead_aes256gcm_encrypt_detached"));
    }

    public DetachedEncryptionResult encryptDetached(Bytes message, Nonce nonce) {
        return this.encryptDetached(message.toArrayUnsafe(), nonce);
    }

    public DetachedEncryptionResult encryptDetached(byte[] message, Nonce nonce) {
        return this.encryptDetached(message, EMPTY_BYTES, nonce);
    }

    public DetachedEncryptionResult encryptDetached(Bytes message, Bytes data, Nonce nonce) {
        return this.encryptDetached(message.toArrayUnsafe(), data.toArrayUnsafe(), nonce);
    }

    public DetachedEncryptionResult encryptDetached(byte[] message, byte[] data, Nonce nonce) {
        this.assertOpen();
        byte[] cipherText = new byte[message.length];
        long abytes = Sodium.crypto_aead_aes256gcm_abytes();
        if (abytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_aead_aes256gcm_abytes: " + abytes + " is too large");
        }
        byte[] mac = new byte[(int)abytes];
        LongLongByReference macLen = new LongLongByReference();
        int rc = Sodium.crypto_aead_aes256gcm_encrypt_detached_afternm(cipherText, mac, macLen, message, message.length, data, data.length, null, nonce.value.pointer(), this.ctx);
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_encrypt_detached_afternm: failed with result " + rc);
        }
        return new DefaultDetachedEncryptionResult(cipherText, AES256GCM.maybeSliceResult(mac, macLen, "crypto_aead_aes256gcm_encrypt_detached_afternm"));
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, Key key, Nonce nonce) {
        byte[] bytes = AES256GCM.decrypt(cipherText.toArrayUnsafe(), key, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decrypt(byte[] cipherText, Key key, Nonce nonce) {
        return AES256GCM.decrypt(cipherText, EMPTY_BYTES, key, nonce);
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, Bytes data, Key key, Nonce nonce) {
        byte[] bytes = AES256GCM.decrypt(cipherText.toArrayUnsafe(), data.toArrayUnsafe(), key, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decrypt(byte[] cipherText, byte[] data, Key key, Nonce nonce) {
        AES256GCM.assertAvailable();
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        byte[] clearText = new byte[AES256GCM.maxClearTextLength(cipherText)];
        LongLongByReference clearTextLen = new LongLongByReference();
        int rc = Sodium.crypto_aead_aes256gcm_decrypt(clearText, clearTextLen, null, cipherText, cipherText.length, data, data.length, nonce.value.pointer(), key.value.pointer());
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_encrypt: failed with result " + rc);
        }
        return AES256GCM.maybeSliceResult(clearText, clearTextLen, "crypto_aead_aes256gcm_decrypt");
    }

    @Nullable
    public Bytes decrypt(Bytes cipherText, Nonce nonce) {
        byte[] bytes = this.decrypt(cipherText.toArrayUnsafe(), nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public byte[] decrypt(byte[] cipherText, Nonce nonce) {
        return this.decrypt(cipherText, EMPTY_BYTES, nonce);
    }

    @Nullable
    public Bytes decrypt(Bytes cipherText, Bytes data, Nonce nonce) {
        byte[] bytes = this.decrypt(cipherText.toArrayUnsafe(), data.toArrayUnsafe(), nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public byte[] decrypt(byte[] cipherText, byte[] data, Nonce nonce) {
        this.assertOpen();
        byte[] clearText = new byte[AES256GCM.maxClearTextLength(cipherText)];
        LongLongByReference clearTextLen = new LongLongByReference();
        int rc = Sodium.crypto_aead_aes256gcm_decrypt_afternm(clearText, clearTextLen, null, cipherText, cipherText.length, data, data.length, nonce.value.pointer(), this.ctx);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_decrypt_afternm: failed with result " + rc);
        }
        return AES256GCM.maybeSliceResult(clearText, clearTextLen, "crypto_aead_aes256gcm_decrypt_afternm");
    }

    private static int maxClearTextLength(byte[] cipherText) {
        long abytes = Sodium.crypto_aead_aes256gcm_abytes();
        if (abytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_aead_aes256gcm_abytes: " + abytes + " is too large");
        }
        if (abytes > (long)cipherText.length) {
            throw new IllegalArgumentException("cipherText is too short");
        }
        return cipherText.length - (int)abytes;
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, Key key, Nonce nonce) {
        byte[] bytes = AES256GCM.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), key, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, Key key, Nonce nonce) {
        return AES256GCM.decryptDetached(cipherText, mac, EMPTY_BYTES, key, nonce);
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, Bytes data, Key key, Nonce nonce) {
        byte[] bytes = AES256GCM.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), data.toArrayUnsafe(), key, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, byte[] data, Key key, Nonce nonce) {
        AES256GCM.assertAvailable();
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        long abytes = Sodium.crypto_aead_aes256gcm_abytes();
        if (abytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_aead_aes256gcm_abytes: " + abytes + " is too large");
        }
        if ((long)mac.length != abytes) {
            throw new IllegalArgumentException("mac must be " + abytes + " bytes, got " + mac.length);
        }
        byte[] clearText = new byte[cipherText.length];
        int rc = Sodium.crypto_aead_aes256gcm_decrypt_detached(clearText, null, cipherText, cipherText.length, mac, data, data.length, nonce.value.pointer(), key.value.pointer());
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_encrypt: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public Bytes decryptDetached(Bytes cipherText, Bytes mac, Nonce nonce) {
        byte[] bytes = this.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public byte[] decryptDetached(byte[] cipherText, byte[] mac, Nonce nonce) {
        return this.decryptDetached(cipherText, mac, EMPTY_BYTES, nonce);
    }

    @Nullable
    public Bytes decryptDetached(Bytes cipherText, Bytes mac, Bytes data, Nonce nonce) {
        byte[] bytes = this.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), data.toArrayUnsafe(), nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public byte[] decryptDetached(byte[] cipherText, byte[] mac, byte[] data, Nonce nonce) {
        AES256GCM.assertAvailable();
        long abytes = Sodium.crypto_aead_aes256gcm_abytes();
        if (abytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_aead_aes256gcm_abytes: " + abytes + " is too large");
        }
        if ((long)mac.length != abytes) {
            throw new IllegalArgumentException("mac must be " + abytes + " bytes, got " + mac.length);
        }
        byte[] clearText = new byte[cipherText.length];
        int rc = Sodium.crypto_aead_aes256gcm_decrypt_detached_afternm(clearText, null, cipherText, cipherText.length, mac, data, data.length, nonce.value.pointer(), this.ctx);
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_aead_aes256gcm_decrypt_detached_afternm: failed with result " + rc);
        }
        return clearText;
    }

    private void assertOpen() {
        if (this.ctx == null) {
            throw new IllegalStateException(this.getClass().getName() + ": already closed");
        }
    }

    private static byte[] maybeSliceResult(byte[] bytes, LongLongByReference actualLength, String methodName) {
        if (actualLength.longValue() == (long)bytes.length) {
            return bytes;
        }
        if (actualLength.longValue() > Integer.MAX_VALUE) {
            throw new SodiumException(methodName + ": result of length " + actualLength.longValue() + " is too large");
        }
        return Arrays.copyOfRange(bytes, 0, actualLength.intValue());
    }

    @Override
    public void close() {
        if (this.ctx != null) {
            Sodium.sodium_free(this.ctx);
            this.ctx = null;
        }
    }

    protected void finalize() {
        this.close();
    }

    public static final class Nonce {
        final Allocated value;

        private Nonce(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        public static Nonce fromBytes(Bytes bytes) {
            return Nonce.fromBytes(bytes.toArrayUnsafe());
        }

        public static Nonce fromBytes(byte[] bytes) {
            AES256GCM.assertAvailable();
            if ((long)bytes.length != Sodium.crypto_aead_aes256gcm_npubbytes()) {
                throw new IllegalArgumentException("nonce must be " + Sodium.crypto_aead_aes256gcm_npubbytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Nonce::new);
        }

        public static int length() {
            AES256GCM.assertAvailable();
            long npubbytes = Sodium.crypto_aead_aes256gcm_npubbytes();
            if (npubbytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_aead_aes256gcm_npubbytes: " + npubbytes + " is too large");
            }
            return (int)npubbytes;
        }

        public static Nonce zero() {
            int length = Nonce.length();
            Pointer ptr = Sodium.malloc(length);
            try {
                Sodium.sodium_memzero(ptr, length);
                return new Nonce(ptr, length);
            }
            catch (Throwable e) {
                Sodium.sodium_free(ptr);
                throw e;
            }
        }

        public static Nonce random() {
            return Sodium.randomBytes(Nonce.length(), Nonce::new);
        }

        public Nonce increment() {
            return Sodium.dupAndIncrement(this.value.pointer(), this.value.length(), Nonce::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Nonce)) {
                return false;
            }
            Nonce other = (Nonce)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }

    public static final class Key
    implements Destroyable {
        final Allocated value;

        private Key(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        @Override
        public void destroy() {
            this.value.destroy();
        }

        @Override
        public boolean isDestroyed() {
            return this.value.isDestroyed();
        }

        public static Key fromBytes(Bytes bytes) {
            return Key.fromBytes(bytes.toArrayUnsafe());
        }

        public static Key fromBytes(byte[] bytes) {
            AES256GCM.assertAvailable();
            if ((long)bytes.length != Sodium.crypto_aead_aes256gcm_keybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_aead_aes256gcm_keybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Key::new);
        }

        public static int length() {
            AES256GCM.assertAvailable();
            long keybytes = Sodium.crypto_aead_aes256gcm_keybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_aead_aes256gcm_keybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public static Key random() {
            AES256GCM.assertAvailable();
            int length = Key.length();
            Pointer ptr = Sodium.malloc(length);
            try {
                Sodium.randombytes_buf(ptr, length);
                return new Key(ptr, length);
            }
            catch (Throwable e) {
                Sodium.sodium_free(ptr);
                throw e;
            }
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Key)) {
                return false;
            }
            Key other = (Key)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }
}

