/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tuweni.crypto.sodium;

import com.google.common.base.Preconditions;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Objects;
import javax.annotation.Nullable;
import javax.security.auth.Destroyable;
import jnr.ffi.Pointer;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.crypto.sodium.Allocated;
import org.apache.tuweni.crypto.sodium.DefaultDetachedEncryptionResult;
import org.apache.tuweni.crypto.sodium.DetachedEncryptionResult;
import org.apache.tuweni.crypto.sodium.GenericHash;
import org.apache.tuweni.crypto.sodium.PasswordHash;
import org.apache.tuweni.crypto.sodium.SHA256Hash;
import org.apache.tuweni.crypto.sodium.Sodium;
import org.apache.tuweni.crypto.sodium.SodiumException;

public final class SecretBox {
    private SecretBox() {
    }

    public static Bytes encrypt(Bytes message, Key key, Nonce nonce) {
        return Bytes.wrap((byte[])SecretBox.encrypt(message.toArrayUnsafe(), key, nonce));
    }

    public static Allocated encrypt(Allocated message, Key key, Nonce nonce) {
        int macbytes = SecretBox.macLength();
        Allocated cipherText = Allocated.allocate(macbytes + message.length());
        int rc = Sodium.crypto_secretbox_easy(cipherText.pointer(), message.pointer(), (long)message.length(), nonce.value.pointer(), key.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_easy: failed with result " + rc);
        }
        return cipherText;
    }

    public static byte[] encrypt(byte[] message, Key key, Nonce nonce) {
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        int macbytes = SecretBox.macLength();
        byte[] cipherText = new byte[macbytes + message.length];
        int rc = Sodium.crypto_secretbox_easy(cipherText, message, (long)message.length, nonce.value.pointer(), key.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_easy: failed with result " + rc);
        }
        return cipherText;
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, Key key, Nonce nonce) {
        return SecretBox.encryptDetached(message.toArrayUnsafe(), key, nonce);
    }

    public static DetachedEncryptionResult encryptDetached(byte[] message, Key key, Nonce nonce) {
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        int macbytes = SecretBox.macLength();
        byte[] cipherText = new byte[message.length];
        byte[] mac = new byte[macbytes];
        int rc = Sodium.crypto_secretbox_detached(cipherText, mac, message, message.length, nonce.value.pointer(), key.value.pointer());
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_detached: failed with result " + rc);
        }
        return new DefaultDetachedEncryptionResult(cipherText, mac);
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, Key key, Nonce nonce) {
        byte[] bytes = SecretBox.decrypt(cipherText.toArrayUnsafe(), key, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static Allocated decrypt(Allocated cipherText, Key key, Nonce nonce) {
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        int macLength = SecretBox.macLength();
        if (macLength > cipherText.length()) {
            throw new IllegalArgumentException("cipherText is too short");
        }
        Allocated clearText = Allocated.allocate(cipherText.length() - macLength);
        int rc = Sodium.crypto_secretbox_open_easy(clearText.pointer(), cipherText.pointer(), (long)cipherText.length(), nonce.value.pointer(), key.value.pointer());
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_open_easy: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public static byte[] decrypt(byte[] cipherText, Key key, Nonce nonce) {
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        int macLength = SecretBox.macLength();
        if (macLength > cipherText.length) {
            throw new IllegalArgumentException("cipherText is too short");
        }
        byte[] clearText = new byte[cipherText.length - macLength];
        int rc = Sodium.crypto_secretbox_open_easy(clearText, cipherText, (long)cipherText.length, nonce.value.pointer(), key.value.pointer());
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_open_easy: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, Key key, Nonce nonce) {
        byte[] bytes = SecretBox.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), key, nonce);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, Key key, Nonce nonce) {
        Preconditions.checkArgument((!key.isDestroyed() ? 1 : 0) != 0, (Object)"Key has been destroyed");
        int macLength = SecretBox.macLength();
        if (macLength != mac.length) {
            throw new IllegalArgumentException("mac must be " + macLength + " bytes, got " + mac.length);
        }
        byte[] clearText = new byte[cipherText.length];
        int rc = Sodium.crypto_secretbox_open_detached(clearText, cipherText, mac, cipherText.length, nonce.value.pointer(), key.value.pointer());
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_open_detached: failed with result " + rc);
        }
        return clearText;
    }

    public static Bytes encrypt(Bytes message, String password) {
        return SecretBox.encrypt(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static byte[] encrypt(byte[] message, String password) {
        return SecretBox.encrypt(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static Bytes encrypt(Bytes message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encrypt(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    public static byte[] encrypt(byte[] message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encrypt(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    public static Bytes encryptInteractive(Bytes message, String password) {
        return SecretBox.encrypt(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static byte[] encryptInteractive(byte[] message, String password) {
        return SecretBox.encrypt(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static Bytes encryptInteractive(Bytes message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encrypt(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    public static byte[] encryptInteractive(byte[] message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encrypt(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    public static Bytes encryptSensitive(Bytes message, String password) {
        return SecretBox.encrypt(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static byte[] encryptSensitive(byte[] message, String password) {
        return SecretBox.encrypt(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static Bytes encryptSensitive(Bytes message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encrypt(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    public static byte[] encryptSensitive(byte[] message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encrypt(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    public static Bytes encrypt(Bytes message, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        return Bytes.wrap((byte[])SecretBox.encrypt(message.toArrayUnsafe(), password, opsLimit, memLimit, algorithm));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] encrypt(byte[] message, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        int rc;
        Objects.requireNonNull(message);
        Objects.requireNonNull(password);
        if (!algorithm.isSupported()) {
            throw new UnsupportedOperationException(algorithm.name() + " is not supported by the currently loaded sodium native library");
        }
        int macLength = SecretBox.macLength();
        byte[] cipherText = new byte[macLength + message.length];
        Nonce nonce = Nonce.random();
        Key key = SecretBox.deriveKeyFromPassword(password, nonce, opsLimit, memLimit, algorithm);
        assert (!key.isDestroyed());
        try {
            rc = Sodium.crypto_secretbox_easy(cipherText, message, (long)message.length, nonce.value.pointer(), key.value.pointer());
        }
        finally {
            key.destroy();
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_easy: failed with result " + rc);
        }
        return SecretBox.prependNonce(nonce, cipherText);
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, String password) {
        return SecretBox.encryptDetached(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static DetachedEncryptionResult encryptDetached(byte[] message, String password) {
        return SecretBox.encryptDetached(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    public static DetachedEncryptionResult encryptDetached(byte[] message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    public static DetachedEncryptionResult encryptInteractiveDetached(Bytes message, String password) {
        return SecretBox.encryptDetached(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static DetachedEncryptionResult encryptInteractiveDetached(byte[] message, String password) {
        return SecretBox.encryptDetached(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static DetachedEncryptionResult encryptInteractiveDetached(Bytes message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    public static DetachedEncryptionResult encryptInteractiveDetached(byte[] message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    public static DetachedEncryptionResult encryptSensitiveDetached(Bytes message, String password) {
        return SecretBox.encryptDetached(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static DetachedEncryptionResult encryptSensitiveDetached(byte[] message, String password) {
        return SecretBox.encryptDetached(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    public static DetachedEncryptionResult encryptSensitiveDetached(Bytes message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    public static DetachedEncryptionResult encryptSensitiveDetached(byte[] message, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    public static DetachedEncryptionResult encryptDetached(Bytes message, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        return SecretBox.encryptDetached(message.toArrayUnsafe(), password, opsLimit, memLimit, algorithm);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static DetachedEncryptionResult encryptDetached(byte[] message, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        int rc;
        Objects.requireNonNull(message);
        Objects.requireNonNull(password);
        if (!algorithm.isSupported()) {
            throw new UnsupportedOperationException(algorithm.name() + " is not supported by the currently loaded sodium native library");
        }
        int macLength = SecretBox.macLength();
        byte[] cipherText = new byte[message.length];
        byte[] mac = new byte[macLength];
        Nonce nonce = Nonce.random();
        Key key = SecretBox.deriveKeyFromPassword(password, nonce, opsLimit, memLimit, algorithm);
        assert (!key.isDestroyed());
        try {
            rc = Sodium.crypto_secretbox_detached(cipherText, mac, message, message.length, nonce.value.pointer(), key.value.pointer());
        }
        finally {
            key.destroy();
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_detached: failed with result " + rc);
        }
        return new DefaultDetachedEncryptionResult(cipherText, SecretBox.prependNonce(nonce, mac));
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, String password) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static byte[] decrypt(byte[] cipherText, String password) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    @Nullable
    public static byte[] decrypt(byte[] cipherText, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    @Nullable
    public static Bytes decryptInteractive(Bytes cipherText, String password) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static byte[] decryptInteractive(byte[] cipherText, String password) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static Bytes decryptInteractive(Bytes cipherText, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    @Nullable
    public static byte[] decryptInteractive(byte[] cipherText, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    @Nullable
    public static Bytes decryptSensitive(Bytes cipherText, String password) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static byte[] decryptSensitive(byte[] cipherText, String password) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static Bytes decryptSensitive(Bytes cipherText, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    @Nullable
    public static byte[] decryptSensitive(byte[] cipherText, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decrypt(cipherText, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    @Nullable
    public static Bytes decrypt(Bytes cipherText, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        byte[] bytes = SecretBox.decrypt(cipherText.toArrayUnsafe(), password, opsLimit, memLimit, algorithm);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public static byte[] decrypt(byte[] cipherText, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        int rc;
        int macLength;
        Objects.requireNonNull(cipherText);
        Objects.requireNonNull(password);
        if (!algorithm.isSupported()) {
            throw new UnsupportedOperationException(algorithm.name() + " is not supported by the currently loaded sodium native library");
        }
        int noncebytes = Nonce.length();
        if (noncebytes + (macLength = SecretBox.macLength()) > cipherText.length) {
            throw new IllegalArgumentException("cipherText is too short");
        }
        byte[] clearText = new byte[cipherText.length - noncebytes - macLength];
        Nonce nonce = Nonce.fromBytes(Arrays.copyOf(cipherText, noncebytes));
        Key key = SecretBox.deriveKeyFromPassword(password, nonce, opsLimit, memLimit, algorithm);
        assert (!key.isDestroyed());
        try {
            rc = Sodium.crypto_secretbox_open_easy(clearText, Arrays.copyOfRange(cipherText, noncebytes, cipherText.length), (long)(cipherText.length - noncebytes), nonce.value.pointer(), key.value.pointer());
        }
        finally {
            key.destroy();
        }
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_open_easy: failed with result " + rc);
        }
        return clearText;
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, String password) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, String password) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.moderateOpsLimit(), PasswordHash.moderateMemLimit(), algorithm);
    }

    @Nullable
    public static Bytes decryptInteractiveDetached(Bytes cipherText, Bytes mac, String password) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static byte[] decryptInteractiveDetached(byte[] cipherText, byte[] mac, String password) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static Bytes decryptInteractiveDetached(Bytes cipherText, Bytes mac, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    @Nullable
    public static byte[] decryptInteractiveDetached(byte[] cipherText, byte[] mac, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.interactiveOpsLimit(), PasswordHash.interactiveMemLimit(), algorithm);
    }

    @Nullable
    public static Bytes decryptSensitiveDetached(Bytes cipherText, Bytes mac, String password) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static byte[] decryptSensitiveDetached(byte[] cipherText, byte[] mac, String password) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), PasswordHash.Algorithm.recommended());
    }

    @Nullable
    public static Bytes decryptSensitiveDetached(Bytes cipherText, Bytes mac, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    @Nullable
    public static byte[] decryptSensitiveDetached(byte[] cipherText, byte[] mac, String password, PasswordHash.Algorithm algorithm) {
        return SecretBox.decryptDetached(cipherText, mac, password, PasswordHash.sensitiveOpsLimit(), PasswordHash.sensitiveMemLimit(), algorithm);
    }

    @Nullable
    public static Bytes decryptDetached(Bytes cipherText, Bytes mac, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        byte[] bytes = SecretBox.decryptDetached(cipherText.toArrayUnsafe(), mac.toArrayUnsafe(), password, opsLimit, memLimit, algorithm);
        return bytes != null ? Bytes.wrap((byte[])bytes) : null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public static byte[] decryptDetached(byte[] cipherText, byte[] mac, String password, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        int rc;
        int macLength;
        Objects.requireNonNull(cipherText);
        Objects.requireNonNull(mac);
        Objects.requireNonNull(password);
        if (!algorithm.isSupported()) {
            throw new UnsupportedOperationException(algorithm.name() + " is not supported by the currently loaded sodium native library");
        }
        int noncebytes = Nonce.length();
        if (noncebytes + (macLength = SecretBox.macLength()) != mac.length) {
            throw new IllegalArgumentException("mac must be " + (noncebytes + macLength) + " bytes, got " + mac.length);
        }
        byte[] clearText = new byte[cipherText.length];
        Nonce nonce = Nonce.fromBytes(Arrays.copyOf(mac, noncebytes));
        Key key = SecretBox.deriveKeyFromPassword(password, nonce, opsLimit, memLimit, algorithm);
        assert (!key.isDestroyed());
        try {
            rc = Sodium.crypto_secretbox_open_detached(clearText, cipherText, Arrays.copyOfRange(mac, noncebytes, mac.length), cipherText.length, nonce.value.pointer(), key.value.pointer());
        }
        finally {
            key.destroy();
        }
        if (rc == -1) {
            return null;
        }
        if (rc != 0) {
            throw new SodiumException("crypto_secretbox_open_detached: failed with result " + rc);
        }
        return clearText;
    }

    private static int macLength() {
        long macbytes = Sodium.crypto_secretbox_macbytes();
        if (macbytes > Integer.MAX_VALUE) {
            throw new IllegalStateException("crypto_secretbox_macbytes: " + macbytes + " is too large");
        }
        return (int)macbytes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Key deriveKeyFromPassword(String password, Nonce nonce, long opsLimit, long memLimit, PasswordHash.Algorithm algorithm) {
        assert (Nonce.length() >= PasswordHash.Salt.length()) : "SecretBox.Nonce has insufficient length for deriving a PasswordHash.Salt (" + Nonce.length() + " < " + PasswordHash.Salt.length() + ")";
        PasswordHash.Salt salt = PasswordHash.Salt.fromBytes(Arrays.copyOfRange(nonce.bytesArray(), 0, PasswordHash.Salt.length()));
        byte[] passwordBytes = password.getBytes(StandardCharsets.UTF_8);
        try {
            Key key;
            byte[] keyBytes = PasswordHash.hash(passwordBytes, Key.length(), salt, opsLimit, memLimit, algorithm);
            try {
                key = Key.fromBytes(keyBytes);
            }
            catch (Throwable throwable) {
                Arrays.fill(keyBytes, (byte)0);
                throw throwable;
            }
            Arrays.fill(keyBytes, (byte)0);
            return key;
        }
        finally {
            Arrays.fill(passwordBytes, (byte)0);
        }
    }

    private static byte[] prependNonce(Nonce nonce, byte[] bytes) {
        int nonceLength = Nonce.length();
        byte[] data = new byte[nonceLength + bytes.length];
        nonce.value.pointer().get(0L, data, 0, nonceLength);
        System.arraycopy(bytes, 0, data, nonceLength, bytes.length);
        return data;
    }

    public static final class Nonce
    implements Destroyable {
        final Allocated value;

        private Nonce(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        public static Nonce fromBytes(Bytes bytes) {
            return Nonce.fromBytes(bytes.toArrayUnsafe());
        }

        public static Nonce fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_secretbox_noncebytes()) {
                throw new IllegalArgumentException("nonce must be " + Sodium.crypto_secretbox_noncebytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Nonce::new);
        }

        public static int length() {
            long noncebytes = Sodium.crypto_secretbox_noncebytes();
            if (noncebytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_secretbox_noncebytes: " + noncebytes + " is too large");
            }
            return (int)noncebytes;
        }

        public static Nonce zero() {
            int length = Nonce.length();
            Pointer ptr = Sodium.malloc(length);
            try {
                Sodium.sodium_memzero(ptr, length);
                return new Nonce(ptr, length);
            }
            catch (Throwable e) {
                Sodium.sodium_free(ptr);
                throw e;
            }
        }

        @Override
        public void destroy() {
            this.value.destroy();
        }

        public static Nonce random() {
            return Sodium.randomBytes(Nonce.length(), Nonce::new);
        }

        public Nonce increment() {
            return Sodium.dupAndIncrement(this.value.pointer(), Nonce.length(), Nonce::new);
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Nonce)) {
                return false;
            }
            Nonce other = (Nonce)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }

    public static final class Key
    implements Destroyable {
        final Allocated value;

        private Key(Pointer ptr, int length) {
            this.value = new Allocated(ptr, length);
        }

        private Key(Allocated value) {
            this.value = value;
        }

        public static Key fromHash(GenericHash.Hash hash) {
            return new Key(hash.value);
        }

        public static Key fromHash(SHA256Hash.Hash hash) {
            return new Key(hash.value);
        }

        @Override
        public void destroy() {
            this.value.destroy();
        }

        @Override
        public boolean isDestroyed() {
            return this.value.isDestroyed();
        }

        public static Key fromBytes(Bytes bytes) {
            return Key.fromBytes(bytes.toArrayUnsafe());
        }

        public static Key fromBytes(byte[] bytes) {
            if ((long)bytes.length != Sodium.crypto_secretbox_keybytes()) {
                throw new IllegalArgumentException("key must be " + Sodium.crypto_secretbox_keybytes() + " bytes, got " + bytes.length);
            }
            return Sodium.dup(bytes, Key::new);
        }

        public static int length() {
            long keybytes = Sodium.crypto_secretbox_keybytes();
            if (keybytes > Integer.MAX_VALUE) {
                throw new SodiumException("crypto_secretbox_keybytes: " + keybytes + " is too large");
            }
            return (int)keybytes;
        }

        public static Key random() {
            int length = Key.length();
            Pointer ptr = Sodium.malloc(length);
            try {
                Sodium.randombytes_buf(ptr, length);
                return new Key(ptr, length);
            }
            catch (Throwable e) {
                Sodium.sodium_free(ptr);
                throw e;
            }
        }

        public boolean equals(Object obj) {
            if (obj == this) {
                return true;
            }
            if (!(obj instanceof Key)) {
                return false;
            }
            Key other = (Key)obj;
            return other.value.equals(this.value);
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public Bytes bytes() {
            return this.value.bytes();
        }

        public byte[] bytesArray() {
            return this.value.bytesArray();
        }
    }
}

