package org.asciidoctor.converter;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Assigns a default backend name to a converter so that it is not required to define
 * the backend name at registration time.
 *
 * <p>Example:
 * <pre>
 * <code>&#64;ConverterFor("myFormat")
 * public class MyConverter extends AbstractConverter {
 *     ...
 * }
 *
 * asciidoctor.javaConverterRegistry().register(MyConverter.class);
 * </code>
 * </pre>
 */
@Retention(RetentionPolicy.RUNTIME)
@Target(ElementType.TYPE)
public @interface ConverterFor {

    static final String UNDEFINED = "";

    String value() default UNDEFINED;

    /**
     * This parameter defines the backend name for this converter.
     * Is can be set if also another parameter like {@link #suffix()} is defined to avoid the parameter named {@link #value()}.
     * <p>Example:
     * <pre>
     * <code>&#64;ConverterFor(format = "myFormat", suffix = "mine")
     * public class MyConverter extends AbstractConverter {
     *     ...
     * }
     * </code>
     * </pre>
     * @return The backend name for this converter.
     */
    String format() default UNDEFINED;

    /**
     * @return The file extension of files generated by this converter, e.g. {@code .txt}.
     * If not defined generated files will have the extension {@code .html}.
     */
    String suffix() default UNDEFINED;

}
